<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// Restrict to Marketing Agent and Chef Marketing roles
$allowedMarketingRoles = ['Agent Marketing', 'Chef Marketing', 'Marketing Agent', 'Marketing Head'];
if (!in_array($_SESSION['role_name'] ?? '', $allowedMarketingRoles, true)) {
    error_log("Unauthorized access to marketing_dashboard.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/dashboards/unauthorized.php");
    exit;
}

// Fetch province_id if needed
$conn = get_db_connection();
$rolesWithoutProvince = ['Marketing Head', 'Chef Marketing', 'Marketing Agent', 'Agent Marketing'];
$needsProvince = !in_array($_SESSION['role_name'] ?? '', $rolesWithoutProvince, true);
$province_id = $needsProvince ? ($_SESSION['province_id'] ?? null) : null;
$province_name = 'Toutes Régions';

if ($needsProvince) {
    if (!isset($_SESSION['province_id'])) {
        $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
        if (!$stmt) {
            error_log("Query preparation failed for province_id: " . $conn->error);
            echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database error. Contact administrator.</p>";
            include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
            exit;
        }
        $stmt->bind_param("i", $_SESSION['user_id']);
        $stmt->execute();
        $result = $stmt->get_result();
        $user = $result->fetch_assoc();
        $_SESSION['province_id'] = $user['province_id'] ?? null;
        $stmt->close();
    }
    $province_id = $_SESSION['province_id'] ?? null;
    if ($province_id === null || $province_id === '') {
        error_log("No province_id for user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Error: No province assigned. Contact administrator.</p>";
        include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
        exit;
    }
    $province_id = (int)$province_id;
    $stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
    if (!$stmt) {
        error_log("Query preparation failed for province name: " . $conn->error);
        echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database error. Contact administrator.</p>";
        include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
        exit;
    }
    $stmt->bind_param("i", $province_id);
    $stmt->execute();
    $province_name = $stmt->get_result()->fetch_assoc()['name'] ?? 'Unknown';
    $stmt->close();
    $_SESSION['province_id'] = $province_id;
} else {
    $_SESSION['province_id'] = null;
}

$provinceFilterActive = $province_id !== null;
$provinceFilterClause = $provinceFilterActive ? ' AND province_id = ?' : '';

// Ensure notifications table exists
$createNotificationSql = "
    CREATE TABLE IF NOT EXISTS user_notification (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        message VARCHAR(255) NOT NULL,
        link VARCHAR(255) DEFAULT NULL,
        is_read TINYINT(1) NOT NULL DEFAULT 0,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_read (user_id, is_read),
        CONSTRAINT fk_notification_user_marketing FOREIGN KEY (user_id) REFERENCES user(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";
if (!$conn->query($createNotificationSql)) {
    error_log('Failed to ensure notification table on marketing dashboard: ' . $conn->error);
}

// Fetch unread notifications and mark as read
$marketingNotifications = [];
$stmtNotif = $conn->prepare("SELECT id, message, link, created_at FROM user_notification WHERE user_id = ? AND is_read = 0 ORDER BY created_at DESC LIMIT 5");
if ($stmtNotif) {
    $stmtNotif->bind_param("i", $_SESSION['user_id']);
    $stmtNotif->execute();
    $marketingNotifications = $stmtNotif->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmtNotif->close();

    if (!empty($marketingNotifications)) {
        $ids = array_map('intval', array_column($marketingNotifications, 'id'));
        $idList = implode(',', $ids);
        $stmtMark = $conn->prepare("UPDATE user_notification SET is_read = 1 WHERE user_id = ? AND id IN ($idList)");
        if ($stmtMark) {
            $stmtMark->bind_param('i', $_SESSION['user_id']);
            $stmtMark->execute();
            $stmtMark->close();
        }
    }
} else {
    error_log('Failed to fetch notifications for marketing dashboard: ' . $conn->error);
}

$rejectionNotifications = [];
$rejectionSql = "
    SELECT id, comments, created_at AS updated_at
    FROM requisition
    WHERE status = 'rejected'
      AND (user_id = ?
";
$rejectionParams = [$_SESSION['user_id']];
$rejectionTypes = 'i';
if ($province_id) {
    $rejectionSql .= " OR province_id = ?";
    $rejectionParams[] = $province_id;
    $rejectionTypes .= 'i';
}
$rejectionSql .= ")
    ORDER BY updated_at DESC
    LIMIT 5
";
$rejectionStmt = $conn->prepare($rejectionSql);
if ($rejectionStmt) {
    $rejectionStmt->bind_param($rejectionTypes, ...$rejectionParams);
    $rejectionStmt->execute();
    $rows = $rejectionStmt->get_result()->fetch_all(MYSQLI_ASSOC);
    foreach ($rows as $row) {
        $message = sprintf(
            'Réquisition #%d rejetée%s.',
            $row['id'],
            !empty($row['comments']) ? ' (Motif: ' . htmlspecialchars($row['comments'], ENT_QUOTES, 'UTF-8') . ')' : ''
        );
        $rejectionNotifications[] = [
            'id' => 'rej_' . $row['id'],
            'message' => $message,
            'link' => '/masunzu_bar_hotel/modules/expenses/expenses_history.php',
            'created_at' => $row['updated_at'] ?? date('Y-m-d H:i:s'),
        ];
    }
    $rejectionStmt->close();
}
$marketingNotifications = array_merge($rejectionNotifications, $marketingNotifications);

// Province label for display
$province = $province_name;

// Fetch metrics from paid_invoices and unpaid_invoices
$today = date('Y-m-d 00:00:00');
$invoiceCountSql = "
    SELECT COUNT(*) as invoices_created 
    FROM (
        SELECT id FROM paid_invoices WHERE created_by = ?{$provinceFilterClause} AND created_at >= ?
        UNION ALL
        SELECT id FROM unpaid_invoices WHERE created_by = ?{$provinceFilterClause} AND created_at >= ?
    ) as combined_invoices
";
$stmt = $conn->prepare($invoiceCountSql);
if (!$stmt) {
    error_log("Query preparation failed for invoices created: " . $conn->error);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database error. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$invoiceCountTypes = 'i';
$invoiceCountParams = [$_SESSION['user_id']];
if ($provinceFilterActive) {
    $invoiceCountTypes .= 'i';
    $invoiceCountParams[] = $province_id;
}
$invoiceCountTypes .= 's';
$invoiceCountParams[] = $today;
$invoiceCountTypes .= 'i';
$invoiceCountParams[] = $_SESSION['user_id'];
if ($provinceFilterActive) {
    $invoiceCountTypes .= 'i';
    $invoiceCountParams[] = $province_id;
}
$invoiceCountTypes .= 's';
$invoiceCountParams[] = $today;
$stmt->bind_param($invoiceCountTypes, ...$invoiceCountParams);
$stmt->execute();
$invoices_created = $stmt->get_result()->fetch_assoc()['invoices_created'] ?? 0;
$stmt->close();

$totalInvoicedSql = "
    SELECT SUM(paid_amount) as total_invoiced 
    FROM (
        SELECT paid_amount FROM paid_invoices WHERE created_by = ?{$provinceFilterClause} AND created_at >= ?
        UNION ALL
        SELECT paid_amount FROM unpaid_invoices WHERE created_by = ?{$provinceFilterClause} AND created_at >= ?
    ) as combined_invoices
";
$stmt = $conn->prepare($totalInvoicedSql);
if (!$stmt) {
    error_log("Query preparation failed for total invoiced: " . $conn->error);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database error. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$totalInvoicedTypes = 'i';
$totalInvoicedParams = [$_SESSION['user_id']];
if ($provinceFilterActive) {
    $totalInvoicedTypes .= 'i';
    $totalInvoicedParams[] = $province_id;
}
$totalInvoicedTypes .= 's';
$totalInvoicedParams[] = $today;
$totalInvoicedTypes .= 'i';
$totalInvoicedParams[] = $_SESSION['user_id'];
if ($provinceFilterActive) {
    $totalInvoicedTypes .= 'i';
    $totalInvoicedParams[] = $province_id;
}
$totalInvoicedTypes .= 's';
$totalInvoicedParams[] = $today;
$stmt->bind_param($totalInvoicedTypes, ...$totalInvoicedParams);
$stmt->execute();
$total_invoiced = $stmt->get_result()->fetch_assoc()['total_invoiced'] ?? 0.00;
$stmt->close();

$paid_today_count = 0;
$paid_today_amount = 0.0;
$paidTodayClause = $provinceFilterActive ? ' AND province_id = ?' : '';
$paidTodayStmt = $conn->prepare("
    SELECT COUNT(*) AS total, COALESCE(SUM(paid_amount), 0) AS amount
    FROM paid_invoices
    WHERE created_by = ?{$paidTodayClause} AND paid_at >= ?
");
if ($paidTodayStmt) {
    $paidTodayTypes = 'i';
    $paidTodayParams = [$_SESSION['user_id']];
    if ($provinceFilterActive) {
        $paidTodayTypes .= 'i';
        $paidTodayParams[] = $province_id;
    }
    $paidTodayTypes .= 's';
    $paidTodayParams[] = $today;
    $paidTodayStmt->bind_param($paidTodayTypes, ...$paidTodayParams);
    $paidTodayStmt->execute();
    $paidTodayRow = $paidTodayStmt->get_result()->fetch_assoc();
    $paid_today_count = (int)($paidTodayRow['total'] ?? 0);
    $paid_today_amount = (float)($paidTodayRow['amount'] ?? 0);
    $paidTodayStmt->close();
} else {
    error_log("Failed to prepare paid today summary: " . $conn->error);
}

$average_invoice_value = $invoices_created > 0 ? $total_invoiced / $invoices_created : 0;
$paid_share_today = $invoices_created > 0 ? min(100, ($paid_today_count / $invoices_created) * 100) : 0;

// Fetch status counts for this agent
$invoice_status_summary = [
    'pending' => 0,
    'approved' => 0,
    'rejected' => 0,
    'paid' => 0,
    'served' => 0,
];

$statusStmt = $conn->prepare("
    SELECT approval_status, COUNT(*) as total
    FROM unpaid_invoices
    WHERE created_by = ?{$provinceFilterClause}
    GROUP BY approval_status
");
if ($statusStmt) {
    $statusTypes = 'i';
    $statusParams = [$_SESSION['user_id']];
    if ($provinceFilterActive) {
        $statusTypes .= 'i';
        $statusParams[] = $province_id;
    }
    $statusStmt->bind_param($statusTypes, ...$statusParams);
    $statusStmt->execute();
    $statusResult = $statusStmt->get_result();
    while ($row = $statusResult->fetch_assoc()) {
        $key = strtolower($row['approval_status']);
        if (isset($invoice_status_summary[$key])) {
            $invoice_status_summary[$key] = (int)$row['total'];
        }
    }
    $statusStmt->close();
} else {
    error_log("Failed to prepare status summary query: " . $conn->error);
}

$paidStmt = $conn->prepare("
    SELECT 
        COUNT(*) AS paid_total,
        SUM(CASE WHEN serve_status = 'Served' THEN 1 ELSE 0 END) AS served_total
    FROM paid_invoices
    WHERE created_by = ?{$provinceFilterClause}
");
if ($paidStmt) {
    $paidTypes = 'i';
    $paidParams = [$_SESSION['user_id']];
    if ($provinceFilterActive) {
        $paidTypes .= 'i';
        $paidParams[] = $province_id;
    }
    $paidStmt->bind_param($paidTypes, ...$paidParams);
    $paidStmt->execute();
    $paidRow = $paidStmt->get_result()->fetch_assoc();
    $invoice_status_summary['paid'] = (int)($paidRow['paid_total'] ?? 0);
    $invoice_status_summary['served'] = (int)($paidRow['served_total'] ?? 0);
    $paidStmt->close();
} else {
    error_log("Failed to prepare paid summary query: " . $conn->error);
}

$paidProvinceClause = $provinceFilterActive ? ' AND pi.province_id = ?' : '';
$unpaidProvinceClause = $provinceFilterActive ? ' AND ui.province_id = ?' : '';

$stmt = $conn->prepare("
    SELECT invoice_number, customer_name, paid_amount, approval_status, payment_status, serve_status, created_at, product_summary
    FROM (
        SELECT 
            pi.invoice_number,
            pi.customer_name,
            pi.paid_amount,
            pi.approval_status,
            pi.status as payment_status,
            pi.serve_status,
            pi.created_at,
            (
                SELECT GROUP_CONCAT(DISTINCT p.name ORDER BY p.name SEPARATOR ', ')
                FROM paid_invoice_items pii
                JOIN product p ON pii.product_id = p.id
                WHERE pii.invoice_id = pi.id
            ) AS product_summary
        FROM paid_invoices pi
        WHERE pi.created_by = ?{$paidProvinceClause}
        UNION ALL
        SELECT 
            ui.invoice_number,
            ui.customer_name,
            ui.paid_amount,
            ui.approval_status,
            ui.status as payment_status,
            NULL AS serve_status,
            ui.created_at,
            (
                SELECT GROUP_CONCAT(DISTINCT p.name ORDER BY p.name SEPARATOR ', ')
                FROM invoice_items ii
                JOIN product p ON ii.product_id = p.id
                WHERE ii.invoice_id = ui.id
            ) AS product_summary
        FROM unpaid_invoices ui
        WHERE ui.created_by = ?{$unpaidProvinceClause}
    ) as combined_invoices
    ORDER BY created_at DESC
    LIMIT 5
");
if (!$stmt) {
    error_log("Query preparation failed for recent invoices: " . $conn->error);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database error. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$recentTypes = 'i';
$recentParams = [$_SESSION['user_id']];
if ($provinceFilterActive) {
    $recentTypes .= 'i';
    $recentParams[] = $province_id;
}
$recentTypes .= 'i';
$recentParams[] = $_SESSION['user_id'];
if ($provinceFilterActive) {
    $recentTypes .= 'i';
    $recentParams[] = $province_id;
}
$stmt->bind_param($recentTypes, ...$recentParams);
$stmt->execute();
$recent_invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

$clientCounts = [
    'Chef Marketing' => 0,
    'Agent Marketing' => 0,
];
$clientCountStmt = $conn->prepare("
    SELECT r.name AS role_name, COUNT(*) AS total_clients
    FROM customer c
    JOIN user u ON c.created_by = u.id
    JOIN role r ON u.role_id = r.id
    WHERE r.name IN ('Chef Marketing', 'Agent Marketing')
    GROUP BY r.name
");
if ($clientCountStmt) {
    $clientCountStmt->execute();
    $clientCountResult = $clientCountStmt->get_result();
    while ($row = $clientCountResult->fetch_assoc()) {
        $roleKey = $row['role_name'] ?? '';
        if (isset($clientCounts[$roleKey])) {
            $clientCounts[$roleKey] = (int)$row['total_clients'];
        }
    }
    $clientCountStmt->close();
}
$priceChanges = [];
$priceHistoryTable = $conn->query("SHOW TABLES LIKE 'product_price_history'");
if ($priceHistoryTable && $priceHistoryTable->num_rows > 0) {
        $priceStmt = $conn->prepare("
            SELECT pph.product_id, pph.old_price, pph.new_price, pph.changed_at, p.name AS product_name, u.full_name AS changed_by
            FROM product_price_history pph
            LEFT JOIN product p ON pph.product_id = p.id
            LEFT JOIN user u ON pph.changed_by = u.id
            ORDER BY pph.changed_at DESC
        LIMIT 4
    ");
    if ($priceStmt) {
        $priceStmt->execute();
        $priceChanges = $priceStmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $priceStmt->close();
    }
}
if ($priceHistoryTable) {
    $priceHistoryTable->free();
}

$catalogProducts = [];
$productStmt = $conn->prepare("
    SELECT p.id, p.name, p.price_per_crate, p.created_at, u.full_name AS creator_name
    FROM product p
    LEFT JOIN user u ON p.created_by = u.id
    ORDER BY p.name ASC
    LIMIT 12
");
if ($productStmt) {
    $productStmt->execute();
    $catalogProducts = $productStmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $productStmt->close();
}

$conn->close();

include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
?>
<style>
    body {
        font-family: 'Inter', 'Segoe UI', system-ui, sans-serif;
        background: #f4f1eb;
        margin: 0;
        color: #2C1810;
    }
    .marketing-dashboard {
        min-height: 100vh;
        padding: 36px 16px 48px;
    }
    .dashboard-shell {
        width: 100%;
        max-width: none;
        margin: 0 auto;
        background: #ffffff;
        border-radius: 0;
        padding: 36px;
        box-shadow: 0 30px 80px rgba(0, 0, 0, 0.08);
    }
    .hero-section {
        display: flex;
        justify-content: space-between;
        gap: 24px;
        align-items: flex-start;
        flex-wrap: wrap;
        border-bottom: 1px solid #f0e6da;
        padding-bottom: 24px;
    }
    .hero-section h1 {
        font-size: 34px;
        margin: 8px 0 4px;
        color: #2C1810;
    }
    .hero-subtitle {
        margin: 0;
        color: #5b3f2c;
        font-size: 15px;
    }
    .hero-meta {
        display: flex;
        gap: 12px;
        flex-wrap: wrap;
    }
    .hero-meta div {
        background: #fdfbf7;
        border: 1px solid #ede3d7;
        border-radius: 14px;
        padding: 12px 18px;
        min-width: 160px;
        text-align: center;
        box-shadow: inset 0 1px 3px rgba(255, 255, 255, 0.8);
    }
    .hero-meta span {
        display: block;
        font-size: 11px;
        text-transform: uppercase;
        letter-spacing: 0.2em;
        color: #7C4A33;
    }
    .hero-meta strong {
        font-size: 18px;
        color: #2C1810;
        display: block;
        margin-top: 6px;
    }
    .status-card.accent {
        background: linear-gradient(135deg, #4B2F1F, #6E4B34);
        color: #F4F0E4;
        border: none;
        box-shadow: 0 18px 45px rgba(75, 47, 31, 0.5);
    }
    .status-card.accent span {
        color: rgba(244, 240, 228, 0.8);
    }
    .content-grid {
        margin-top: 32px;
        display: grid;
        grid-template-columns: 2fr 1fr;
        gap: 22px;
        align-items: start;
    }
    .recent-card,
    .notifications-card,
    .price-card,
    .catalog-card {
        background: #fff;
        border-radius: 20px;
        border: 1px solid #ede3d7;
        padding: 24px;
        box-shadow: 0 18px 38px rgba(0, 0, 0, 0.06);
    }
    .recent-card header,
    .notifications-card header,
    .price-card header {
        margin-bottom: 16px;
        display: flex;
        flex-direction: column;
        gap: 4px;
    }
    .recent-card h3,
    .notifications-card h3,
    .price-card h3 {
        margin: 0;
        font-size: 22px;
        color: #2C1810;
    }
    .recent-card p,
    .notifications-card p,
    .price-card p {
        margin: 0;
        color: #7C4A33;
        font-size: 14px;
    }
    .table-wrapper {
        overflow-x: auto;
        margin-top: 12px;
    }
    table.recent-table {
        width: 100%;
        border-collapse: collapse;
        min-width: 600px;
    }
    table.recent-table th,
    table.recent-table td {
        padding: 12px 10px;
        font-size: 14px;
        text-align: left;
        border-bottom: 1px solid #f2ede3;
        color: #4B2F1F;
    }
    table.recent-table thead {
        background: #2C1810;
        color: #F4F0E4;
    }
    table.recent-table tbody tr:nth-child(odd) {
        background: #faf5ef;
    }
    .notifications-list {
        list-style: none;
        padding: 0;
        margin: 0;
        display: flex;
        flex-direction: column;
        gap: 16px;
    }
    .notifications-list li {
        padding-bottom: 12px;
        border-bottom: 1px solid #efe5d8;
    }
    .notifications-list strong {
        display: block;
        color: #4B2F1F;
        font-size: 15px;
    }
    .notifications-list small {
        color: #7C4A33;
        font-size: 12px;
    }
    .notification-link {
        color: #2563eb;
        font-size: 12px;
        text-decoration: none;
    }
    .price-card ul {
        list-style: none;
        padding: 0;
        margin: 0;
        display: flex;
        flex-direction: column;
        gap: 12px;
    }
    .price-card li {
        padding-bottom: 10px;
        border-bottom: 1px solid #f0e6da;
        display: flex;
        flex-direction: column;
        gap: 4px;
    }
    .price-card li:last-child {
        border-bottom: none;
    }
    .price-card strong {
        display: block;
        color: #2C1810;
        font-size: 16px;
    }
    .price-card span {
        color: #4B2F1F;
        font-size: 13px;
    }
    .price-card small {
        color: #7C4A33;
        font-size: 12px;
    }
    .price-card-link {
        color: #4B2F1F;
        font-size: 12px;
        font-weight: 600;
        text-decoration: none;
    }
    .table-note {
        color: #7C4A33;
        font-size: 13px;
        padding: 12px 0;
    }
    @media (max-width: 1020px) {
        .content-grid {
            grid-template-columns: 1fr;
        }
        .status-grid {
            grid-template-columns: repeat(auto-fit, minmax(160px, 1fr));
        }
    }
    @media (max-width: 620px) {
        .dashboard-shell {
            padding: 24px;
        }
        .hero-section {
            flex-direction: column;
        }
        .metrics-grid {
            grid-template-columns: 1fr;
        }
        .metric-card {
            min-height: 130px;
        }
    }
</style>
<div class="marketing-dashboard">
    <div class="dashboard-shell">
        <header class="hero-section">
            <div>
                <p class="eyebrow">Stratégie Marketing</p>
                <h1>Tableau de bord Marketing</h1>
                <p class="hero-subtitle">Bonjour <?php echo htmlspecialchars($_SESSION['full_name'], ENT_QUOTES, 'UTF-8'); ?> · <?php echo htmlspecialchars($province, ENT_QUOTES, 'UTF-8'); ?></p>
            </div>
            <div class="hero-meta">
                <div>
                    <span>Rôle</span>
                    <strong><?php echo htmlspecialchars($_SESSION['role_label'] ?? $_SESSION['role_name'], ENT_QUOTES, 'UTF-8'); ?></strong>
                </div>
                <div>
                    <span>Province</span>
                    <strong><?php echo htmlspecialchars($province, ENT_QUOTES, 'UTF-8'); ?></strong>
                </div>
            </div>
        </header>
        <?php /* Metrics removed per request */ ?>
        <section class="content-grid">
            <div class="sidebar">
                <div class="notifications-card">
                    <header>
                        <div>
                            <h3>Notifications</h3>
                            <p>Alertes terrain</p>
                        </div>
                    </header>
                    <?php if (!empty($marketingNotifications)): ?>
                        <ul class="notifications-list">
                            <?php foreach ($marketingNotifications as $note): ?>
                                <li>
                                    <strong><?php echo htmlspecialchars($note['message'], ENT_QUOTES, 'UTF-8'); ?></strong>
                                    <small><?php echo htmlspecialchars(date('d/m/Y H:i', strtotime($note['created_at'])), ENT_QUOTES, 'UTF-8'); ?></small>
                                    <?php if (!empty($note['link'])): ?>
                                        <a class="notification-link" href="<?php echo htmlspecialchars($note['link'], ENT_QUOTES, 'UTF-8'); ?>">Voir</a>
                                    <?php endif; ?>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    <?php else: ?>
                        <p class="table-note">Aucune notification récente.</p>
                    <?php endif; ?>
                </div>
                <div class="price-card">
                    <header>
                        <div>
                            <h3>Derniers ajustements de prix</h3>
                            <p>Surveillance des gammes produits</p>
                        </div>
                    </header>
                    <?php if (!empty($priceChanges)): ?>
                        <ul>
                            <?php foreach ($priceChanges as $change): ?>
                                <li>
                                    <strong><?php echo htmlspecialchars($change['product_name'] ?? 'Produit inconnu', ENT_QUOTES, 'UTF-8'); ?></strong>
                                    <span><?php echo number_format((int)($change['old_price'] ?? 0), 0, ',', '.'); ?> → <?php echo number_format((int)($change['new_price'] ?? 0), 0, ',', '.'); ?> BIF</span>
                                    <small>Modifié par <?php echo htmlspecialchars($change['changed_by'] ?? 'Système', ENT_QUOTES, 'UTF-8'); ?> le <?php echo $change['changed_at'] ? htmlspecialchars(date('d/m/Y', strtotime($change['changed_at'])), ENT_QUOTES, 'UTF-8') : '—'; ?></small>
                                    <?php if (!empty($change['product_id'])): ?>
                                        <a class="price-card-link" href="/masunzu_bar_hotel/modules/operation/product_history.php?id=<?php echo (int)$change['product_id']; ?>">Voir l'historique</a>
                                    <?php endif; ?>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    <?php else: ?>
                        <p class="table-note">Aucun ajustement de prix relevé.</p>
                    <?php endif; ?>
                </div>
            </div>
        </section>
    </div>
</div>
<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
