<?php
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/functions.php';
$appConfig = require __DIR__ . '/../config/app.php';

// Check if user is logged in
if (!is_logged_in()) {
    error_log("User not logged in, redirecting from header.php");
    header("Location: /masunzu_bar_hotel/modules/login/login.php");
    exit;
}

// Initialize database connection
$header_conn = get_db_connection();
if (!$header_conn) {
    error_log("Database connection failed in header.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    exit;
}

// Fetch user role and depot/region assignments
$stmt = $header_conn->prepare("SELECT u.full_name, r.name AS role_name, u.province_id, u.region_id, r.id AS role_id FROM user u JOIN role r ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$user_result = $stmt->get_result();
$user = $user_result->fetch_assoc();
$stmt->close();
$header_conn->close();

if (!$user) {
    error_log("User data not found for user_id {$_SESSION['user_id']} in header.php");
    header("Location: /masunzu_bar_hotel/modules/login/logout.php");
    exit;
}

// Set session variables
$_SESSION['full_name'] = $user['full_name'];
$_SESSION['role_name'] = $user['role_name'];
$_SESSION['role_id'] = $user['role_id'];
// Provide UI-friendly labels and department metadata
$normalizedRoleName = mbh_normalize_role_name($_SESSION['role_name']);
$roleLabel = mbh_get_role_label($normalizedRoleName);
$roleDepartments = [];
foreach (mbh_get_role_departments($normalizedRoleName) as $departmentKey) {
    if (isset($appConfig['departments'][$departmentKey])) {
        $roleDepartments[] = $appConfig['departments'][$departmentKey]['label'];
    } else {
        $roleDepartments[] = $departmentKey;
    }
}
$_SESSION['role_label'] = $roleLabel;
$_SESSION['role_departments'] = $roleDepartments;

// Handle depot/region requirements based on role metadata
$requiresDepot = mbh_role_requires_depot($_SESSION['role_name']);
$requiresRegion = mbh_role_requires_region($_SESSION['role_name']);

if (!isset($_SESSION['province_id'])) {
    $_SESSION['province_id'] = $requiresDepot ? ($user['province_id'] ?? null) : null;
} elseif (!$requiresDepot) {
    $_SESSION['province_id'] = null;
} elseif (is_null($_SESSION['province_id'])) {
    $_SESSION['province_id'] = $user['province_id'] ?? null;
}

if ($requiresDepot && is_null($_SESSION['province_id'])) {
    error_log("No depot assigned for user {$user['full_name']} (ID: {$_SESSION['user_id']}, Role: {$_SESSION['role_name']})");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur: Aucun dépôt assigné. Contactez l'administrateur.</p>";
    include __DIR__ . '/footer.php';
    exit;
}

if (!isset($_SESSION['region_id'])) {
    $_SESSION['region_id'] = $requiresRegion ? ($user['region_id'] ?? null) : null;
} elseif (!$requiresRegion) {
    $_SESSION['region_id'] = null;
} elseif (is_null($_SESSION['region_id'])) {
    $_SESSION['region_id'] = $user['region_id'] ?? null;
}

if ($requiresRegion && is_null($_SESSION['region_id'])) {
    error_log("No region assigned for user {$user['full_name']} (ID: {$_SESSION['user_id']}, Role: {$_SESSION['role_name']})");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur: Aucune région assignée. Contactez l'administrateur.</p>";
    include __DIR__ . '/footer.php';
    exit;
}

// Mirror depot_id for upcoming region/depot terminology
$_SESSION['depot_id'] = $_SESSION['province_id'];

// Define role-specific dashboard paths
$dashboard_map = [
    'Board' => 'admin_dashboard.php',
    'DG' => 'admin_dashboard.php',
    'DGA' => 'admin_dashboard.php',
    'Admin' => 'admin_dashboard.php',
    'Operation Manager' => 'operation_dashboard.php',
    'Procurement Manager' => 'procurement_dashboard.php',
    'Finance Head' => 'finance_dashboard.php',
    'Marketing Head' => 'marketing_dashboard.php',
    'Chef Marketing' => 'marketing_dashboard.php',
    'Stock Manager' => 'stock_manager_dashboard.php',
    'Accountant' => 'accountant_dashboard.php',
    'Marketing Agent' => 'marketing_dashboard.php',
    'Agent Marketing' => 'marketing_dashboard.php',
    'Caissier_Comptable' => 'cashier_dashboard.php'
];
$requestUri = $_SERVER['REQUEST_URI'] ?? '';
$roleName = $_SESSION['role_name'];
$normalizedRoleName = mbh_normalize_role_name($roleName);
$roleId = (int) ($_SESSION['role_id'] ?? 0);

if ($normalizedRoleName !== $roleName) {
    $_SESSION['role_name'] = $normalizedRoleName;
}
$roleName = $normalizedRoleName;
$dashboard = $dashboard_map[$normalizedRoleName] ?? 'unauthorized.php';

if (!function_exists('mbh_nav_link_visible')) {
    function mbh_nav_link_visible(array $link): bool
    {
        return isset($link['show']) ? (bool) $link['show'] : true;
    }

    function mbh_nav_link_active(array $link, string $requestUri): bool
    {
        if (isset($link['is_active']) && is_callable($link['is_active'])) {
            return (bool) call_user_func($link['is_active'], $requestUri);
        }

        if (!empty($link['matchAll'])) {
            foreach ((array) $link['matchAll'] as $pattern) {
                if ($pattern === '' || strpos($requestUri, $pattern) === false) {
                    return false;
                }
            }
            return true;
        }

        $patterns = $link['match'] ?? [];
        if (is_string($patterns)) {
            $patterns = [$patterns];
        }

        foreach ($patterns as $pattern) {
            if ($pattern !== '' && strpos($requestUri, $pattern) !== false) {
                return true;
            }
        }

        return false;
    }

    function mbh_nav_group_active(array $links, string $requestUri): bool
    {
        foreach ($links as $link) {
            if (!mbh_nav_link_visible($link)) {
                continue;
            }

            if (mbh_nav_link_active($link, $requestUri)) {
                return true;
            }
        }

        return false;
    }
}

$menuGroups = [];

// Requisitions menu - available for all roles with appropriate permissions
$requisitionLinks = [];

// Create Requisition - available to every profile for direct submissions
$requisitionLinks[] = [
    'label' => 'Créer Réquisition',
    'href' => '/masunzu_bar_hotel/modules/expenses/create.php',
    'show' => true,
    'match' => 'modules/expenses/create.php',
];

// Approve Requisitions - for DG and DGA
$requisitionLinks[] = [
    'label' => 'Approuver Réquisitions',
    'href' => '/masunzu_bar_hotel/modules/admin/approve_expenses.php',
    'show' => in_array($normalizedRoleName, ['DG', 'DGA', 'Admin'], true),
    'match' => 'approve_expenses.php',
];

// Pay Requisitions - pour les Caissiers Comptables
$requisitionLinks[] = [
    'label' => 'Payer Réquisitions',
    'href' => '/masunzu_bar_hotel/modules/cashiers/pay_requisitions.php',
    'show' => $roleName === 'Caissier_Comptable',
    'match' => 'modules/cashiers/pay_requisitions.php',
];

// Requisition History - for multiple roles
$requisitionLinks[] = [
    'label' => 'Historique Réquisitions',
    'href' => '/masunzu_bar_hotel/modules/expenses/expenses_history.php',
    'show' => in_array($normalizedRoleName, [
        'Board',
        'DG',
        'DGA',
        'Admin',
        'Stock Manager',
        'Operation Manager',
        'Chef Marketing',
        'Agent Marketing',
        'Accountant',
        'Caissier_Comptable'
    ], true),
    'match' => 'expenses_history.php',
];

// Paid Requisitions History - for viewing paid requisitions
$requisitionLinks[] = [
    'label' => 'Historique Réquisitions Payées',
    'href' => '/masunzu_bar_hotel/modules/accountant/paid_requisitions_history.php',
    'show' => in_array($normalizedRoleName, [
        'Board',
        'DG',
        'DGA',
        'Admin',
        'Accountant',
        'Caissier_Comptable',
        'Stock Manager',
        'Operation Manager',
        'Chef Marketing',
        'Agent Marketing'
    ], true),
    'match' => 'paid_requisitions_history.php',
];

// Only add Requisitions menu if user has access to at least one link
$visibleRequisitionLinks = array_filter($requisitionLinks, 'mbh_nav_link_visible');
if (!empty($visibleRequisitionLinks)) {
    $menuGroups[] = [
        'label' => 'Réquisitions',
        'forceDropdown' => true, // Always show as dropdown for better organization
        'links' => $requisitionLinks,
    ];
}

if (in_array($roleName, ['DGA', 'Admin'], true)) {
    $leadershipGroups = [
        'Personnel & Accès' => [
            [
                'label' => 'Gérer Utilisateurs',
                'href' => '/masunzu_bar_hotel/modules/users/manage.php',
                'match' => 'modules/users/manage.php',
            ],
            [
                'label' => 'Gérer Comptables',
                'href' => '/masunzu_bar_hotel/modules/accounting/manage_accountant.php',
                'match' => 'manage_accountant.php',
            ],
            [
                'label' => 'Historique Utilisateurs',
                'href' => '/masunzu_bar_hotel/modules/users/user_history.php',
                'match' => 'user_history.php',
            ],
            [
                'label' => 'Sessions & Connexions',
                'href' => '/masunzu_bar_hotel/modules/users/session_history.php',
                'match' => 'session_history.php',
            ],
        ],
        'Opérations' => [
            [
                'label' => 'Tableau Opérations',
                'href' => '/masunzu_bar_hotel/dashboards/operation_dashboard.php',
                'match' => 'operation_dashboard.php',
            ],
            [
                'label' => 'Vue d\'Ensemble des Stocks',
                'href' => '/masunzu_bar_hotel/modules/admin/view_all_stock.php',
                'match' => 'view_all_stock.php',
            ],
            [
                'label' => 'Historique Dépôts Banque',
                'href' => '/masunzu_bar_hotel/modules/banking/deposit_history.php',
                'match' => 'modules/banking/deposit_history.php',
            ],
            [
                'label' => 'Créer Produit',
                'href' => '/masunzu_bar_hotel/modules/operation/create_product.php',
                'match' => 'create_product.php',
            ],
            [
                'label' => 'Ajouter au Stock Principal',
                'href' => '/masunzu_bar_hotel/modules/operation/add_main_stock.php',
                'match' => 'add_main_stock.php',
            ],
            [
                'label' => 'Historique Stock Principal',
                'href' => '/masunzu_bar_hotel/modules/operation/add_main_stock_history.php',
                'match' => 'add_main_stock_history.php',
            ],
            [
                'label' => 'Catalogue Produits',
                'href' => '/masunzu_bar_hotel/modules/inventory/product_catalog.php',
                'match' => 'product_catalog.php',
            ],
            [
                'label' => 'Initier Transfert',
                'href' => '/masunzu_bar_hotel/modules/operation/initiate_transfer.php',
                'match' => 'initiate_transfer.php',
            ],
            [
                'label' => 'Historique Stock Transferts',
                'href' => '/masunzu_bar_hotel/modules/operation/stock_transfert.php',
                'match' => 'stock_transfert.php',
            ],
            [
                'label' => 'Rapports',
                'href' => '/masunzu_bar_hotel/modules/admin/reports.php',
                'match' => 'modules/admin/reports.php',
            ],
        ],
        'Finance - Actions' => [
            [
                'label' => 'Approuver Commandes',
                'href' => '/masunzu_bar_hotel/modules/admin/approve_invoices.php',
                'match' => 'approve_invoices.php',
            ],
            [
                'label' => 'Voir Toutes les Commandes',
                'href' => '/masunzu_bar_hotel/modules/admin/invoices.php',
                'match' => 'modules/admin/invoices.php',
            ],
            [
                'label' => 'Gestion Crédits',
                'href' => '/masunzu_bar_hotel/modules/accounting/loan_management.php',
                'match' => 'loan_management.php',
            ],
            [
                'label' => 'Comptes Bancaires',
                'href' => '/masunzu_bar_hotel/modules/admin/bank_accounts.php',
                'match' => 'modules/admin/bank_accounts.php',
            ],
        ],
        'Finance - Suivi' => [
            [
                'label' => 'Historique Commandes',
                'href' => '/masunzu_bar_hotel/modules/admin/orders_history.php',
                'match' => 'orders_history.php',
            ],
            [
                'label' => 'Transactions Utilisateurs',
                'href' => '/masunzu_bar_hotel/modules/accounting/transactions_overview.php',
                'match' => 'transactions_overview.php',
            ],
            [
                'label' => 'Historique Remboursements',
                'href' => '/masunzu_bar_hotel/modules/accounting/loan_repayment_history.php',
                'match' => 'loan_repayment_history.php',
            ],
            [
                'label' => 'Historique Transferts Espèces',
                'href' => '/masunzu_bar_hotel/modules/admin/cash_transfer_history.php',
                'match' => 'cash_transfer_history.php',
            ],
        ],
        'Entreprise' => [
            [
                'label' => 'Gérer Entreprise',
                'href' => '/masunzu_bar_hotel/modules/admin/manage_company_info.php',
                'match' => 'manage_company_info.php',
            ],
            [
                'label' => 'Gérer Comptes Bancaires',
                'href' => '/masunzu_bar_hotel/modules/admin/bank_accounts.php',
                'match' => 'modules/admin/bank_accounts.php',
            ],
        ],
    ];

    foreach ($leadershipGroups as $label => $links) {
        $menuGroups[] = [
            'label' => $label,
            'links' => $links,
        ];
    }
} elseif ($roleName === 'Board') {
    // Conseil d'Administration - Vue et Rapports uniquement (pas de création)
    $boardGroups = [
        'Vue d\'Ensemble' => [
            [
                'label' => 'Tableau Opérations',
                'href' => '/masunzu_bar_hotel/dashboards/operation_dashboard.php',
                'match' => 'operation_dashboard.php',
            ],
            [
                'label' => 'Vue d\'Ensemble des Stocks',
                'href' => '/masunzu_bar_hotel/modules/admin/view_all_stock.php',
                'match' => 'view_all_stock.php',
            ],
            [
                'label' => 'Catalogue Produits',
                'href' => '/masunzu_bar_hotel/modules/inventory/product_catalog.php',
                'match' => 'product_catalog.php',
            ],
            [
                'label' => 'Historique Stock Principal',
                'href' => '/masunzu_bar_hotel/modules/operation/add_main_stock_history.php',
                'match' => 'add_main_stock_history.php',
            ],
            [
                'label' => 'Transfert Stock',
                'href' => '/masunzu_bar_hotel/modules/operation/stock_transfert.php',
                'match' => 'stock_transfert.php',
            ],
        ],
        'Commandes & Ventes' => [
            [
                'label' => 'Voir Toutes les Commandes',
                'href' => '/masunzu_bar_hotel/modules/admin/invoices.php',
                'match' => 'modules/admin/invoices.php',
            ],
            [
                'label' => 'Historique Commandes',
                'href' => '/masunzu_bar_hotel/modules/admin/orders_history.php',
                'match' => 'orders_history.php',
            ],
        ],
        'Finance - Opérations' => [
            [
                'label' => 'Transactions Utilisateurs',
                'href' => '/masunzu_bar_hotel/modules/accounting/transactions_overview.php',
                'match' => 'transactions_overview.php',
            ],
            [
                'label' => 'Voir Dépôts Bancaires',
                'href' => '/masunzu_bar_hotel/modules/accounting/view_bank_deposits.php',
                'match' => 'view_bank_deposits.php',
            ],
        ],
        'Finance - Suivi' => [
            [
                'label' => 'Historique Dépôts Banque',
                'href' => '/masunzu_bar_hotel/modules/banking/deposit_history.php',
                'match' => 'modules/banking/deposit_history.php',
            ],
            [
                'label' => 'Historique Transferts Espèces',
                'href' => '/masunzu_bar_hotel/modules/admin/cash_transfer_history.php',
                'match' => 'cash_transfer_history.php',
            ],
            [
                'label' => 'Historique Réquisitions Payées',
                'href' => '/masunzu_bar_hotel/modules/accountant/paid_requisitions_history.php',
                'match' => 'paid_requisitions_history.php',
            ],
            [
                'label' => 'Historique Dépenses',
                'href' => '/masunzu_bar_hotel/modules/expenses/expenses_history.php',
                'match' => 'expenses_history.php',
            ],
        ],
        'Crédits' => [
            [
                'label' => 'Gestion Crédits',
                'href' => '/masunzu_bar_hotel/modules/accounting/loan_management.php',
                'match' => 'loan_management.php',
            ],
            [
                'label' => 'Remboursements Crédits',
                'href' => '/masunzu_bar_hotel/modules/cashiers/loan_repayments.php',
                'match' => 'modules/cashiers/loan_repayments.php',
            ],
            [
                'label' => 'Historique Remboursements',
                'href' => '/masunzu_bar_hotel/modules/accounting/loan_repayment_history.php',
                'match' => 'loan_repayment_history.php',
            ],
        ],
        'Clients' => [
            [
                'label' => 'Historique Créations',
                'href' => '/masunzu_bar_hotel/modules/customers/history.php',
                'match' => 'modules/customers/history.php',
            ],
            [
                'label' => 'Observations Terrain',
                'href' => '/masunzu_bar_hotel/modules/customers/observations.php',
                'match' => 'modules/customers/observations.php',
            ],
        ],
        'Rapports' => [
            [
                'label' => 'Rapports',
                'href' => '/masunzu_bar_hotel/modules/admin/reports.php',
                'match' => 'modules/admin/reports.php',
            ],
        ],
    ];

    foreach ($boardGroups as $label => $links) {
        $menuGroups[] = [
            'label' => $label,
            'links' => $links,
        ];
    }
} elseif ($roleName === 'DG') {
    $menuGroups[] = [
        'label' => 'Administration',
        'links' => [
            [
                'label' => 'Gérer Utilisateurs',
                'href' => '/masunzu_bar_hotel/modules/users/manage.php',
                'match' => 'modules/users/manage.php',
            ],
            [
                'label' => 'Vue d\'Ensemble des Stocks',
                'href' => '/masunzu_bar_hotel/modules/admin/view_all_stock.php',
                'match' => 'view_all_stock.php',
            ],
            [
                'label' => 'Historique Dépôts Banque',
                'href' => '/masunzu_bar_hotel/modules/banking/deposit_history.php',
                'match' => 'modules/banking/deposit_history.php',
            ],
            [
                'label' => 'Rapports',
                'href' => '/masunzu_bar_hotel/modules/admin/reports.php',
                'match' => 'modules/admin/reports.php',
            ],
            [
                'label' => 'Tableau Opérations',
                'href' => '/masunzu_bar_hotel/dashboards/operation_dashboard.php',
                'match' => 'operation_dashboard.php',
            ],
            [
                'label' => 'Créer Produit',
                'href' => '/masunzu_bar_hotel/modules/operation/create_product.php',
                'match' => 'create_product.php',
            ],
            [
                'label' => 'Catalogue Produits',
                'href' => '/masunzu_bar_hotel/modules/inventory/product_catalog.php',
                'match' => 'product_catalog.php',
            ],
            [
                'label' => 'Voir Toutes les Commandes',
                'href' => '/masunzu_bar_hotel/modules/admin/invoices.php',
                'match' => 'modules/admin/invoices.php',
            ],
            [
                'label' => 'Historique Commandes',
                'href' => '/masunzu_bar_hotel/modules/admin/orders_history.php',
                'match' => 'orders_history.php',
            ],
            [
                'label' => 'Gestion Crédits',
                'href' => '/masunzu_bar_hotel/modules/accounting/loan_management.php',
                'match' => 'loan_management.php',
            ],
            [
                'label' => 'Gérer Entreprise',
                'href' => '/masunzu_bar_hotel/modules/admin/manage_company_info.php',
                'match' => 'manage_company_info.php',
            ],
            [
                'label' => 'Gérer Comptables',
                'href' => '/masunzu_bar_hotel/modules/accounting/manage_accountant.php',
                'match' => 'manage_accountant.php',
            ],
            [
                'label' => 'Voir Dépôts Bancaires',
                'href' => '/masunzu_bar_hotel/modules/accounting/view_bank_deposits.php',
                'match' => 'view_bank_deposits.php',
            ],
            [
                'label' => 'Remboursements Crédits',
                'href' => '/masunzu_bar_hotel/modules/cashiers/loan_repayments.php',
                'match' => 'modules/cashiers/loan_repayments.php',
            ],
            [
                'label' => 'Historique Transferts Espèces',
                'href' => '/masunzu_bar_hotel/modules/admin/cash_transfer_history.php',
                'match' => 'cash_transfer_history.php',
            ],
        ],
    ];
}

if ($roleName === 'Stock Manager') {
    $menuGroups[] = [
        'label' => 'Gestion Stock',
        'links' => [
            [
                'label' => 'Recevoir Stock',
                'href' => '/masunzu_bar_hotel/modules/operation/receive_stock.php',
                'match' => 'receive_stock.php',
            ],
            [
                'label' => 'Catalogue Produits',
                'href' => '/masunzu_bar_hotel/modules/inventory/product_catalog.php',
                'match' => 'product_catalog.php',
            ],
            [
                'label' => 'Historique Réceptions',
                'href' => '/masunzu_bar_hotel/modules/operation/receiving_stock_history.php',
                'match' => 'receiving_stock_history.php',
            ],
            [
                'label' => 'Historique Dépôts Banque',
                'href' => '/masunzu_bar_hotel/modules/banking/deposit_history.php',
                'match' => 'modules/banking/deposit_history.php',
            ],
            [
                'label' => 'Historique Transferts',
                'href' => '/masunzu_bar_hotel/modules/operation/stock_transfer_history.php',
                'match' => 'stock_transfer_history.php',
            ],
            [
                'label' => 'Ventes',
                'href' => '/masunzu_bar_hotel/modules/cashiers/sold_products.php',
                'match' => 'modules/cashiers/sold_products.php',
            ],
            [
                'label' => 'Créer Réquisition',
                'href' => '/masunzu_bar_hotel/modules/expenses/create.php',
                'match' => 'modules/expenses/create.php',
            ],
            [
                'label' => 'Historique Transferts Espèces',
                'href' => '/masunzu_bar_hotel/modules/admin/cash_transfer_history.php',
                'match' => 'cash_transfer_history.php',
            ],
        ],
    ];
    $menuGroups[] = [
        'label' => 'Commandes',
        'links' => [
            [
                'label' => 'Approuver Commandes',
                'href' => '/masunzu_bar_hotel/modules/admin/approve_invoices.php',
                'match' => 'approve_invoices.php',
            ],
            [
                'label' => 'Commandes en attente',
                'href' => '/masunzu_bar_hotel/modules/admin/approve_invoices.php?status=pending',
                'match' => 'approve_invoices.php?status=pending',
            ],
            [
                'label' => 'Historique Commandes',
                'href' => '/masunzu_bar_hotel/modules/admin/orders_history.php',
                'match' => 'orders_history.php',
            ],
        ],
    ];
}

if ($roleName === 'Operation Manager') {
    $menuGroups[] = [
        'label' => 'Gestion Stock',
        'links' => [
            [
                'label' => 'Tableau Opérations',
                'href' => '/masunzu_bar_hotel/dashboards/operation_dashboard.php',
                'match' => 'operation_dashboard.php',
            ],
            [
                'label' => 'Envoyer du Stock',
                'href' => '/masunzu_bar_hotel/modules/operation/send_stock.php',
                'match' => 'send_stock.php',
            ],
            [
                'label' => 'Recevoir Stock',
                'href' => '/masunzu_bar_hotel/modules/operation/receive_stock.php',
                'match' => 'receive_stock.php',
            ],
            [
                'label' => 'Commandes Région',
                'href' => '/masunzu_bar_hotel/modules/operation/orders.php',
                'match' => 'modules/operation/orders.php',
            ],
            [
                'label' => 'Historique Stock Principal',
                'href' => '/masunzu_bar_hotel/modules/operation/add_main_stock_history.php',
                'match' => 'add_main_stock_history.php',
            ],
            [
                'label' => 'Catalogue Produits',
                'href' => '/masunzu_bar_hotel/modules/inventory/product_catalog.php',
                'match' => 'product_catalog.php',
            ],
            [
                'label' => 'Historique Transferts Espèces',
                'href' => '/masunzu_bar_hotel/modules/admin/cash_transfer_history.php',
                'match' => 'cash_transfer_history.php',
            ],
        ],
    ];
}

if ($roleName === 'Caissier_Comptable') {
    $menuGroups[] = [
        'label' => 'Caisse',
        'links' => [
            [
                'label' => 'Créer Commande',
                'href' => '/masunzu_bar_hotel/modules/cashiers/create_invoice.php',
                'match' => 'modules/cashiers/create_invoice.php',
            ],
            [
                'label' => 'Traiter Paiement',
                'href' => '/masunzu_bar_hotel/modules/cashiers/process_payment.php',
                'match' => 'process_payment.php',
            ],
            // Pay Requisitions is now in the Requisitions menu dropdown
            [
                'label' => 'Gestion des Sessions',
                'href' => '/masunzu_bar_hotel/modules/cashiers/balance_management.php',
                'match' => 'balance_management.php',
            ],
            [
                'label' => 'Transactions',
                'href' => '/masunzu_bar_hotel/modules/cashiers/transaction_history.php',
                'match' => 'transaction_history.php',
            ],
            [
                'label' => 'Ventes',
                'href' => '/masunzu_bar_hotel/modules/cashiers/sold_products.php',
                'match' => 'modules/cashiers/sold_products.php',
            ],
            [
                'label' => 'Commandes',
                'href' => '/masunzu_bar_hotel/modules/cashiers/invoices.php',
                'match' => 'modules/cashiers/invoices.php',
            ],
            [
                'label' => 'Remboursements Crédits',
                'href' => '/masunzu_bar_hotel/modules/cashiers/loan_repayments.php',
                'match' => 'modules/cashiers/loan_repayments.php',
            ],
            [
                'label' => 'Déposer à la Banque',
                'href' => '/masunzu_bar_hotel/modules/banking/deposit.php',
                'match' => 'modules/banking/deposit.php',
            ],
            [
                'label' => 'Historique Dépôts',
                'href' => '/masunzu_bar_hotel/modules/banking/deposit_history.php',
                'match' => 'modules/banking/deposit_history.php',
            ],
            [
                'label' => 'Transférer Espèces',
                'href' => '/masunzu_bar_hotel/modules/cashiers/transfer_cash.php',
                'match' => 'transfer_cash.php',
            ],
            [
                'label' => 'Catalogue Produits',
                'href' => '/masunzu_bar_hotel/modules/inventory/product_catalog.php',
                'match' => 'product_catalog.php',
            ],
        ],
    ];
    
    // Add Clients menu for cashiers
    $cashierCustomerLinks = [
        [
            'label' => 'Gérer Clients',
            'href' => '/masunzu_bar_hotel/modules/customers/manage_customers.php',
            'match' => 'modules/customers/manage_customers.php',
        ],
        [
            'label' => 'Observations Terrain',
            'href' => '/masunzu_bar_hotel/modules/customers/observations.php',
            'match' => 'modules/customers/observations.php',
        ],
    ];
    $menuGroups[] = [
        'label' => 'Clients',
        'forceDropdown' => true,
        'links' => $cashierCustomerLinks,
    ];
}

if (in_array($roleName, ['Agent Marketing', 'Chef Marketing'], true)) {
    $menuGroups[] = [
        'label' => 'Terrain Marketing',
        'links' => [
            [
                'label' => 'Créer Commande',
                'href' => '/masunzu_bar_hotel/modules/cashiers/create_invoice.php',
                'match' => 'modules/cashiers/create_invoice.php',
            ],
            [
                'label' => 'Créer Réquisition',
                'href' => '/masunzu_bar_hotel/modules/expenses/create.php',
                'match' => 'modules/expenses/create.php',
            ],
            [
                'label' => 'Catalogue Produits',
                'href' => '/masunzu_bar_hotel/modules/inventory/product_catalog.php',
                'match' => 'product_catalog.php',
            ],
        ],
    ];
}

// Low stock alerts access
$lowStockRoles = ['Admin', 'DG', 'DGA', 'Board', 'Accountant', 'Operation Manager', 'Stock Manager', 'Caissier_Comptable'];
if (in_array($roleName, $lowStockRoles, true)) {
    $menuGroups[] = [
        'label' => 'Alertes Stock',
        'links' => [
            [
                'label' => 'Alertes Stock Faible',
                'href' => '/masunzu_bar_hotel/modules/admin/low_stock_alerts.php',
                'match' => 'low_stock_alerts.php',
            ],
        ],
    ];
}

if ($roleName === 'Procurement Manager') {
    $menuGroups[] = [
        'label' => 'Achats',
        'links' => [
            [
                'label' => 'Commandes Fournisseur',
                'href' => '/masunzu_bar_hotel/modules/procurement/purchase_orders.php',
                'match' => 'purchase_orders.php',
            ],
            [
                'label' => 'Gestion Fournisseurs',
                'href' => '/masunzu_bar_hotel/modules/procurement/supplier_management.php',
                'match' => 'supplier_management.php',
            ],
            [
                'label' => 'Catalogue Produits',
                'href' => '/masunzu_bar_hotel/modules/inventory/product_catalog.php',
                'match' => 'product_catalog.php',
            ],
        ],
    ];
}

if ($roleName === 'Finance Head') {
    $menuGroups[] = [
        'label' => 'Finance',
        'links' => [
            [
                'label' => 'Transactions',
                'href' => '/masunzu_bar_hotel/modules/finance/transactions.php',
                'match' => 'modules/finance/transactions.php',
            ],
            [
                'label' => 'Crédits',
                'href' => '/masunzu_bar_hotel/modules/finance/credits.php',
                'match' => 'modules/finance/credits.php',
            ],
            [
                'label' => 'Catalogue Produits',
                'href' => '/masunzu_bar_hotel/modules/inventory/product_catalog.php',
                'match' => 'product_catalog.php',
            ],
        ],
    ];
}

if ($roleName === 'Marketing Head') {
    $menuGroups[] = [
        'label' => 'Stratégie Marketing',
        'links' => [
            [
                'label' => 'Gestion Campagnes',
                'href' => '/masunzu_bar_hotel/modules/marketing/campaign_management.php',
                'match' => 'campaign_management.php',
            ],
            [
                'label' => 'Analyse Marché',
                'href' => '/masunzu_bar_hotel/modules/marketing/market_analysis.php',
                'match' => 'market_analysis.php',
            ],
            [
                'label' => 'Catalogue Produits',
                'href' => '/masunzu_bar_hotel/modules/inventory/product_catalog.php',
                'match' => 'product_catalog.php',
            ],
        ],
    ];
}

$normalizedRoleCheck = function ($name) {
    return strtolower(trim($name)) === 'accountant';
};

if ($roleName === 'Accountant' || $normalizedRoleCheck($roleName)) {
    // Sépare les actions courantes et la consultation pour alléger le menu
    $accountingOperations = [
        [
            'label' => 'Recevoir Espèces',
            'href' => '/masunzu_bar_hotel/modules/accounting/receive_cash.php',
            'match' => 'receive_cash.php',
        ],
        [
            'label' => 'Dépôt Banque',
            'href' => '/masunzu_bar_hotel/modules/accounting/bank_accounts.php',
            'is_active' => function (string $uri): bool {
                return strpos($uri, 'bank_accounts.php') !== false && strpos($uri, 'show=accounts') === false;
            },
        ],
        [
            'label' => 'Comptes Banque',
            'href' => '/masunzu_bar_hotel/modules/accounting/bank_accounts.php?show=accounts#accounts',
            'is_active' => function (string $uri): bool {
                return strpos($uri, 'bank_accounts.php') !== false && strpos($uri, 'show=accounts') !== false;
            },
        ],
        [
            'label' => 'Solde Espèces',
            'href' => '/masunzu_bar_hotel/modules/accounting/accountant_balance.php',
            'match' => 'accountant_balance.php',
        ],
        [
            'label' => 'Stock Espèces',
            'href' => '/masunzu_bar_hotel/modules/accounting/accountant_stock.php',
            'match' => 'accountant_stock.php',
        ],
    ];

    $accountingFollowup = [
        [
            'label' => 'Historique Dépôts',
            'href' => '/masunzu_bar_hotel/modules/banking/deposit_history.php',
            'match' => 'modules/banking/deposit_history.php',
        ],
        [
            'label' => 'Voir Dépôts Bancaires',
            'href' => '/masunzu_bar_hotel/modules/accounting/view_bank_deposits.php',
            'match' => 'view_bank_deposits.php',
        ],
        [
            'label' => 'Gestion Crédits',
            'href' => '/masunzu_bar_hotel/modules/accounting/loan_management.php',
            'match' => 'loan_management.php',
        ],
        [
            'label' => 'Historique Remboursements',
            'href' => '/masunzu_bar_hotel/modules/accounting/loan_repayment_history.php',
            'match' => 'loan_repayment_history.php',
        ],
        [
            'label' => 'Historique Transferts Espèces',
            'href' => '/masunzu_bar_hotel/modules/admin/cash_transfer_history.php',
            'match' => 'cash_transfer_history.php',
        ],
        [
            'label' => 'Historique Dépenses',
            'href' => '/masunzu_bar_hotel/modules/expenses/expenses_history.php',
            'match' => 'expenses_history.php',
        ],
        [
            'label' => 'Catalogue Produits',
            'href' => '/masunzu_bar_hotel/modules/inventory/product_catalog.php',
            'match' => 'product_catalog.php',
        ],
    ];

    $menuGroups[] = [
        'label' => 'Comptabilité - Opérations',
        'links' => $accountingOperations,
    ];
    $menuGroups[] = [
        'label' => 'Comptabilité - Suivi',
        'links' => $accountingFollowup,
    ];
}

$customerLinks = [];
$customerManagers = ['DG', 'DGA', 'Admin', 'Chef Marketing', 'Agent Marketing', 'Accountant'];
if (in_array($roleName, $customerManagers, true)) {
    $customerLinks[] = [
        'label' => 'Gérer Clients',
        'href' => '/masunzu_bar_hotel/modules/customers/manage_customers.php',
        'match' => 'modules/customers/manage_customers.php',
    ];
}
if (in_array($roleName, ['DG', 'DGA', 'Admin', 'Chef Marketing', 'Agent Marketing', 'Accountant'], true)) {
    $customerLinks[] = [
        'label' => 'Historique Créations',
        'href' => '/masunzu_bar_hotel/modules/customers/history.php',
        'match' => 'modules/customers/history.php',
    ];
}
if (in_array($roleName, ['DG', 'DGA', 'Admin', 'Chef Marketing', 'Agent Marketing', 'Accountant'], true)) {
    $customerLinks[] = [
        'label' => 'Observations Terrain',
        'href' => '/masunzu_bar_hotel/modules/customers/observations.php',
        'match' => 'modules/customers/observations.php',
    ];
}
if (in_array($roleName, ['DG', 'DGA', 'Admin'], true)) {
    $customerLinks[] = [
        'label' => 'Vue Observations',
        'href' => '/masunzu_bar_hotel/modules/customers/admin_observations.php',
        'match' => 'modules/customers/admin_observations.php',
    ];
}
if (!empty($customerLinks)) {
    $menuGroups[] = [
        'label' => 'Clients',
        'forceDropdown' => true,
        'links' => $customerLinks,
    ];
}

if ($roleName === 'Admin') {
    $menuGroups[] = [
        'label' => 'Administration',
        'links' => [
            [
                'label' => 'Historique Transferts Espèces',
                'href' => '/masunzu_bar_hotel/modules/admin/cash_transfer_history.php',
                'match' => 'cash_transfer_history.php',
            ],
            [
                'label' => 'Historique Dépenses',
                'href' => '/masunzu_bar_hotel/modules/expenses/expenses_history.php',
                'match' => 'expenses_history.php',
            ],
            [
                'label' => 'Assistant Initial',
                'href' => '/masunzu_bar_hotel/modules/admin/setup.php',
                'match' => 'modules/admin/setup.php',
            ],
        ],
    ];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($lang['dashboard'] ?? 'Tableau de Bord', ENT_QUOTES, 'UTF-8'); ?> -Société Masunzu Bar Hotel</title>
    <link rel="apple-touch-icon" sizes="180x180" href="/masunzu_bar_hotel/assets/favicon_io/apple-touch-icon.png">
    <link rel="icon" type="image/png" sizes="32x32" href="/masunzu_bar_hotel/assets/favicon_io/favicon-32x32.png">
    <link rel="icon" type="image/png" sizes="16x16" href="/masunzu_bar_hotel/assets/favicon_io/favicon-16x16.png">
    <link rel="manifest" href="/masunzu_bar_hotel/assets/favicon_io/site.webmanifest">
    <link rel="shortcut icon" href="/masunzu_bar_hotel/assets/favicon_io/favicon.ico">
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Bootstrap Icons -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        body {
            font-family: Arial, sans-serif;
            background-color: #F4F0E4;
            color: #4B2F1F;
            margin: 0;
            padding: 0;
            font-size: 18px;
        }
        header {
            background-color: #4B2F1F;
            color: #F4F0E4;
            padding: 15px;
            text-align: center;
            box-shadow: 0 2px 4px rgba(0,0,0,0.3);
        }
        header h1 {
            margin: 0;
            font-size: 36px;
            font-weight: bold;
        }
        nav {
            background-color: #4B2F1F;
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
            width: 100%;
            box-sizing: border-box;
            position: sticky;
            top: 0;
            z-index: 1000;
            box-shadow: 0 2px 4px rgba(0,0,0,0.3);
            backdrop-filter: blur(5px);
        }
        nav div:first-child {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            align-items: center;
        }
        nav a {
            color: #F4F0E4;
            text-decoration: none;
            font-weight: 500;
            padding: 12px 20px;
            border-radius: 8px;
            font-size: 18px;
            transition: all 0.3s ease;
            position: relative;
        }
        nav a:hover, nav a.active {
            background-color: #F4A261;
            color: #4B2F1F;
            transform: translateY(-2px);
            box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        }
        nav a::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            width: 100%;
            height: 2px;
            background-color: #F4A261;
            transform: scaleX(0);
            transform-origin: bottom right;
            transition: transform 0.3s ease;
        }
        nav a:hover::after, nav a.active::after {
            transform: scaleX(1);
            transform-origin: bottom left;
        }
        .dropdown {
            position: relative;
            display: inline-block;
        }
        .dropdown:hover .dropdown-content, .dropdown .dropbtn:focus + .dropdown-content {
            display: block;
        }
        .dropdown-content {
            display: none;
            position: absolute;
            background-color: #FFFFFF;
            min-width: 200px;
            box-shadow: 0 8px 16px rgba(0,0,0,0.2);
            z-index: 1;
            border-radius: 8px;
            overflow: hidden;
            top: 100%;
            left: 0;
        }
        .dropdown-content a {
            display: block;
            padding: 12px 16px;
            text-decoration: none;
            color: #4B2F1F;
            font-size: 16px;
            border-bottom: 1px solid #F4F0E4;
            background-color: #FFFFFF;
        }
        .dropdown-content a:hover {
            background-color: #F4A261;
            color: #4B2F1F;
        }
        .dropdown-content a.active {
            background-color: #F4A261;
            color: #4B2F1F;
        }
        .user-menu {
            position: relative;
            display: inline-block;
        }
        .user-icon {
            background-color: #F4A261;
            color: #4B2F1F;
            border: none;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            cursor: pointer;
            font-size: 20px;
            font-weight: bold;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
        }
        .user-icon:hover, .user-icon:focus {
            background-color: #F4F0E4;
            transform: scale(1.1);
        }
        .dropdown-content.user-dropdown {
            right: 0;
            left: auto;
        }
        .dropdown-content .user-info {
            background-color: #4B2F1F;
            color: #F4F0E4;
            padding: 12px 16px;
            font-size: 16px;
            font-weight: 500;
            text-align: center;
            border-bottom: 1px solid #F4F0E4;
        }
        @media (max-width: 768px) {
            nav {
                flex-direction: column;
                align-items: flex-start;
                padding: 15px;
            }
            nav div:first-child {
                flex-direction: column;
                width: 100%;
                gap: 10px;
            }
            nav a, .dropdown .dropbtn {
                width: 100%;
                text-align: center;
                padding: 10px;
                font-size: 16px;
            }
            .dropdown-content {
                position: static;
                width: 100%;
                box-shadow: none;
                border: 1px solid #4B2F1F;
            }
            .user-menu {
                align-self: flex-end;
                margin-top: 10px;
            }
            .user-icon {
                width: 36px;
                height: 36px;
                font-size: 18px;
            }
            .dropdown-content.user-dropdown {
                min-width: 160px;
            }
        }
    </style>
</head>
<body>
    <header>
        <h1> Société Masunzu Bar Hôtel</h1>
    </header>
    <nav>
                <div>
            <a href="/masunzu_bar_hotel/dashboards/<?php echo htmlspecialchars($dashboard, ENT_QUOTES, 'UTF-8'); ?>" <?php echo strpos($_SERVER['REQUEST_URI'], $dashboard) !== false ? 'class="active"' : ''; ?>>Tableau de Bord</a>
            <?php foreach ($menuGroups as $group): ?>
                <?php
                    $visibleLinks = array_values(array_filter($group['links'], 'mbh_nav_link_visible'));
                    if (!$visibleLinks) {
                        continue;
                    }
                    $isDropdown = ($group['type'] ?? 'dropdown') === 'dropdown';
                    $forceDropdown = $group['forceDropdown'] ?? false;
                    $dropdownActive = mbh_nav_group_active($visibleLinks, $requestUri);
                ?>
                <?php if ($isDropdown && (count($visibleLinks) > 1 || $forceDropdown)): ?>
                    <div class="dropdown">
                        <a href="#" class="dropbtn <?php echo $dropdownActive ? 'active' : ''; ?>"><?php echo htmlspecialchars($group['label'], ENT_QUOTES, 'UTF-8'); ?></a>
                        <div class="dropdown-content">
                            <?php foreach ($visibleLinks as $link): ?>
                                <a href="<?php echo htmlspecialchars($link['href'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo mbh_nav_link_active($link, $requestUri) ? 'class="active"' : ''; ?>>
                                    <?php echo htmlspecialchars($link['label'], ENT_QUOTES, 'UTF-8'); ?>
                                </a>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php else: ?>
                    <?php $link = $visibleLinks[0]; ?>
                    <a href="<?php echo htmlspecialchars($link['href'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo mbh_nav_link_active($link, $requestUri) ? 'class="active"' : ''; ?>>
                        <?php echo htmlspecialchars($link['label'], ENT_QUOTES, 'UTF-8'); ?>
                    </a>
                <?php endif; ?>
            <?php endforeach; ?>
        </div>

        <div class="user-menu">
            <button class="user-icon" aria-label="Menu Utilisateur"><?php echo htmlspecialchars(substr($_SESSION['full_name'] ?? 'U', 0, 1), ENT_QUOTES, 'UTF-8'); ?></button>
            <div class="dropdown-content user-dropdown">
                <div class="user-info">
                    <?php
                        $currentRoleLabel = $_SESSION['role_label'] ?? $_SESSION['role_name'] ?? 'Inconnu';
                        echo htmlspecialchars($_SESSION['full_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8') . ' (' . htmlspecialchars($currentRoleLabel, ENT_QUOTES, 'UTF-8') . ')';
                    ?>
                </div>
                <?php if (!empty($_SESSION['role_departments'])): ?>
                    <div class="user-info">
                        Département(s) :
                        <?php echo htmlspecialchars(implode(', ', $_SESSION['role_departments']), ENT_QUOTES, 'UTF-8'); ?>
                    </div>
                <?php endif; ?>
                <a href="/masunzu_bar_hotel/modules/users/profile.php">Profil</a>
                <a href="/masunzu_bar_hotel/modules/users/change_password.php">Mot de Passe</a>
                <a href="/masunzu_bar_hotel/modules/login/logout.php">Déconnexion</a>
            </div>
        </div>
    </nav>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const userIcon = document.querySelector('.user-icon');
            const userDropdown = document.querySelector('.dropdown-content.user-dropdown');
            const dropdowns = document.querySelectorAll('.dropdown');

            // Toggle user dropdown
            userIcon.addEventListener('click', function(event) {
                event.stopPropagation();
                userDropdown.style.display = userDropdown.style.display === 'block' ? 'none' : 'block';
            });

            // Toggle dropdowns on click
            dropdowns.forEach(dropdown => {
                const dropbtn = dropdown.querySelector('.dropbtn');
                const dropdownContent = dropdown.querySelector('.dropdown-content');

                dropbtn.addEventListener('click', function(event) {
                    event.stopPropagation();
                    // Close other dropdowns
                    dropdowns.forEach(otherDropdown => {
                        if (otherDropdown !== dropdown) {
                            otherDropdown.querySelector('.dropdown-content').style.display = 'none';
                        }
                    });
                    // Toggle current dropdown
                    dropdownContent.style.display = dropdownContent.style.display === 'block' ? 'none' : 'block';
                });

                // Allow navigation for dropdown-content links
                dropdownContent.querySelectorAll('a').forEach(link => {
                    link.addEventListener('click', function(event) {
                        event.stopPropagation();
                        window.location.href = this.getAttribute('href');
                    });
                });
            });

            // Close all dropdowns when clicking outside
            document.addEventListener('click', function(event) {
                if (!userIcon.contains(event.target) && !userDropdown.contains(event.target)) {
                    userDropdown.style.display = 'none';
                }
                dropdowns.forEach(dropdown => {
                    const dropdownContent = dropdown.querySelector('.dropdown-content');
                    if (!dropdown.contains(event.target)) {
                        dropdownContent.style.display = 'none';
                    }
                });
            });

            // Ensure only one dropdown is active
            dropdowns.forEach(dropdown => {
                const dropbtn = dropdown.querySelector('.dropbtn');
                if (dropbtn.classList.contains('active')) {
                    dropdown.querySelector('.dropdown-content').style.display = 'block';
                }
            });
        });
    </script>
    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
