<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';
require_login();

// Restrict to DG, DGA, Board (Admin) roles
$normalizedRole = mbh_normalize_role_name($_SESSION['role_name'] ?? '');
if (!in_array($normalizedRole, ['DG', 'DGA', 'Board', 'Admin'], true)) {
    error_log("Unauthorized access to admin invoices.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/index.php");
    exit;
}

// Handle CSV export
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    $conn = get_db_connection();
    if (!$conn) {
        http_response_code(500);
        die('Échec de la connexion à la base de données');
    }
    
    $start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-d', strtotime('-7 days'));
    $end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-d');
    $province_filter = isset($_GET['province_id']) ? (int)$_GET['province_id'] : null;
    
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $start_date)) {
        $start_date = date('Y-m-d', strtotime('-7 days'));
    }
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $end_date)) {
        $end_date = date('Y-m-d');
    }
    
    $start_datetime = $start_date . ' 00:00:00';
    $end_datetime = $end_date . ' 23:59:59';
    
    // Fetch invoices for export from all tables (ALL provinces, or filtered by province)
    $export_sql = "
        SELECT
            o.invoice_number,
            o.customer_name,
            o.paid_amount,
            o.payment_method,
            o.created_at,
            o.paid_status,
            o.serve_status,
            p.name AS product_name,
            p.crate_quantity,
            o.unit_type,
            o.quantity,
            u.full_name AS cashier_name,
            u.full_name AS initiator_name,
            NULL AS approver_name,
            prov.name AS province_name
        FROM `order` o
        INNER JOIN product p ON o.product_id = p.id
        INNER JOIN user u ON o.created_by = u.id
        LEFT JOIN province prov ON o.province_id = prov.id
        WHERE
            o.created_at BETWEEN ? AND ?
    ";
    
    $params = [$start_datetime, $end_datetime];
    $param_types = 'ss';
    
    if ($province_filter) {
        $export_sql .= " AND o.province_id = ?";
        $params[] = $province_filter;
        $param_types .= 'i';
    }
    
    $export_sql .= "
        UNION ALL
        
        SELECT
            ui.invoice_number,
            ui.customer_name,
            ui.paid_amount AS paid_amount,
            NULL AS payment_method,
            ui.created_at,
            CASE WHEN ui.status = 'paid' THEN 'Paid' ELSE 'Unpaid' END AS paid_status,
            CASE WHEN ui.status = 'served' THEN 'Served' ELSE 'Pending' END AS serve_status,
            (SELECT GROUP_CONCAT(p.name SEPARATOR ', ') FROM invoice_items ii JOIN product p ON ii.product_id = p.id WHERE ii.invoice_id = ui.id LIMIT 1) AS product_name,
            NULL AS crate_quantity,
            'bottles' AS unit_type,
            (SELECT SUM(ii.quantity) FROM invoice_items ii WHERE ii.invoice_id = ui.id) AS quantity,
            u.full_name AS cashier_name,
            u.full_name AS initiator_name,
            ua.full_name AS approver_name,
            prov.name AS province_name
        FROM unpaid_invoices ui
        LEFT JOIN user u ON ui.created_by = u.id
        LEFT JOIN user ua ON ui.approved_by = ua.id
        LEFT JOIN province prov ON ui.province_id = prov.id
        WHERE
            ui.created_at BETWEEN ? AND ?
    ";
    
    $params[] = $start_datetime;
    $params[] = $end_datetime;
    $param_types .= 'ss';
    
    if ($province_filter) {
        $export_sql .= " AND (ui.province_id = ? OR ui.province_id IS NULL)";
        $params[] = $province_filter;
        $param_types .= 'i';
    }
    
    $export_sql .= "
        UNION ALL
        
        SELECT
            pi.invoice_number,
            pi.customer_name,
            pi.paid_amount AS paid_amount,
            NULL AS payment_method,
            pi.created_at,
            'Paid' AS paid_status,
            CASE WHEN pi.status = 'served' THEN 'Served' ELSE 'Pending' END AS serve_status,
            (SELECT GROUP_CONCAT(p.name SEPARATOR ', ') FROM paid_invoice_items pii JOIN product p ON pii.product_id = p.id WHERE pii.invoice_id = pi.id LIMIT 1) AS product_name,
            NULL AS crate_quantity,
            'bottles' AS unit_type,
            (SELECT SUM(pii.quantity) FROM paid_invoice_items pii WHERE pii.invoice_id = pi.id) AS quantity,
            u.full_name AS cashier_name,
            u.full_name AS initiator_name,
            ua.full_name AS approver_name,
            prov.name AS province_name
        FROM paid_invoices pi
        LEFT JOIN user u ON pi.created_by = u.id
        LEFT JOIN user ua ON pi.approved_by = ua.id
        LEFT JOIN province prov ON pi.province_id = prov.id
        WHERE
            pi.created_at BETWEEN ? AND ?
    ";
    
    $params[] = $start_datetime;
    $params[] = $end_datetime;
    $param_types .= 'ss';
    
    if ($province_filter) {
        $export_sql .= " AND (pi.province_id = ? OR pi.province_id IS NULL)";
        $params[] = $province_filter;
        $param_types .= 'i';
    }
    
    $export_sql .= " ORDER BY created_at DESC";
    
    $stmt = $conn->prepare($export_sql);
    if ($stmt) {
        $stmt->bind_param($param_types, ...$params);
        $stmt->execute();
        $export_invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    } else {
        $export_invoices = [];
    }
    
    // Generate CSV
    $filename = 'all_invoices_' . $start_date . '_to_' . $end_date . '_' . date('YmdHis') . '.csv';
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Cache-Control: no-cache, no-store, must-revalidate');
    header('Pragma: no-cache');
    header('Expires: 0');
    
    $output = fopen('php://output', 'w');
    
    // Add BOM for UTF-8 Excel compatibility
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // Headers
    fputcsv($output, ['Date/Heure', 'Numéro Commande', 'Nom Client', 'Nom Produit', 'Quantité (Bouteilles)', 'Montant (BIF)', 'Méthode Paiement', 'Statut Paiement', 'Statut Service', 'Province', 'Caissier', 'Initiateur', 'Approbateur']);
    
    // Data rows
    foreach ($export_invoices as $invoice) {
        $unit_type = isset($invoice['unit_type']) ? $invoice['unit_type'] : 'bottles';
        $crate_quantity = isset($invoice['crate_quantity']) ? (int)$invoice['crate_quantity'] : 1;
        $quantity = isset($invoice['quantity']) ? (int)$invoice['quantity'] : 0;
        $quantity_bottles = ($unit_type === 'crates' && $crate_quantity > 0)
            ? $quantity * $crate_quantity
            : $quantity;
        
        $created_at = isset($invoice['created_at']) ? $invoice['created_at'] : 'now';
        $invoice_number = isset($invoice['invoice_number']) ? $invoice['invoice_number'] : 'N/A';
        $customer_name = isset($invoice['customer_name']) ? $invoice['customer_name'] : 'Client de passage';
        $product_name = isset($invoice['product_name']) ? $invoice['product_name'] : 'Inconnu';
        $payment_method = isset($invoice['payment_method']) ? $invoice['payment_method'] : 'N/A';
        $paid_status = isset($invoice['paid_status']) ? $invoice['paid_status'] : 'Non Payé';
        $serve_status = isset($invoice['serve_status']) ? $invoice['serve_status'] : 'En Attente';
        $cashier_name = isset($invoice['cashier_name']) ? $invoice['cashier_name'] : 'Inconnu';
        $initiator_name = isset($invoice['initiator_name']) ? $invoice['initiator_name'] : 'Inconnu';
        $approver_name = isset($invoice['approver_name']) ? $invoice['approver_name'] : 'N/A';
        $province_name = isset($invoice['province_name']) ? $invoice['province_name'] : 'N/A';
        
        // Translate statuses for CSV
        $paid_status_fr = ($paid_status === 'Paid') ? 'Payé' : (($paid_status === 'Unpaid') ? 'Non Payé' : $paid_status);
        $serve_status_fr = ($serve_status === 'Served') ? 'Servi' : (($serve_status === 'Pending') ? 'En Attente' : $serve_status);
        
        fputcsv($output, [
            date('Y-m-d H:i:s', strtotime($created_at)),
            $invoice_number,
            $customer_name,
            $product_name,
            $quantity_bottles,
            number_format((int)$invoice['paid_amount'], 0, ',', '.'),
            $payment_method,
            $paid_status_fr,
            $serve_status_fr,
            $province_name,
            $cashier_name,
            $initiator_name,
            $approver_name
        ]);
    }
    
    fclose($output);
    $conn->close();
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in admin invoices.php: " . mysqli_connect_error());
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Fetch all provinces for filter
$provinces = [];
$province_stmt = $conn->prepare("SELECT id, name FROM province ORDER BY name");
if ($province_stmt) {
    $province_stmt->execute();
    $province_result = $province_stmt->get_result();
    while ($row = $province_result->fetch_assoc()) {
        $provinces[] = $row;
    }
    $province_stmt->close();
}

$start_date = isset($_POST['start_date']) ? $_POST['start_date'] : date('Y-m-d', strtotime('-7 days'));
$end_date = isset($_POST['end_date']) ? $_POST['end_date'] : date('Y-m-d');
$selected_province_id = isset($_POST['province_id']) ? (int)$_POST['province_id'] : (isset($_GET['province_id']) ? (int)$_GET['province_id'] : null);

if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $start_date)) {
    $start_date = date('Y-m-d', strtotime('-7 days'));
}
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $end_date)) {
    $end_date = date('Y-m-d');
}

$start_datetime = $start_date . ' 00:00:00';
$end_datetime = $end_date . ' 23:59:59';

// Fetch filtered invoices from all tables (ALL provinces, or filtered by province)
$invoices_sql = "
    SELECT
        o.id,
        o.invoice_number,
        o.customer_name,
        o.paid_amount,
        o.payment_method,
        o.created_at,
        o.paid_status,
        o.serve_status,
        p.name AS product_name,
        p.crate_quantity,
        o.unit_type,
        o.quantity,
        u.full_name AS cashier_name,
        u.full_name AS initiator_name,
        NULL AS approver_name,
        prov.name AS province_name,
        'order' AS invoice_source
    FROM `order` o
    INNER JOIN product p ON o.product_id = p.id
    INNER JOIN user u ON o.created_by = u.id
    LEFT JOIN province prov ON o.province_id = prov.id
    WHERE
        o.created_at BETWEEN ? AND ?
";

$params = [$start_datetime, $end_datetime];
$param_types = 'ss';

if ($selected_province_id) {
    $invoices_sql .= " AND o.province_id = ?";
    $params[] = $selected_province_id;
    $param_types .= 'i';
}

$invoices_sql .= "
    UNION ALL
    
    SELECT
        ui.id,
        ui.invoice_number,
        ui.customer_name,
        ui.paid_amount AS paid_amount,
        NULL AS payment_method,
        ui.created_at,
        CASE WHEN ui.status = 'paid' THEN 'Paid' ELSE 'Unpaid' END AS paid_status,
        CASE WHEN ui.status = 'served' THEN 'Served' ELSE 'Pending' END AS serve_status,
        (SELECT GROUP_CONCAT(p.name SEPARATOR ', ') FROM invoice_items ii JOIN product p ON ii.product_id = p.id WHERE ii.invoice_id = ui.id LIMIT 1) AS product_name,
        NULL AS crate_quantity,
        'bottles' AS unit_type,
        (SELECT SUM(ii.quantity) FROM invoice_items ii WHERE ii.invoice_id = ui.id) AS quantity,
        u.full_name AS cashier_name,
        u.full_name AS initiator_name,
        ua.full_name AS approver_name,
        prov.name AS province_name,
        'unpaid' AS invoice_source
    FROM unpaid_invoices ui
    LEFT JOIN user u ON ui.created_by = u.id
    LEFT JOIN user ua ON ui.approved_by = ua.id
    LEFT JOIN province prov ON ui.province_id = prov.id
    WHERE
        ui.created_at BETWEEN ? AND ?
";

$params[] = $start_datetime;
$params[] = $end_datetime;
$param_types .= 'ss';

if ($selected_province_id) {
    $invoices_sql .= " AND (ui.province_id = ? OR ui.province_id IS NULL)";
    $params[] = $selected_province_id;
    $param_types .= 'i';
}

$invoices_sql .= "
    UNION ALL
    
    SELECT
        pi.id,
        pi.invoice_number,
        pi.customer_name,
        pi.paid_amount AS paid_amount,
        NULL AS payment_method,
        pi.created_at,
        'Paid' AS paid_status,
        CASE WHEN pi.status = 'served' THEN 'Served' ELSE 'Pending' END AS serve_status,
        (SELECT GROUP_CONCAT(p.name SEPARATOR ', ') FROM paid_invoice_items pii JOIN product p ON pii.product_id = p.id WHERE pii.invoice_id = pi.id LIMIT 1) AS product_name,
        NULL AS crate_quantity,
        'bottles' AS unit_type,
        (SELECT SUM(pii.quantity) FROM paid_invoice_items pii WHERE pii.invoice_id = pi.id) AS quantity,
        u.full_name AS cashier_name,
        u.full_name AS initiator_name,
        ua.full_name AS approver_name,
        prov.name AS province_name,
        'paid' AS invoice_source
    FROM paid_invoices pi
    LEFT JOIN user u ON pi.created_by = u.id
    LEFT JOIN user ua ON pi.approved_by = ua.id
    LEFT JOIN province prov ON pi.province_id = prov.id
    WHERE
        pi.created_at BETWEEN ? AND ?
";

$params[] = $start_datetime;
$params[] = $end_datetime;
$param_types .= 'ss';

if ($selected_province_id) {
    $invoices_sql .= " AND (pi.province_id = ? OR pi.province_id IS NULL)";
    $params[] = $selected_province_id;
    $param_types .= 'i';
}

$invoices_sql .= " ORDER BY created_at DESC";

$stmt = $conn->prepare($invoices_sql);
if (!$stmt) {
    error_log("Failed to prepare invoices query: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Failed to load invoices. Contact administrator.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}
$stmt->bind_param($param_types, ...$params);
$stmt->execute();
$invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Today's invoices from all tables
$today = date('Y-m-d');
$today_sql = "
    SELECT
        o.id,
        o.invoice_number,
        o.customer_name,
        o.paid_amount,
        o.payment_method,
        o.created_at,
        o.paid_status,
        o.serve_status,
        p.name AS product_name,
        p.crate_quantity,
        o.unit_type,
        o.quantity,
        u.full_name AS cashier_name,
        u.full_name AS initiator_name,
        NULL AS approver_name,
        prov.name AS province_name,
        'order' AS invoice_source
    FROM `order` o
    INNER JOIN product p ON o.product_id = p.id
    INNER JOIN user u ON o.created_by = u.id
    LEFT JOIN province prov ON o.province_id = prov.id
    WHERE
        DATE(o.created_at) = ?
";

$today_params = [$today];
$today_param_types = 's';

if ($selected_province_id) {
    $today_sql .= " AND o.province_id = ?";
    $today_params[] = $selected_province_id;
    $today_param_types .= 'i';
}

$today_sql .= "
    UNION ALL
    
    SELECT
        ui.id,
        ui.invoice_number,
        ui.customer_name,
        ui.paid_amount AS paid_amount,
        NULL AS payment_method,
        ui.created_at,
        CASE WHEN ui.status = 'paid' THEN 'Paid' ELSE 'Unpaid' END AS paid_status,
        CASE WHEN ui.status = 'served' THEN 'Served' ELSE 'Pending' END AS serve_status,
        (SELECT GROUP_CONCAT(p.name SEPARATOR ', ') FROM invoice_items ii JOIN product p ON ii.product_id = p.id WHERE ii.invoice_id = ui.id LIMIT 1) AS product_name,
        NULL AS crate_quantity,
        'bottles' AS unit_type,
        (SELECT SUM(ii.quantity) FROM invoice_items ii WHERE ii.invoice_id = ui.id) AS quantity,
        u.full_name AS cashier_name,
        u.full_name AS initiator_name,
        ua.full_name AS approver_name,
        prov.name AS province_name,
        'unpaid' AS invoice_source
    FROM unpaid_invoices ui
    LEFT JOIN user u ON ui.created_by = u.id
    LEFT JOIN user ua ON ui.approved_by = ua.id
    LEFT JOIN province prov ON ui.province_id = prov.id
    WHERE
        DATE(ui.created_at) = ?
";

$today_params[] = $today;
$today_param_types .= 's';

if ($selected_province_id) {
    $today_sql .= " AND (ui.province_id = ? OR ui.province_id IS NULL)";
    $today_params[] = $selected_province_id;
    $today_param_types .= 'i';
}

$today_sql .= "
    UNION ALL
    
    SELECT
        pi.id,
        pi.invoice_number,
        pi.customer_name,
        pi.paid_amount AS paid_amount,
        NULL AS payment_method,
        pi.created_at,
        'Paid' AS paid_status,
        CASE WHEN pi.status = 'served' THEN 'Served' ELSE 'Pending' END AS serve_status,
        (SELECT GROUP_CONCAT(p.name SEPARATOR ', ') FROM paid_invoice_items pii JOIN product p ON pii.product_id = p.id WHERE pii.invoice_id = pi.id LIMIT 1) AS product_name,
        NULL AS crate_quantity,
        'bottles' AS unit_type,
        (SELECT SUM(pii.quantity) FROM paid_invoice_items pii WHERE pii.invoice_id = pi.id) AS quantity,
        u.full_name AS cashier_name,
        u.full_name AS initiator_name,
        ua.full_name AS approver_name,
        prov.name AS province_name,
        'paid' AS invoice_source
    FROM paid_invoices pi
    LEFT JOIN user u ON pi.created_by = u.id
    LEFT JOIN user ua ON pi.approved_by = ua.id
    LEFT JOIN province prov ON pi.province_id = prov.id
    WHERE
        DATE(pi.created_at) = ?
";

$today_params[] = $today;
$today_param_types .= 's';

if ($selected_province_id) {
    $today_sql .= " AND (pi.province_id = ? OR pi.province_id IS NULL)";
    $today_params[] = $selected_province_id;
    $today_param_types .= 'i';
}

$today_sql .= " ORDER BY created_at DESC";

$stmt = $conn->prepare($today_sql);
if (!$stmt) {
    error_log("Failed to prepare today's invoices query: " . $conn->error);
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Failed to load today's invoices. Contact administrator.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}
$stmt->bind_param($today_param_types, ...$today_params);
$stmt->execute();
$todays_invoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

$conn->close();

require_once __DIR__ . '/../../includes/header.php';
?>
<style>
    @media (max-width: 768px) {
        .invoice-table-container {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
        }
        .invoice-table {
            min-width: 1200px;
        }
        .filter-form {
            grid-template-columns: 1fr !important;
        }
        .action-buttons {
            flex-direction: column;
            gap: 10px;
        }
        .action-buttons a, .action-buttons button {
            width: 100% !important;
        }
        .stats-grid {
            grid-template-columns: 1fr !important;
        }
    }
    @media (max-width: 480px) {
        .invoice-header {
            flex-direction: column;
            align-items: flex-start !important;
            gap: 15px;
        }
        .invoice-title {
            margin-bottom: 10px;
        }
    }
</style>
<div style="padding: 20px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh; width: 100%; box-sizing: border-box;">
    <div style="background-color: #F4F0E4; padding: 25px; border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.2); width: 100%; max-width: 100%; margin: 0; box-sizing: border-box;">
        <div class="invoice-header" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; flex-wrap: wrap; gap: 15px;">
            <div class="invoice-title">
                <h2 style="color: #4B2F1F; margin: 0 0 8px 0; font-size: 28px; font-weight: 700;">Toutes les Commandes - Vue Admin</h2>
                <p style="color: #666; font-size: 15px; margin: 0;">Voir toutes les commandes de toutes les provinces (Accès DG/DGA)</p>
            </div>
            <div class="action-buttons" style="display: flex; gap: 10px; align-items: center; flex-wrap: wrap;">
                <a href="/masunzu_bar_hotel/modules/admin/invoices.php?export=csv&start_date=<?php echo urlencode($start_date); ?>&end_date=<?php echo urlencode($end_date); ?><?php echo $selected_province_id ? '&province_id=' . urlencode($selected_province_id) : ''; ?>" 
                   style="padding: 12px 20px; background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%); color: #FFFFFF; border: none; border-radius: 8px; font-size: 15px; font-weight: 600; text-decoration: none; display: inline-flex; align-items: center; gap: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.2); transition: transform 0.2s;">
                    📥 Exporter en CSV
                </a>
                <a href="/masunzu_bar_hotel/dashboards/admin_dashboard.php" 
                   style="padding: 12px 20px; background-color: #6c757d; color: #FFFFFF; border: none; border-radius: 8px; font-size: 15px; font-weight: 600; text-decoration: none; display: inline-flex; align-items: center; gap: 8px;">
                    ← Retour au Tableau de Bord
                </a>
            </div>
        </div>

        <form method="POST" class="filter-form" style="display: grid; gap: 15px; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); margin-bottom: 25px; padding: 20px; background-color: #FFFFFF; border-radius: 10px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
            <div>
                <label style="color: #4B2F1F; font-size: 15px; font-weight: 600; display: block; margin-bottom: 8px;">Date de Début</label>
                <input type="date" name="start_date" value="<?php echo htmlspecialchars($start_date, ENT_QUOTES, 'UTF-8'); ?>" style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 15px; background-color: #FFFFFF; box-sizing: border-box;">
            </div>
            <div>
                <label style="color: #4B2F1F; font-size: 15px; font-weight: 600; display: block; margin-bottom: 8px;">Date de Fin</label>
                <input type="date" name="end_date" value="<?php echo htmlspecialchars($end_date, ENT_QUOTES, 'UTF-8'); ?>" style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 15px; background-color: #FFFFFF; box-sizing: border-box;">
            </div>
            <div>
                <label style="color: #4B2F1F; font-size: 15px; font-weight: 600; display: block; margin-bottom: 8px;">Province (Optionnel)</label>
                <select name="province_id" style="width: 100%; padding: 12px; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 15px; background-color: #FFFFFF; box-sizing: border-box;">
                    <option value="">Toutes les Provinces</option>
                    <?php foreach ($provinces as $province): ?>
                        <option value="<?php echo (int)$province['id']; ?>" <?php echo $selected_province_id === (int)$province['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($province['name'], ENT_QUOTES, 'UTF-8'); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div style="display: flex; align-items: flex-end;">
                <button type="submit" style="width: 100%; padding: 12px 20px; background: linear-gradient(135deg, #4B2F1F 0%, #5A3F2F 100%); color: #F4F0E4; border: none; border-radius: 8px; font-size: 16px; cursor: pointer; font-weight: 600; box-shadow: 0 2px 4px rgba(0,0,0,0.2); transition: transform 0.2s;">
                    🔍 Appliquer les Filtres
                </button>
            </div>
        </form>

        <?php
        // Calculate statistics
        $total_amount = 0;
        $total_invoices = count($invoices);
        $paid_count = 0;
        $pending_count = 0;
        foreach ($invoices as $inv) {
            $total_amount += (int)$inv['paid_amount'];
            if ((isset($inv['paid_status']) ? $inv['paid_status'] : '') === 'Paid') {
                $paid_count++;
            } else {
                $pending_count++;
            }
        }
        ?>
        <div class="stats-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 25px;">
            <div style="background: linear-gradient(135deg, #4B2F1F 0%, #6B4F2F 100%); color: #FFFFFF; padding: 20px; border-radius: 10px; box-shadow: 0 2px 8px rgba(0,0,0,0.15);">
                <div style="font-size: 13px; opacity: 0.9; margin-bottom: 5px;">Total Commandes</div>
                <div style="font-size: 28px; font-weight: 700;"><?php echo number_format($total_invoices, 0, ',', '.'); ?></div>
            </div>
            <div style="background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%); color: #FFFFFF; padding: 20px; border-radius: 10px; box-shadow: 0 2px 8px rgba(0,0,0,0.15);">
                <div style="font-size: 13px; opacity: 0.9; margin-bottom: 5px;">Montant Total</div>
                <div style="font-size: 28px; font-weight: 700;"><?php echo number_format($total_amount, 0, ',', '.'); ?> BIF</div>
            </div>
            <div style="background: linear-gradient(135deg, #F4A261 0%, #E76F51 100%); color: #FFFFFF; padding: 20px; border-radius: 10px; box-shadow: 0 2px 8px rgba(0,0,0,0.15);">
                <div style="font-size: 13px; opacity: 0.9; margin-bottom: 5px;">Payé</div>
                <div style="font-size: 28px; font-weight: 700;"><?php echo number_format($paid_count, 0, ',', '.'); ?></div>
            </div>
            <div style="background: linear-gradient(135deg, #F77F00 0%, #FCBF49 100%); color: #FFFFFF; padding: 20px; border-radius: 10px; box-shadow: 0 2px 8px rgba(0,0,0,0.15);">
                <div style="font-size: 13px; opacity: 0.9; margin-bottom: 5px;">En Attente</div>
                <div style="font-size: 28px; font-weight: 700;"><?php echo number_format($pending_count, 0, ',', '.'); ?></div>
            </div>
        </div>

        <!-- Today's Invoices -->
        <div style="background-color: #FFFFFF; padding: 20px; border-radius: 10px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); margin-bottom: 30px;">
            <h3 style="color: #4B2F1F; margin: 0 0 15px 0; font-size: 22px; font-weight: 700;">Commandes d'Aujourd'hui (<?php echo htmlspecialchars(date('Y-m-d'), ENT_QUOTES, 'UTF-8'); ?>)</h3>
            <input type="text" id="today-invoices-search" placeholder="🔍 Rechercher par commande, client, produit, province ou statut…" style="width: 100%; padding: 12px; border-radius: 8px; border: 2px solid #4B2F1F; margin-bottom: 15px; font-size: 15px; box-sizing: border-box;">
            <?php if (!empty($todays_invoices)): ?>
                <div class="invoice-table-container" style="overflow-x: auto; -webkit-overflow-scrolling: touch;">
                    <table id="today-invoices-table" class="invoice-table" style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 8px; overflow: hidden; min-width: 1200px;">
                        <thead>
                            <tr style="background: linear-gradient(135deg, #4B2F1F 0%, #5A3F2F 100%); color: #F4F0E4;">
                                <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Heure</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Commande</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Client</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Produit</th>
                                <th style="padding: 12px; text-align: right; font-weight: 600; font-size: 14px;">Quantité</th>
                                <th style="padding: 12px; text-align: right; font-weight: 600; font-size: 14px;">Montant (BIF)</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Paiement</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Province</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Caissier</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Initiateur</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Approbateur</th>
                                <th style="padding: 12px; text-align: center; font-weight: 600; font-size: 14px;">Statut</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($todays_invoices as $index => $invoice):
                                $unit_type = isset($invoice['unit_type']) ? $invoice['unit_type'] : 'bottles';
                                $crate_quantity = isset($invoice['crate_quantity']) ? (int)$invoice['crate_quantity'] : 1;
                                $quantity = isset($invoice['quantity']) ? (int)$invoice['quantity'] : 0;
                                $quantity_bottles = ($unit_type === 'crates' && $crate_quantity > 0)
                                    ? $quantity * $crate_quantity
                                    : $quantity;
                                $amount = isset($invoice['paid_amount']) ? (int)$invoice['paid_amount'] : 0;
                                $row_bg = $index % 2 === 0 ? '#FFFFFF' : '#FDFBF7';
                                $created_at = isset($invoice['created_at']) ? $invoice['created_at'] : 'now';
                                $invoice_number = isset($invoice['invoice_number']) ? $invoice['invoice_number'] : 'N/A';
                                $customer_name = isset($invoice['customer_name']) ? $invoice['customer_name'] : 'Client de passage';
                                $product_name = isset($invoice['product_name']) ? $invoice['product_name'] : 'Inconnu';
                                $payment_method = isset($invoice['payment_method']) ? $invoice['payment_method'] : 'N/A';
                                $paid_status = isset($invoice['paid_status']) ? $invoice['paid_status'] : 'Non Payé';
                                $serve_status = isset($invoice['serve_status']) ? $invoice['serve_status'] : 'En Attente';
                                $cashier_name = isset($invoice['cashier_name']) ? $invoice['cashier_name'] : 'Inconnu';
                                $initiator_name = isset($invoice['initiator_name']) ? $invoice['initiator_name'] : 'Inconnu';
                                $approver_name = isset($invoice['approver_name']) ? $invoice['approver_name'] : 'N/A';
                                $province_name = isset($invoice['province_name']) ? $invoice['province_name'] : 'N/A';
                            ?>
                                <tr style="background-color: <?php echo $row_bg; ?>; border-bottom: 1px solid #EFE8DE; transition: background-color 0.2s;">
                                    <td style="padding: 12px; color: #4B2F1F; font-weight: 600;"><?php echo htmlspecialchars(date('H:i', strtotime($created_at)), ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 12px; color: #2C1810;"><?php echo htmlspecialchars($invoice_number, ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 12px; color: #2C1810;"><?php echo htmlspecialchars($customer_name, ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 12px; color: #2C1810;"><?php echo htmlspecialchars($product_name, ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 12px; text-align: right; color: #4B2F1F; font-weight: 600;"><?php echo number_format($quantity_bottles, 0, ',', '.'); ?></td>
                                    <td style="padding: 12px; text-align: right; color: #2E7D32; font-weight: 700;"><?php echo number_format($amount, 0, ',', '.'); ?></td>
                                    <td style="padding: 12px; color: #666; text-transform: capitalize;"><?php echo htmlspecialchars(str_replace('_', ' ', $payment_method), ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 12px; color: #2C1810; font-size: 13px; font-weight: 600;"><?php echo htmlspecialchars($province_name, ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 12px; color: #2C1810; font-size: 13px;"><?php echo htmlspecialchars($cashier_name, ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 12px; color: #2C1810; font-size: 13px;"><?php echo htmlspecialchars($initiator_name, ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 12px; color: #666; font-size: 13px;"><?php echo htmlspecialchars($approver_name, ENT_QUOTES, 'UTF-8'); ?></td>
                                    <td style="padding: 12px; text-align: center;">
                                        <span style="padding: 6px 12px; border-radius: 6px; font-size: 12px; font-weight: 600; display: inline-block; <?php
                                            if ($paid_status === 'Paid') {
                                                echo 'background: linear-gradient(135deg, #4CAF50 0%, #66BB6A 100%); color: #FFFFFF;';
                                            } elseif ($serve_status === 'Served') {
                                                echo 'background: linear-gradient(135deg, #4B2F1F 0%, #5A3F2F 100%); color: #F4F0E4;';
                                            } else {
                                                echo 'background: linear-gradient(135deg, #F77F00 0%, #FCBF49 100%); color: #FFFFFF;';
                                            }
                                        ?>">
                                            <?php echo htmlspecialchars($paid_status === 'Paid' ? 'Payé' : ($serve_status === 'Served' ? 'Servi' : 'En Attente'), ENT_QUOTES, 'UTF-8'); ?>
                                        </span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <p style="color: #666; font-size: 15px; text-align: center; padding: 40px; background-color: #FDFBF7; border-radius: 8px; border: 2px dashed #D3C2B6;">Aucune commande créée aujourd'hui.</p>
            <?php endif; ?>
        </div>

        <!-- Filtered Invoices -->
        <?php if ($start_date !== date('Y-m-d', strtotime('-7 days')) || $end_date !== date('Y-m-d') || $selected_province_id): ?>
            <div style="background-color: #FFFFFF; padding: 20px; border-radius: 10px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); margin-bottom: 30px;">
                <h3 style="color: #4B2F1F; margin: 0 0 15px 0; font-size: 22px; font-weight: 700;">Commandes Filtrées (<?php echo htmlspecialchars($start_date, ENT_QUOTES, 'UTF-8'); ?> à <?php echo htmlspecialchars($end_date, ENT_QUOTES, 'UTF-8'); ?><?php echo $selected_province_id ? ' - ' . htmlspecialchars($provinces[array_search($selected_province_id, array_column($provinces, 'id'))]['name'] ?? 'Province Sélectionnée', ENT_QUOTES, 'UTF-8') : ' - Toutes les Provinces'; ?>)</h3>
                <?php if (!empty($invoices)): ?>
                    <div class="invoice-table-container" style="overflow-x: auto; -webkit-overflow-scrolling: touch;">
                        <table class="invoice-table" style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 8px; overflow: hidden; min-width: 1200px;">
                            <thead>
                                <tr style="background: linear-gradient(135deg, #4B2F1F 0%, #5A3F2F 100%); color: #F4F0E4;">
                                    <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Date/Heure</th>
                                    <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Commande</th>
                                    <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Client</th>
                                    <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Produit</th>
                                    <th style="padding: 12px; text-align: right; font-weight: 600; font-size: 14px;">Quantité</th>
                                    <th style="padding: 12px; text-align: right; font-weight: 600; font-size: 14px;">Montant (BIF)</th>
                                    <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Paiement</th>
                                    <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Province</th>
                                    <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Caissier</th>
                                    <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Initiateur</th>
                                    <th style="padding: 12px; text-align: left; font-weight: 600; font-size: 14px;">Approbateur</th>
                                    <th style="padding: 12px; text-align: center; font-weight: 600; font-size: 14px;">Statut</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($invoices as $index => $invoice):
                                    $unit_type = isset($invoice['unit_type']) ? $invoice['unit_type'] : 'bottles';
                                    $crate_quantity = isset($invoice['crate_quantity']) ? (int)$invoice['crate_quantity'] : 1;
                                    $quantity = isset($invoice['quantity']) ? (int)$invoice['quantity'] : 0;
                                    $quantity_bottles = ($unit_type === 'crates' && $crate_quantity > 0)
                                        ? $quantity * $crate_quantity
                                        : $quantity;
                                    $amount = isset($invoice['paid_amount']) ? (int)$invoice['paid_amount'] : 0;
                                    $row_bg = $index % 2 === 0 ? '#FFFFFF' : '#FDFBF7';
                                    $created_at = isset($invoice['created_at']) ? $invoice['created_at'] : 'now';
                                    $invoice_number = isset($invoice['invoice_number']) ? $invoice['invoice_number'] : 'N/A';
                                    $customer_name = isset($invoice['customer_name']) ? $invoice['customer_name'] : 'Client de passage';
                                    $product_name = isset($invoice['product_name']) ? $invoice['product_name'] : 'Inconnu';
                                    $payment_method = isset($invoice['payment_method']) ? $invoice['payment_method'] : 'N/A';
                                    $paid_status = isset($invoice['paid_status']) ? $invoice['paid_status'] : 'Non Payé';
                                    $serve_status = isset($invoice['serve_status']) ? $invoice['serve_status'] : 'En Attente';
                                    $cashier_name = isset($invoice['cashier_name']) ? $invoice['cashier_name'] : 'Inconnu';
                                    $initiator_name = isset($invoice['initiator_name']) ? $invoice['initiator_name'] : 'Inconnu';
                                    $approver_name = isset($invoice['approver_name']) ? $invoice['approver_name'] : 'N/A';
                                    $province_name = isset($invoice['province_name']) ? $invoice['province_name'] : 'N/A';
                                ?>
                                    <tr style="background-color: <?php echo $row_bg; ?>; border-bottom: 1px solid #EFE8DE; transition: background-color 0.2s;">
                                        <td style="padding: 12px; color: #4B2F1F; font-weight: 600;"><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($created_at)), ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td style="padding: 12px; color: #2C1810;"><?php echo htmlspecialchars($invoice_number, ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td style="padding: 12px; color: #2C1810;"><?php echo htmlspecialchars($customer_name, ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td style="padding: 12px; color: #2C1810;"><?php echo htmlspecialchars($product_name, ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td style="padding: 12px; text-align: right; color: #4B2F1F; font-weight: 600;"><?php echo number_format($quantity_bottles, 0, ',', '.'); ?></td>
                                        <td style="padding: 12px; text-align: right; color: #2E7D32; font-weight: 700;"><?php echo number_format($amount, 0, ',', '.'); ?></td>
                                        <td style="padding: 12px; color: #666; text-transform: capitalize;"><?php echo htmlspecialchars(str_replace('_', ' ', $payment_method), ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td style="padding: 12px; color: #2C1810; font-size: 13px; font-weight: 600;"><?php echo htmlspecialchars($province_name, ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td style="padding: 12px; color: #2C1810; font-size: 13px;"><?php echo htmlspecialchars($cashier_name, ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td style="padding: 12px; color: #2C1810; font-size: 13px;"><?php echo htmlspecialchars($initiator_name, ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td style="padding: 12px; color: #666; font-size: 13px;"><?php echo htmlspecialchars($approver_name, ENT_QUOTES, 'UTF-8'); ?></td>
                                        <td style="padding: 12px; text-align: center;">
                                            <span style="padding: 6px 12px; border-radius: 6px; font-size: 12px; font-weight: 600; display: inline-block; <?php
                                                if ($paid_status === 'Paid') {
                                                    echo 'background: linear-gradient(135deg, #4CAF50 0%, #66BB6A 100%); color: #FFFFFF;';
                                                } elseif ($serve_status === 'Served') {
                                                    echo 'background: linear-gradient(135deg, #4B2F1F 0%, #5A3F2F 100%); color: #F4F0E4;';
                                                } else {
                                                    echo 'background: linear-gradient(135deg, #F77F00 0%, #FCBF49 100%); color: #FFFFFF;';
                                                }
                                            ?>">
                                                <?php echo htmlspecialchars($paid_status === 'Paid' ? 'Payé' : ($serve_status === 'Served' ? 'Servi' : 'En Attente'), ENT_QUOTES, 'UTF-8'); ?>
                                            </span>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p style="color: #666; font-size: 15px; text-align: center; padding: 40px; background-color: #FDFBF7; border-radius: 8px; border: 2px dashed #D3C2B6;">Aucune commande trouvée pour la période sélectionnée.</p>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
</div>
<script>
document.addEventListener('DOMContentLoaded', function () {
    const searchInput = document.getElementById('today-invoices-search');
    const table = document.getElementById('today-invoices-table');
    if (!searchInput || !table) {
        return;
    }
    searchInput.addEventListener('input', function () {
        const term = this.value.toLowerCase();
        table.querySelectorAll('tbody tr').forEach(function (row) {
            const text = row.textContent.toLowerCase();
            row.style.display = text.includes(term) ? '' : 'none';
        });
    });
});
</script>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
