<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$baseDir = dirname(__DIR__, 2);
require_once $baseDir . '/includes/auth.php';
require_once $baseDir . '/includes/functions.php';
require_once $baseDir . '/config/database.php';

require_login();

$roleName = $_SESSION['role_name'] ?? '';
$province_id = $_SESSION['province_id'] ?? 0;
$user_id = $_SESSION['user_id'] ?? 0;

if (!in_array(mbh_normalize_role_name($roleName), ['DG', 'DGA', 'Board', 'Admin'], true)) {
    header('Location: /masunzu_bar_hotel/dashboards/admin_dashboard.php');
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    die('Database connection failed: ' . mysqli_connect_error());
}

$today = date('Y-m-d');
$defaultStart = date('Y-m-d', strtotime('-30 days'));

$startDate = filter_input(INPUT_GET, 'start_date', FILTER_SANITIZE_FULL_SPECIAL_CHARS) ?: $defaultStart;
$endDate = filter_input(INPUT_GET, 'end_date', FILTER_SANITIZE_FULL_SPECIAL_CHARS) ?: $today;
$provinceFilter = filter_input(INPUT_GET, 'province_id', FILTER_VALIDATE_INT, ['options' => ['default' => 0]]);
$exportTarget = strtolower(trim(filter_input(INPUT_GET, 'export', FILTER_SANITIZE_FULL_SPECIAL_CHARS) ?? ''));

$startDateTime = $startDate . ' 00:00:00';
$endDateTime = $endDate . ' 23:59:59';

$hasProvinceFilter = $provinceFilter > 0 && $roleName === 'Board';
$provinceParam = $provinceFilter;

if (!$hasProvinceFilter && $province_id > 0 && $roleName !== 'Board') {
    $hasProvinceFilter = true;
    $provinceParam = $province_id;
}

$provinces = [];
$provinceStmt = $conn->prepare('SELECT id, name FROM province ORDER BY name');
if ($provinceStmt) {
    $provinceStmt->execute();
    $provinces = $provinceStmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $provinceStmt->close();
}

function appendProvinceFilter(string $query, bool $hasProvinceFilter, string $column = 'province_id'): string {
    if (!$hasProvinceFilter) return $query;
    $clause = " AND {$column} = ?";
    if (stripos($query, 'WHERE') === false) {
        $clause = " WHERE {$column} = ?";
    }
    return $query . $clause;
}

$currentProvinceName = 'All Provinces';
if ($hasProvinceFilter && $provinceParam > 0) {
    $stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
    $stmt->bind_param("i", $provinceParam);
    $stmt->execute();
    $res = $stmt->get_result()->fetch_assoc();
    $currentProvinceName = $res['name'] ?? 'Unknown';
    $stmt->close();
}

if ($exportTarget) {
    handleExport($exportTarget, $conn, $startDate, $endDate, $startDateTime, $endDateTime, $hasProvinceFilter, $provinceParam, $currentProvinceName);
    exit;
}

$salesByProvince = [];
$query = "SELECT p.name as province_name,
                 COUNT(pi.id) as sales_count,
                 SUM(pi.paid_amount) as total_sales,
                 SUM(r.total_amount) as total_requisitions
          FROM province p
          LEFT JOIN paid_invoices pi ON p.id = pi.province_id AND COALESCE(pi.paid_at, pi.created_at) BETWEEN ? AND ?
          LEFT JOIN requisition r ON p.id = r.province_id AND r.status = 'paid' AND r.created_at BETWEEN ? AND ?
          GROUP BY p.id, p.name ORDER BY p.name";
// Note: Sales by Province is grouped by province, not time-ordered
$stmt = $conn->prepare($query);
if ($stmt) {
    $stmt->bind_param("ssss", $startDateTime, $endDateTime, $startDateTime, $endDateTime);
    $stmt->execute();
    $salesByProvince = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

$cashierBalances = [];
$query = "SELECT p.name as province_name,
                 u.full_name as cashier_name,
                 (SELECT end_balance FROM cashier_balance WHERE cashier_id = u.id AND shift_end IS NOT NULL ORDER BY shift_end DESC LIMIT 1) as last_expected_balance,
                 (SELECT counted_amount FROM cashier_shift_closure csc 
                  INNER JOIN cashier_balance cb ON csc.shift_id = cb.id 
                  WHERE cb.cashier_id = u.id ORDER BY cb.shift_end DESC LIMIT 1) as last_counted_cash,
                 (SELECT difference_amount FROM cashier_shift_closure csc 
                  INNER JOIN cashier_balance cb ON csc.shift_id = cb.id 
                  WHERE cb.cashier_id = u.id ORDER BY cb.shift_end DESC LIMIT 1) as last_difference
          FROM province p
          LEFT JOIN user u ON p.id = u.province_id AND u.role_id IN (SELECT id FROM role WHERE name = 'Caissier_Comptable')
          WHERE u.id IS NOT NULL
          ORDER BY p.name, u.full_name";
$stmt = $conn->prepare($query);
if ($stmt) {
    $stmt->execute();
    $cashierBalances = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

$accountantBalances = [];
$query = "SELECT u.id AS accountant_id,
                 u.full_name AS accountant_name,
                 p.name AS province_name,
                 COALESCE(ab.balance, 0) AS balance,
                 COALESCE(SUM(e.amount), 0) AS total_expenses,
                 COALESCE(ab.balance, 0) - COALESCE(SUM(e.amount), 0) AS available_balance,
                 ab.last_updated
          FROM user u
          JOIN role r ON u.role_id = r.id
          LEFT JOIN province p ON u.province_id = p.id
          LEFT JOIN accountant_balance ab ON u.id = ab.accountant_id
          LEFT JOIN expense e ON u.id = e.accountant_id
          WHERE r.name = 'Accountant'
          GROUP BY u.id, u.full_name, p.name, ab.balance, ab.last_updated
          ORDER BY p.name, u.full_name";
$stmt = $conn->prepare($query);
if ($stmt) {
    $stmt->execute();
    $accountantBalances = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

$pendingInvoices = [];
$query = "SELECT ui.id, ui.status, ui.approval_status, ui.created_at, ui.paid_amount AS total_amount,
                 ui.invoice_number,
                 u.full_name AS initiator_name, p.name AS province_name
          FROM unpaid_invoices ui
          LEFT JOIN user u ON ui.created_by = u.id
          LEFT JOIN province p ON ui.province_id = p.id
          WHERE (ui.approval_status = 'pending' OR ui.approval_status IS NULL) 
          AND ui.status = 'pending'
          AND ui.created_at BETWEEN ? AND ?";
if ($hasProvinceFilter) {
    $query .= " AND ui.province_id = ?";
}
$query .= " ORDER BY ui.created_at DESC LIMIT 50";
$stmt = $conn->prepare($query);
if ($stmt) {
    if ($hasProvinceFilter) {
        $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
    } else {
        $stmt->bind_param("ss", $startDateTime, $endDateTime);
    }
    $stmt->execute();
    $pendingInvoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

$requisitions = [];
$query = "SELECT r.id, r.status, r.total_amount, r.created_at,
                 u.full_name AS initiator_name, p.name AS province_name
          FROM requisition r
          LEFT JOIN user u ON r.user_id = u.id
          LEFT JOIN province p ON r.province_id = p.id
          WHERE r.created_at BETWEEN ? AND ?";
if ($hasProvinceFilter) {
    $query .= " AND r.province_id = ?";
}
$query .= " ORDER BY r.created_at DESC LIMIT 50";
$stmt = $conn->prepare($query);
if ($stmt) {
    if ($hasProvinceFilter) {
        $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
    } else {
        $stmt->bind_param("ss", $startDateTime, $endDateTime);
    }
    $stmt->execute();
    $requisitions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

$paidInvoices = [];
$query = "SELECT pi.id, pi.created_at, pi.paid_amount,
                 u.full_name AS cashier_name, p.name AS province_name
          FROM paid_invoices pi
          JOIN user u ON pi.created_by = u.id
          JOIN province p ON pi.province_id = p.id
          WHERE COALESCE(pi.paid_at, pi.created_at) BETWEEN ? AND ?";
if ($hasProvinceFilter) {
    $query .= " AND pi.province_id = ?";
}
$query .= " ORDER BY pi.created_at DESC LIMIT 50";
$stmt = $conn->prepare($query);
if ($stmt) {
    if ($hasProvinceFilter) {
        $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
    } else {
        $stmt->bind_param("ss", $startDateTime, $endDateTime);
    }
    $stmt->execute();
    $paidInvoices = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

$topProducts = [];
$query = "SELECT p.name, SUM(pii.quantity) as total_quantity, SUM(pii.quantity * pii.unit_price) as total_amount
          FROM paid_invoice_items pii
          INNER JOIN paid_invoices pi ON pi.id = pii.invoice_id
          INNER JOIN product p ON p.id = pii.product_id
          WHERE COALESCE(pi.paid_at, pi.created_at) BETWEEN ? AND ?";
if ($hasProvinceFilter) {
    $query .= " AND pi.province_id = ?";
}
$query .= " GROUP BY p.id, p.name ORDER BY total_quantity DESC LIMIT 15";
$stmt = $conn->prepare($query);
if ($stmt) {
    if ($hasProvinceFilter) {
        $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
    } else {
        $stmt->bind_param("ss", $startDateTime, $endDateTime);
    }
    $stmt->execute();
    $topProducts = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

$expenses = [];
$query = "SELECT e.id, e.amount, e.payment_method, e.created_at,
                 COALESCE(r.recipient_external_name, recip.full_name, 'N/A') AS recipient_name,
                 u.full_name AS accountant_name, p.name AS province_name
          FROM expense e
          LEFT JOIN requisition r ON e.requisition_id = r.id
          LEFT JOIN user recip ON r.recipient_id = recip.id
          LEFT JOIN user u ON e.accountant_id = u.id
          LEFT JOIN province p ON r.province_id = p.id
          WHERE e.created_at BETWEEN ? AND ?";
if ($hasProvinceFilter) {
    $query .= " AND r.province_id = ?";
}
$query .= " ORDER BY e.created_at DESC LIMIT 50";
$stmt = $conn->prepare($query);
if ($stmt) {
    if ($hasProvinceFilter) {
        $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
    } else {
        $stmt->bind_param("ss", $startDateTime, $endDateTime);
    }
    $stmt->execute();
    $expenses = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

$loanPayments = [];
$query = "SELECT lp.id, lp.payment_date, lp.payment_amount, lp.payment_method,
                 c.full_name AS customer_name, p.name AS province_name
          FROM loan_payment lp
          LEFT JOIN customer c ON lp.customer_id = c.id
          LEFT JOIN province p ON lp.province_id = p.id
          WHERE lp.payment_date BETWEEN ? AND ?";
if ($hasProvinceFilter) {
    $query .= " AND lp.province_id = ?";
}
$query .= " ORDER BY lp.payment_date DESC LIMIT 50";
$stmt = $conn->prepare($query);
if ($stmt) {
    if ($hasProvinceFilter) {
        $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
    } else {
        $stmt->bind_param("ss", $startDateTime, $endDateTime);
    }
    $stmt->execute();
    $loanPayments = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

// Unpaid Loans Report
$unpaidLoans = [];
$query = "SELECT 
            ui.id,
            ui.invoice_number,
            COALESCE(c.full_name, ui.customer_name) AS customer_name,
            ui.loan_outstanding_bif,
            ui.loan_due_date,
            ui.created_at,
            p.name AS province_name,
            CASE 
                WHEN ui.loan_due_date IS NOT NULL AND ui.loan_due_date < CURDATE() THEN 'Late'
                WHEN ui.loan_due_date IS NOT NULL THEN 'Pending'
                ELSE 'No Due Date'
            END AS payment_status
          FROM unpaid_invoices ui
          LEFT JOIN customer c ON ui.customer_id = c.id
          LEFT JOIN province p ON ui.province_id = p.id
          WHERE ui.is_loan_sale = 1 
          AND ui.loan_outstanding_bif > 0 
          AND (ui.approval_status IS NULL OR ui.approval_status != 'rejected')
          AND ui.created_at BETWEEN ? AND ?";
if ($hasProvinceFilter) {
    $query .= " AND ui.province_id = ?";
}
$query .= " ORDER BY ui.created_at DESC, ui.loan_due_date ASC LIMIT 100";
$stmt = $conn->prepare($query);
if ($stmt) {
    if ($hasProvinceFilter) {
        $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
    } else {
        $stmt->bind_param("ss", $startDateTime, $endDateTime);
    }
    $stmt->execute();
    $unpaidLoans = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

// Late Payments Report
$latePayments = [];
$query = "SELECT 
            ui.id,
            ui.invoice_number,
            COALESCE(c.full_name, ui.customer_name) AS customer_name,
            ui.loan_outstanding_bif,
            ui.loan_due_date,
            DATEDIFF(CURDATE(), ui.loan_due_date) AS days_overdue,
            ui.created_at,
            p.name AS province_name
          FROM unpaid_invoices ui
          LEFT JOIN customer c ON ui.customer_id = c.id
          LEFT JOIN province p ON ui.province_id = p.id
          WHERE ui.is_loan_sale = 1 
          AND ui.loan_outstanding_bif > 0 
          AND ui.loan_due_date IS NOT NULL
          AND ui.loan_due_date < CURDATE()
          AND (ui.approval_status IS NULL OR ui.approval_status != 'rejected')
          AND ui.created_at BETWEEN ? AND ?";
if ($hasProvinceFilter) {
    $query .= " AND ui.province_id = ?";
}
$query .= " ORDER BY ui.created_at DESC, ui.loan_due_date ASC LIMIT 100";
$stmt = $conn->prepare($query);
if ($stmt) {
    if ($hasProvinceFilter) {
        $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
    } else {
        $stmt->bind_param("ss", $startDateTime, $endDateTime);
    }
    $stmt->execute();
    $latePayments = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

// Customer Loan Status Report
$customerLoanStatus = [];
$query = "SELECT 
            c.id,
            c.full_name,
            c.loan_limit_bif,
            c.loan_balance_bif,
            c.loan_status,
            c.can_pay_by_loan,
            COALESCE(SUM(CASE WHEN ui.is_loan_sale = 1 AND ui.loan_outstanding_bif > 0 AND (ui.approval_status IS NULL OR ui.approval_status != 'rejected') THEN ui.loan_outstanding_bif ELSE 0 END), 0) as total_unpaid,
            COUNT(CASE WHEN ui.is_loan_sale = 1 AND ui.loan_outstanding_bif > 0 AND (ui.approval_status IS NULL OR ui.approval_status != 'rejected') AND ui.loan_due_date < CURDATE() THEN 1 END) as late_invoice_count,
            (SELECT p.name FROM unpaid_invoices ui2 
             LEFT JOIN province p ON ui2.province_id = p.id 
             WHERE (ui2.customer_id = c.id OR (ui2.customer_id IS NULL AND ui2.customer_name = c.full_name AND ui2.phone_number = c.tel))
             AND ui2.is_loan_sale = 1 
             LIMIT 1) AS province_name
          FROM customer c
          LEFT JOIN unpaid_invoices ui ON (ui.customer_id = c.id OR (ui.customer_id IS NULL AND ui.customer_name = c.full_name AND ui.phone_number = c.tel))
          WHERE (c.can_pay_by_loan = 1 OR c.loan_balance_bif > 0)";
if ($hasProvinceFilter) {
    $query .= " AND EXISTS (
        SELECT 1 FROM unpaid_invoices ui3 
        WHERE (ui3.customer_id = c.id OR (ui3.customer_id IS NULL AND ui3.customer_name = c.full_name AND ui3.phone_number = c.tel))
        AND ui3.province_id = ?
    )";
}
$query .= " GROUP BY c.id, c.full_name, c.loan_limit_bif, c.loan_balance_bif, c.loan_status, c.can_pay_by_loan
          ORDER BY total_unpaid DESC, c.full_name
          LIMIT 100";
$stmt = $conn->prepare($query);
if ($stmt) {
    if ($hasProvinceFilter) {
        $stmt->bind_param("i", $provinceParam);
    }
    $stmt->execute();
    $customerLoanStatus = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

// Shift Closures Report
$shiftClosures = [];
$query = "SELECT 
            cb.id,
            cb.shift_start,
            cb.shift_end,
            cb.beginning_balance,
            cb.end_balance,
            cb.deposits,
            cb.withdrawals,
            csc.counted_amount,
            csc.difference_amount,
            u.full_name AS cashier_name,
            p.name AS province_name
          FROM cashier_balance cb
          LEFT JOIN cashier_shift_closure csc ON csc.shift_id = cb.id
          LEFT JOIN user u ON cb.cashier_id = u.id
          LEFT JOIN province p ON u.province_id = p.id
          WHERE cb.shift_end IS NOT NULL
          AND cb.shift_end BETWEEN ? AND ?";
if ($hasProvinceFilter) {
    $query .= " AND u.province_id = ?";
}
$query .= " ORDER BY cb.shift_end DESC LIMIT 50";
$stmt = $conn->prepare($query);
if ($stmt) {
    if ($hasProvinceFilter) {
        $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
    } else {
        $stmt->bind_param("ss", $startDateTime, $endDateTime);
    }
    $stmt->execute();
    $shiftClosures = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

// Stock Addition History Report
$stockAdditions = [];
$query = "SELECT 
            st.id,
            st.quantity,
            st.unit_type,
            st.sent_at,
            p.name AS product_name,
            pr.crate_quantity,
            u.full_name AS added_by_name
          FROM stock_transfer st
          LEFT JOIN product p ON st.product_id = p.id
          LEFT JOIN product pr ON st.product_id = pr.id
          LEFT JOIN user u ON st.operation_manager_id = u.id
          WHERE st.status = 'confirmed'
          AND st.sent_at BETWEEN ? AND ?
          ORDER BY st.sent_at DESC LIMIT 100";
$stmt = $conn->prepare($query);
if ($stmt) {
    $stmt->bind_param("ss", $startDateTime, $endDateTime);
    $stmt->execute();
    $stockAdditions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

$conn->close();

function handleExport($target, $conn, $startDate, $endDate, $startDateTime, $endDateTime, $hasProvinceFilter, $provinceParam, $currentProvinceName) {
    $timeframe = "Report Period: {$startDate} to {$endDate} | Province: {$currentProvinceName}";
    
    header('Content-Type: text/csv; charset=utf-8');
    
    if ($target === 'sales_province') {
        header('Content-Disposition: attachment; filename=sales_by_province_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['Province', 'Sales Count', 'Total Sales (BIF)', 'Total Requisitions (BIF)']);
        
        $query = "SELECT p.name, COUNT(pi.id) as sales_count, SUM(pi.paid_amount) as total_sales,
                         SUM(r.total_amount) as total_requisitions
                  FROM province p
                  LEFT JOIN paid_invoices pi ON p.id = pi.province_id AND COALESCE(pi.paid_at, pi.created_at) BETWEEN ? AND ?
                  LEFT JOIN requisition r ON p.id = r.province_id AND r.status = 'paid' AND r.created_at BETWEEN ? AND ?
                  GROUP BY p.id ORDER BY p.name";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            $stmt->bind_param("ssss", $startDateTime, $endDateTime, $startDateTime, $endDateTime);
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                fputcsv($out, [$row['name'], $row['sales_count'] ?? 0, $row['total_sales'] ?? 0, $row['total_requisitions'] ?? 0]);
            }
            $stmt->close();
        }
        fclose($out);
    } elseif ($target === 'cashier_balances') {
        header('Content-Disposition: attachment; filename=cashier_balances_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['Province', 'Caissier_Comptable Name', 'Last Expected Balance (BIF)', 'Last Counted Cash (BIF)', 'Last Difference (BIF)']);
        
        $query = "SELECT p.name, u.full_name,
                         (SELECT end_balance FROM cashier_balance WHERE cashier_id = u.id AND shift_end IS NOT NULL ORDER BY shift_end DESC LIMIT 1) as last_expected_balance,
                         (SELECT counted_amount FROM cashier_shift_closure csc 
                          INNER JOIN cashier_balance cb ON csc.shift_id = cb.id 
                          WHERE cb.cashier_id = u.id ORDER BY cb.shift_end DESC LIMIT 1) as last_counted_cash,
                         (SELECT difference_amount FROM cashier_shift_closure csc 
                          INNER JOIN cashier_balance cb ON csc.shift_id = cb.id 
                          WHERE cb.cashier_id = u.id ORDER BY cb.shift_end DESC LIMIT 1) as last_difference
                  FROM province p
                  LEFT JOIN user u ON p.id = u.province_id AND u.role_id IN (SELECT id FROM role WHERE name = 'Caissier_Comptable')
                  WHERE u.id IS NOT NULL ORDER BY p.name, u.full_name";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                fputcsv($out, [
                    $row['name'], 
                    $row['full_name'], 
                    $row['last_expected_balance'] ?? 0,
                    $row['last_counted_cash'] ?? 'N/A',
                    $row['last_difference'] ?? 'N/A'
                ]);
            }
            $stmt->close();
        }
        fclose($out);
    } elseif ($target === 'accountant_balances') {
        header('Content-Disposition: attachment; filename=accountant_balances_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['Accountant', 'Province', 'Total Received (BIF)', 'Total Expenses (BIF)', 'Available Balance (BIF)', 'Last Updated']);
        
        $query = "SELECT u.full_name, p.name, COALESCE(ab.balance, 0) as balance,
                         COALESCE(SUM(e.amount), 0) as total_expenses,
                         COALESCE(ab.balance, 0) - COALESCE(SUM(e.amount), 0) as available_balance,
                         ab.last_updated
                  FROM user u
                  JOIN role r ON u.role_id = r.id
                  LEFT JOIN province p ON u.province_id = p.id
                  LEFT JOIN accountant_balance ab ON u.id = ab.accountant_id
                  LEFT JOIN expense e ON u.id = e.accountant_id
                  WHERE r.name = 'Accountant'
                  GROUP BY u.id ORDER BY p.name, u.full_name";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                fputcsv($out, [$row['full_name'], $row['name'] ?? 'N/A', $row['balance'], $row['total_expenses'], 
                              max(0, (float)$row['available_balance']), $row['last_updated'] ?? 'Never']);
            }
            $stmt->close();
        }
        fclose($out);
    } elseif ($target === 'requisitions') {
        header('Content-Disposition: attachment; filename=requisitions_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['ID', 'Status', 'Amount (BIF)', 'Province', 'Initiator', 'Created Date']);
        
        $query = "SELECT r.id, r.status, r.total_amount, p.name, u.full_name, r.created_at
                  FROM requisition r
                  LEFT JOIN user u ON r.user_id = u.id
                  LEFT JOIN province p ON r.province_id = p.id
                  WHERE r.created_at BETWEEN ? AND ?";
        if ($hasProvinceFilter) {
            $query .= " AND r.province_id = ?";
        }
        $query .= " ORDER BY r.created_at DESC";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            if ($hasProvinceFilter) {
                $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
            } else {
                $stmt->bind_param("ss", $startDateTime, $endDateTime);
            }
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                fputcsv($out, [$row['id'], $row['status'], $row['total_amount'] ?? 0, $row['name'] ?? 'N/A', 
                              $row['full_name'] ?? 'N/A', $row['created_at']]);
            }
            $stmt->close();
        }
        fclose($out);
    } elseif ($target === 'paid_invoices') {
        header('Content-Disposition: attachment; filename=paid_invoices_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['ID', 'Amount (BIF)', 'Province', 'Caissier_Comptable', 'Created Date']);
        
        $query = "SELECT pi.id, pi.paid_amount, p.name, u.full_name, pi.created_at
                  FROM paid_invoices pi
                  JOIN user u ON pi.created_by = u.id
                  JOIN province p ON pi.province_id = p.id
                  WHERE COALESCE(pi.paid_at, pi.created_at) BETWEEN ? AND ?";
        if ($hasProvinceFilter) {
            $query .= " AND pi.province_id = ?";
        }
        $query .= " ORDER BY pi.created_at DESC";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            if ($hasProvinceFilter) {
                $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
            } else {
                $stmt->bind_param("ss", $startDateTime, $endDateTime);
            }
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                fputcsv($out, [$row['id'], $row['paid_amount'] ?? 0, $row['name'], $row['full_name'], $row['created_at']]);
            }
            $stmt->close();
        }
        fclose($out);
    } elseif ($target === 'top_products') {
        header('Content-Disposition: attachment; filename=top_products_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['Product Name', 'Total Quantity', 'Total Amount (BIF)']);
        
        $query = "SELECT p.name, SUM(pii.quantity) as total_quantity, SUM(pii.quantity * pii.unit_price) as total_amount
                  FROM paid_invoice_items pii
                  INNER JOIN paid_invoices pi ON pi.id = pii.invoice_id
                  INNER JOIN product p ON p.id = pii.product_id
                  WHERE COALESCE(pi.paid_at, pi.created_at) BETWEEN ? AND ?";
        if ($hasProvinceFilter) {
            $query .= " AND pi.province_id = ?";
        }
        $query .= " GROUP BY p.id ORDER BY total_quantity DESC";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            if ($hasProvinceFilter) {
                $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
            } else {
                $stmt->bind_param("ss", $startDateTime, $endDateTime);
            }
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                fputcsv($out, [$row['name'], $row['total_quantity'] ?? 0, $row['total_amount'] ?? 0]);
            }
            $stmt->close();
        }
        fclose($out);
    } elseif ($target === 'expenses') {
        header('Content-Disposition: attachment; filename=expenses_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['ID', 'Amount (BIF)', 'Method', 'Recipient', 'Accountant', 'Province', 'Date']);
        
        $query = "SELECT e.id, e.amount, e.payment_method, COALESCE(r.recipient_external_name, recip.full_name, 'N/A') as recipient_name,
                         u.full_name as accountant_name, p.name as province_name, e.created_at
                  FROM expense e
                  LEFT JOIN requisition r ON e.requisition_id = r.id
                  LEFT JOIN user recip ON r.recipient_id = recip.id
                  LEFT JOIN user u ON e.accountant_id = u.id
                  LEFT JOIN province p ON r.province_id = p.id
                  WHERE e.created_at BETWEEN ? AND ?";
        if ($hasProvinceFilter) {
            $query .= " AND r.province_id = ?";
        }
        $query .= " ORDER BY e.created_at DESC";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            if ($hasProvinceFilter) {
                $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
            } else {
                $stmt->bind_param("ss", $startDateTime, $endDateTime);
            }
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                fputcsv($out, [$row['id'], $row['amount'] ?? 0, $row['payment_method'], $row['recipient_name'], 
                              $row['accountant_name'] ?? 'N/A', $row['province_name'] ?? 'N/A', $row['created_at']]);
            }
            $stmt->close();
        }
        fclose($out);
    } elseif ($target === 'loan_payments') {
        header('Content-Disposition: attachment; filename=loan_payments_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['ID', 'Customer', 'Amount (BIF)', 'Method', 'Province', 'Payment Date']);
        
        $query = "SELECT lp.id, c.full_name, lp.payment_amount, lp.payment_method, p.name, lp.payment_date
                  FROM loan_payment lp
                  LEFT JOIN customer c ON lp.customer_id = c.id
                  LEFT JOIN province p ON lp.province_id = p.id
                  WHERE lp.payment_date BETWEEN ? AND ?";
        if ($hasProvinceFilter) {
            $query .= " AND lp.province_id = ?";
        }
        $query .= " ORDER BY lp.payment_date DESC";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            if ($hasProvinceFilter) {
                $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
            } else {
                $stmt->bind_param("ss", $startDateTime, $endDateTime);
            }
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                fputcsv($out, [$row['id'], $row['full_name'] ?? 'N/A', $row['payment_amount'] ?? 0, $row['payment_method'], 
                              $row['name'] ?? 'N/A', $row['payment_date']]);
            }
            $stmt->close();
        }
        fclose($out);
    } elseif ($target === 'unpaid_loans') {
        header('Content-Disposition: attachment; filename=unpaid_loans_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['Invoice Number', 'Customer', 'Outstanding Amount (BIF)', 'Due Date', 'Status', 'Province', 'Created Date']);
        
        $query = "SELECT 
                    ui.invoice_number,
                    COALESCE(c.full_name, ui.customer_name) AS customer_name,
                    ui.loan_outstanding_bif,
                    ui.loan_due_date,
                    CASE 
                        WHEN ui.loan_due_date IS NOT NULL AND ui.loan_due_date < CURDATE() THEN 'Late'
                        WHEN ui.loan_due_date IS NOT NULL THEN 'Pending'
                        ELSE 'No Due Date'
                    END AS payment_status,
                    p.name AS province_name,
                    ui.created_at
                  FROM unpaid_invoices ui
                  LEFT JOIN customer c ON ui.customer_id = c.id
                  LEFT JOIN province p ON ui.province_id = p.id
                  WHERE ui.is_loan_sale = 1 
                  AND ui.loan_outstanding_bif > 0 
                  AND (ui.approval_status IS NULL OR ui.approval_status != 'rejected')
                  AND ui.created_at BETWEEN ? AND ?";
        if ($hasProvinceFilter) {
            $query .= " AND ui.province_id = ?";
        }
        $query .= " ORDER BY ui.created_at DESC, ui.loan_due_date ASC";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            if ($hasProvinceFilter) {
                $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
            } else {
                $stmt->bind_param("ss", $startDateTime, $endDateTime);
            }
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                fputcsv($out, [
                    $row['invoice_number'] ?? 'N/A',
                    $row['customer_name'] ?? 'N/A',
                    $row['loan_outstanding_bif'] ?? 0,
                    $row['loan_due_date'] ?? 'N/A',
                    $row['payment_status'] ?? 'N/A',
                    $row['province_name'] ?? 'N/A',
                    $row['created_at'] ?? 'N/A'
                ]);
            }
            $stmt->close();
        }
        fclose($out);
    } elseif ($target === 'late_payments') {
        header('Content-Disposition: attachment; filename=late_payments_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['Invoice Number', 'Customer', 'Outstanding Amount (BIF)', 'Due Date', 'Days Overdue', 'Province', 'Created Date']);
        
        $query = "SELECT 
                    ui.invoice_number,
                    COALESCE(c.full_name, ui.customer_name) AS customer_name,
                    ui.loan_outstanding_bif,
                    ui.loan_due_date,
                    DATEDIFF(CURDATE(), ui.loan_due_date) AS days_overdue,
                    p.name AS province_name,
                    ui.created_at
                  FROM unpaid_invoices ui
                  LEFT JOIN customer c ON ui.customer_id = c.id
                  LEFT JOIN province p ON ui.province_id = p.id
                  WHERE ui.is_loan_sale = 1 
                  AND ui.loan_outstanding_bif > 0 
                  AND ui.loan_due_date IS NOT NULL
                  AND ui.loan_due_date < CURDATE()
                  AND (ui.approval_status IS NULL OR ui.approval_status != 'rejected')
                  AND ui.created_at BETWEEN ? AND ?";
        if ($hasProvinceFilter) {
            $query .= " AND ui.province_id = ?";
        }
        $query .= " ORDER BY ui.created_at DESC, ui.loan_due_date ASC";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            if ($hasProvinceFilter) {
                $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
            } else {
                $stmt->bind_param("ss", $startDateTime, $endDateTime);
            }
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                fputcsv($out, [
                    $row['invoice_number'] ?? 'N/A',
                    $row['customer_name'] ?? 'N/A',
                    $row['loan_outstanding_bif'] ?? 0,
                    $row['loan_due_date'] ?? 'N/A',
                    $row['days_overdue'] ?? 0,
                    $row['province_name'] ?? 'N/A',
                    $row['created_at'] ?? 'N/A'
                ]);
            }
            $stmt->close();
        }
        fclose($out);
    } elseif ($target === 'customer_loans') {
        header('Content-Disposition: attachment; filename=customer_loan_status_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['Customer Name', 'Loan Limit (BIF)', 'Outstanding Balance (BIF)', 'Total Unpaid (BIF)', 'Late Invoices', 'Loan Status', 'Province']);
        
        $query = "SELECT 
                    c.full_name,
                    c.loan_limit_bif,
                    c.loan_balance_bif,
                    COALESCE(SUM(CASE WHEN ui.is_loan_sale = 1 AND ui.loan_outstanding_bif > 0 AND (ui.approval_status IS NULL OR ui.approval_status != 'rejected') THEN ui.loan_outstanding_bif ELSE 0 END), 0) as total_unpaid,
                    COUNT(CASE WHEN ui.is_loan_sale = 1 AND ui.loan_outstanding_bif > 0 AND (ui.approval_status IS NULL OR ui.approval_status != 'rejected') AND ui.loan_due_date < CURDATE() THEN 1 END) as late_invoice_count,
                    c.loan_status,
                    (SELECT p.name FROM unpaid_invoices ui2 
                     LEFT JOIN province p ON ui2.province_id = p.id 
                     WHERE (ui2.customer_id = c.id OR (ui2.customer_id IS NULL AND ui2.customer_name = c.full_name AND ui2.phone_number = c.tel))
                     AND ui2.is_loan_sale = 1 
                     LIMIT 1) AS province_name
                  FROM customer c
                  LEFT JOIN unpaid_invoices ui ON (ui.customer_id = c.id OR (ui.customer_id IS NULL AND ui.customer_name = c.full_name AND ui.phone_number = c.tel))
                  WHERE (c.can_pay_by_loan = 1 OR c.loan_balance_bif > 0)";
        if ($hasProvinceFilter) {
            $query .= " AND EXISTS (
                SELECT 1 FROM unpaid_invoices ui3 
                WHERE (ui3.customer_id = c.id OR (ui3.customer_id IS NULL AND ui3.customer_name = c.full_name AND ui3.phone_number = c.tel))
                AND ui3.province_id = ?
            )";
        }
        $query .= " GROUP BY c.id, c.full_name, c.loan_limit_bif, c.loan_balance_bif, c.loan_status
                  ORDER BY total_unpaid DESC, c.full_name";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            if ($hasProvinceFilter) {
                $stmt->bind_param("i", $provinceParam);
            }
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                fputcsv($out, [
                    $row['full_name'] ?? 'N/A',
                    $row['loan_limit_bif'] ?? 0,
                    $row['loan_balance_bif'] ?? 0,
                    $row['total_unpaid'] ?? 0,
                    $row['late_invoice_count'] ?? 0,
                    ucfirst($row['loan_status'] ?? 'inactive'),
                    $row['province_name'] ?? 'N/A'
                ]);
            }
            $stmt->close();
        }
        fclose($out);
    } elseif ($target === 'shift_closures') {
        header('Content-Disposition: attachment; filename=shift_closures_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['Shift Start', 'Shift End', 'Caissier_Comptable', 'Province', 'Beginning Balance (BIF)', 'Deposits (BIF)', 'Withdrawals (BIF)', 'Expected Balance (BIF)', 'Counted Cash (BIF)', 'Difference (BIF)']);
        
        $query = "SELECT 
                    cb.shift_start,
                    cb.shift_end,
                    u.full_name AS cashier_name,
                    p.name AS province_name,
                    cb.beginning_balance,
                    cb.deposits,
                    cb.withdrawals,
                    cb.end_balance,
                    csc.counted_amount,
                    csc.difference_amount
                  FROM cashier_balance cb
                  LEFT JOIN cashier_shift_closure csc ON csc.shift_id = cb.id
                  LEFT JOIN user u ON cb.cashier_id = u.id
                  LEFT JOIN province p ON u.province_id = p.id
                  WHERE cb.shift_end IS NOT NULL
                  AND cb.shift_end BETWEEN ? AND ?";
        if ($hasProvinceFilter) {
            $query .= " AND u.province_id = ?";
        }
        $query .= " ORDER BY cb.shift_end DESC";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            if ($hasProvinceFilter) {
                $stmt->bind_param("ssi", $startDateTime, $endDateTime, $provinceParam);
            } else {
                $stmt->bind_param("ss", $startDateTime, $endDateTime);
            }
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                fputcsv($out, [
                    $row['shift_start'] ?? 'N/A',
                    $row['shift_end'] ?? 'N/A',
                    $row['cashier_name'] ?? 'N/A',
                    $row['province_name'] ?? 'N/A',
                    $row['beginning_balance'] ?? 0,
                    $row['deposits'] ?? 0,
                    $row['withdrawals'] ?? 0,
                    $row['end_balance'] ?? 0,
                    $row['counted_amount'] ?? 'N/A',
                    $row['difference_amount'] ?? 'N/A'
                ]);
            }
            $stmt->close();
        }
        fclose($out);
    } elseif ($target === 'stock_additions') {
        header('Content-Disposition: attachment; filename=stock_additions_' . $startDate . '_to_' . $endDate . '.csv');
        $out = fopen('php://output', 'w');
        fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));
        fputcsv($out, [$timeframe]);
        fputcsv($out, ['Product Name', 'Quantity (Crates)', 'Quantity (Bottles)', 'Unit Type', 'Added By', 'Date Added']);
        
        $query = "SELECT 
                    p.name AS product_name,
                    st.quantity,
                    st.unit_type,
                    st.sent_at,
                    pr.crate_quantity,
                    u.full_name AS added_by_name
                  FROM stock_transfer st
                  LEFT JOIN product p ON st.product_id = p.id
                  LEFT JOIN product pr ON st.product_id = pr.id
                  LEFT JOIN user u ON st.operation_manager_id = u.id
                  WHERE st.status = 'confirmed'
                  AND st.sent_at BETWEEN ? AND ?
                  ORDER BY st.sent_at DESC";
        $stmt = $conn->prepare($query);
        if ($stmt) {
            $stmt->bind_param("ss", $startDateTime, $endDateTime);
            $stmt->execute();
            $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            foreach ($rows as $row) {
                $bottles = $row['unit_type'] === 'crates' ? $row['quantity'] * ($row['crate_quantity'] ?? 1) : $row['quantity'];
                fputcsv($out, [
                    $row['product_name'] ?? 'N/A',
                    $row['quantity'] ?? 0,
                    $bottles,
                    $row['unit_type'] ?? 'N/A',
                    $row['added_by_name'] ?? 'Unknown',
                    $row['sent_at'] ?? 'N/A'
                ]);
            }
            $stmt->close();
        }
        fclose($out);
    }
}

include $baseDir . '/includes/header.php';

// Get active section from URL or default to first
$activeSection = filter_input(INPUT_GET, 'section', FILTER_SANITIZE_FULL_SPECIAL_CHARS) ?: 'sales_province';
?>

<style>
    .reports-container { padding: 15px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh; }
    .reports-card { background-color: #FDFBF7; padding: 20px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); }
    .reports-header { margin-bottom: 20px; }
    .reports-title { color: #4B2F1F; font-size: 24px; margin: 0 0 5px 0; }
    .reports-subtitle { color: #666; font-size: 13px; margin: 0; }
    .dashboard-container { display: flex; gap: 20px; }
    .nav-sidebar { position: sticky; top: 100px; width: 220px; flex-shrink: 0; height: fit-content; background: #FFFFFF; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); overflow-y: auto; max-height: calc(100vh - 120px); }
    .nav-sidebar a { display: block; padding: 12px 16px; color: #4B2F1F; text-decoration: none; font-size: 13px; font-weight: 500; border-left: 3px solid transparent; transition: all 0.2s; }
    .nav-sidebar a:hover { background-color: #F4F0E4; border-left-color: #F4A261; }
    .nav-sidebar a.active { background-color: #F4F0E4; border-left-color: #4B2F1F; color: #4B2F1F; font-weight: 600; }
    .main-content { flex: 1; padding-bottom: 40px; }
    .filter-section { background-color: #FFFFFF; padding: 20px; border-radius: 8px; margin-bottom: 20px; border: 1px solid rgba(75,47,31,0.1); }
    .filter-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; align-items: end; }
    .filter-group { display: flex; flex-direction: column; }
    .filter-label { color: #4B2F1F; font-weight: 600; margin-bottom: 6px; font-size: 12px; }
    .filter-input { padding: 10px; border: 1px solid #4B2F1F; border-radius: 6px; background-color: #F4F0E4; font-size: 14px; }
    .filter-btn { background-color: #4B2F1F; color: #FDFBF7; padding: 10px 20px; border: none; border-radius: 6px; cursor: pointer; font-weight: 600; font-size: 14px; transition: background-color 0.3s; }
    .filter-btn:hover { background-color: #3a2418; }
    .filter-btn.reset { background-color: #F4A261; }
    .filter-btn.reset:hover { background-color: #e8914f; }
    .export-btn { background-color: #4CAF50; color: #FFFFFF; padding: 10px 20px; border-radius: 6px; text-decoration: none; font-weight: 600; font-size: 14px; border: none; cursor: pointer; display: inline-block; transition: background-color 0.3s; }
    .export-btn:hover { background-color: #45a049; }
    .section-box { background-color: #FFFFFF; border-radius: 8px; margin-bottom: 20px; border: 1px solid rgba(75,47,31,0.1); overflow: hidden; display: none; }
    .section-box.active { display: block; }
    .section-content { padding: 20px; }
    table { width: 100%; border-collapse: collapse; }
    thead { background-color: #4B2F1F; color: #FDFBF7; }
    th { padding: 12px; text-align: left; font-weight: 600; font-size: 13px; }
    th:first-child { width: 50px; text-align: center; }
    td { padding: 10px 12px; border-bottom: 1px solid #E0E0E0; font-size: 13px; }
    td:first-child { text-align: center; font-weight: 600; color: #4B2F1F; }
    tbody tr:hover { background-color: #F9F9F9; }
    .empty-message { color: #999; text-align: center; padding: 30px; }
    .status-badge { display: inline-block; padding: 4px 12px; border-radius: 4px; font-size: 11px; font-weight: 600; }
    .status-pending { background-color: #FFF3CD; color: #856404; }
    .status-approved { background-color: #D4EDDA; color: #155724; }
    .status-paid { background-color: #CCE5FF; color: #004085; }
    .filter-actions { display: flex; gap: 10px; align-items: center; }
</style>

<div class="reports-container">
    <div class="reports-card">
        <div class="reports-header">
            <h1 class="reports-title">📊 Comprehensive Reports Dashboard</h1>
            <p class="reports-subtitle">Generate detailed reports filtered by date range and province. Export data to CSV format.</p>
        </div>

        <div class="dashboard-container">
            <!-- NAVIGATION SIDEBAR -->
            <div class="nav-sidebar">
                <a href="?section=sales_province&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'sales_province' ? 'active' : ''; ?>">Sales by Province</a>
                <a href="?section=cashier_balances&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'cashier_balances' ? 'active' : ''; ?>">Caissier_Comptable Balances</a>
                <a href="?section=accountant_balances&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'accountant_balances' ? 'active' : ''; ?>">Accountant Balances</a>
                <a href="?section=pending_invoices&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'pending_invoices' ? 'active' : ''; ?>">Pending Invoices</a>
                <a href="?section=requisitions&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'requisitions' ? 'active' : ''; ?>">Requisitions</a>
                <a href="?section=paid_invoices&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'paid_invoices' ? 'active' : ''; ?>">Paid Invoices</a>
                <a href="?section=top_products&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'top_products' ? 'active' : ''; ?>">Top Products</a>
                <a href="?section=expenses&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'expenses' ? 'active' : ''; ?>">Expenses</a>
                <a href="?section=loan_payments&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'loan_payments' ? 'active' : ''; ?>">Loan Payments</a>
                <a href="?section=unpaid_loans&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'unpaid_loans' ? 'active' : ''; ?>">Unpaid Loans</a>
                <a href="?section=late_payments&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'late_payments' ? 'active' : ''; ?>">Late Payments</a>
                <a href="?section=customer_loans&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'customer_loans' ? 'active' : ''; ?>">Customer Loan Status</a>
                <a href="?section=shift_closures&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'shift_closures' ? 'active' : ''; ?>">Shift Closures</a>
                <a href="?section=stock_additions&start_date=<?php echo urlencode($startDate); ?>&end_date=<?php echo urlencode($endDate); ?>&province_id=<?php echo $provinceFilter; ?>" class="<?php echo $activeSection === 'stock_additions' ? 'active' : ''; ?>">Stock Additions</a>
            </div>

            <!-- MAIN CONTENT -->
            <div class="main-content">
                <!-- Filter Section -->
                <form method="GET" class="filter-section">
                    <input type="hidden" name="section" value="<?php echo htmlspecialchars($activeSection); ?>">
                    <div class="filter-grid">
                        <div class="filter-group">
                            <label class="filter-label">From Date</label>
                            <input type="date" name="start_date" class="filter-input" value="<?php echo htmlspecialchars($startDate); ?>">
                        </div>
                        <div class="filter-group">
                            <label class="filter-label">To Date</label>
                            <input type="date" name="end_date" class="filter-input" value="<?php echo htmlspecialchars($endDate); ?>">
                        </div>
                        <div class="filter-group">
                            <label class="filter-label">Province</label>
                            <select name="province_id" class="filter-input">
                                <option value="0">All Provinces</option>
                                <?php foreach ($provinces as $prov): ?>
                                    <option value="<?php echo (int)$prov['id']; ?>" <?php echo ((int)$prov['id'] === (int)$provinceFilter) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($prov['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="filter-group filter-actions">
                            <button type="submit" class="filter-btn">Apply Filters</button>
                            <a href="?section=<?php echo htmlspecialchars($activeSection); ?>" class="filter-btn reset" style="text-decoration: none; display: inline-block;">Reset</a>
                            <button type="button" onclick="exportData('<?php echo htmlspecialchars($activeSection); ?>')" class="export-btn">📥 Export CSV</button>
                        </div>
                    </div>
                </form>

                <!-- Sales by Province Section -->
                <div class="section-box <?php echo $activeSection === 'sales_province' ? 'active' : ''; ?>" id="section-sales_province">
                    <div class="section-content">
                        <?php if (empty($salesByProvince)): ?>
                            <p class="empty-message">No sales data available for the selected period.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Province</th>
                                            <th>Sales Count</th>
                                            <th>Total Sales (BIF)</th>
                                            <th>Requisitions (BIF)</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($salesByProvince as $row): ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td><strong><?php echo htmlspecialchars($row['province_name']); ?></strong></td>
                                                <td><?php echo $row['sales_count'] ?? 0; ?></td>
                                                <td><?php echo number_format($row['total_sales'] ?? 0, 0, '.', ','); ?></td>
                                                <td><?php echo number_format($row['total_requisitions'] ?? 0, 0, '.', ','); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Caissier_Comptable Balances Section -->
                <div class="section-box <?php echo $activeSection === 'cashier_balances' ? 'active' : ''; ?>" id="section-cashier_balances">
                    <div class="section-content">
                        <?php if (empty($cashierBalances)): ?>
                            <p class="empty-message">No cashier data available.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Province</th>
                                            <th>Caissier_Comptable Name</th>
                                            <th>Last Expected Balance (BIF)</th>
                                            <th>Last Counted Cash (BIF)</th>
                                            <th>Last Difference (BIF)</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($cashierBalances as $row): ?>
                                            <?php 
                                            $difference = isset($row['last_difference']) ? (int)$row['last_difference'] : null;
                                            $diff_color = '';
                                            if ($difference !== null) {
                                                $diff_color = $difference < 0 ? '#dc3545' : ($difference > 0 ? '#28a745' : '#6c757d');
                                            }
                                            ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo htmlspecialchars($row['cashier_name']); ?></td>
                                                <td><strong style="color: #2196F3;"><?php echo number_format($row['last_expected_balance'] ?? 0, 0, '.', ','); ?></strong></td>
                                                <td><?php echo isset($row['last_counted_cash']) ? number_format($row['last_counted_cash'], 0, '.', ',') : 'N/A'; ?></td>
                                                <td style="color: <?php echo $diff_color; ?>; font-weight: 600;">
                                                    <?php echo $difference !== null ? number_format($difference, 0, '.', ',') : 'N/A'; ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Accountant Balances Section -->
                <div class="section-box <?php echo $activeSection === 'accountant_balances' ? 'active' : ''; ?>" id="section-accountant_balances">
                    <div class="section-content">
                        <?php if (empty($accountantBalances)): ?>
                            <p class="empty-message">No accountant data available.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Accountant</th>
                                            <th>Province</th>
                                            <th>Total Received (BIF)</th>
                                            <th>Total Expenses (BIF)</th>
                                            <th>Available Balance (BIF)</th>
                                            <th>Last Updated</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($accountantBalances as $row): ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td><?php echo htmlspecialchars($row['accountant_name']); ?></td>
                                                <td><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo number_format($row['balance'] ?? 0, 0, '.', ','); ?></td>
                                                <td style="color: #D9534F;"><?php echo number_format($row['total_expenses'] ?? 0, 0, '.', ','); ?></td>
                                                <td style="color: <?php echo $row['available_balance'] > 0 ? '#28a745' : '#4B2F1F'; ?>; font-weight: 600;">
                                                    <?php echo number_format(max(0, (float)$row['available_balance']), 0, '.', ','); ?>
                                                </td>
                                                <td><?php echo $row['last_updated'] ? date('Y-m-d H:i', strtotime($row['last_updated'])) : 'Never'; ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Pending Invoices Section -->
                <div class="section-box <?php echo $activeSection === 'pending_invoices' ? 'active' : ''; ?>" id="section-pending_invoices">
                    <div class="section-content">
                        <?php if (empty($pendingInvoices)): ?>
                            <p class="empty-message">No pending invoices for the selected period.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Invoice Number</th>
                                            <th>Province</th>
                                            <th>Initiator</th>
                                            <th>Amount (BIF)</th>
                                            <th>Status</th>
                                            <th>Created Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($pendingInvoices as $row): ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td><strong><?php echo htmlspecialchars($row['invoice_number'] ?? '#' . $row['id']); ?></strong></td>
                                                <td><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo htmlspecialchars($row['initiator_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo number_format($row['total_amount'] ?? 0, 0, '.', ','); ?></td>
                                                <td><span class="status-badge status-pending"><?php echo htmlspecialchars($row['approval_status'] ?? 'pending'); ?></span></td>
                                                <td><?php echo date('Y-m-d H:i', strtotime($row['created_at'])); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Requisitions Section -->
                <div class="section-box <?php echo $activeSection === 'requisitions' ? 'active' : ''; ?>" id="section-requisitions">
                    <div class="section-content">
                        <?php if (empty($requisitions)): ?>
                            <p class="empty-message">No requisitions for the selected period.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>ID</th>
                                            <th>Province</th>
                                            <th>Initiator</th>
                                            <th>Amount (BIF)</th>
                                            <th>Status</th>
                                            <th>Created Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($requisitions as $row): ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td><strong>#<?php echo $row['id']; ?></strong></td>
                                                <td><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo htmlspecialchars($row['initiator_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo number_format($row['total_amount'] ?? 0, 0, '.', ','); ?></td>
                                                <td><span class="status-badge status-<?php echo strtolower($row['status']); ?>"><?php echo htmlspecialchars($row['status']); ?></span></td>
                                                <td><?php echo date('Y-m-d H:i', strtotime($row['created_at'])); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Paid Invoices Section -->
                <div class="section-box <?php echo $activeSection === 'paid_invoices' ? 'active' : ''; ?>" id="section-paid_invoices">
                    <div class="section-content">
                        <?php if (empty($paidInvoices)): ?>
                            <p class="empty-message">No paid invoices for the selected period.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>ID</th>
                                            <th>Province</th>
                                            <th>Caissier_Comptable</th>
                                            <th>Amount (BIF)</th>
                                            <th>Created Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($paidInvoices as $row): ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td><strong>#<?php echo $row['id']; ?></strong></td>
                                                <td><?php echo htmlspecialchars($row['province_name']); ?></td>
                                                <td><?php echo htmlspecialchars($row['cashier_name']); ?></td>
                                                <td style="color: #4CAF50; font-weight: 600;"><?php echo number_format($row['paid_amount'] ?? 0, 0, '.', ','); ?></td>
                                                <td><?php echo date('Y-m-d H:i', strtotime($row['created_at'])); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Top Products Section -->
                <div class="section-box <?php echo $activeSection === 'top_products' ? 'active' : ''; ?>" id="section-top_products">
                    <div class="section-content">
                        <?php if (empty($topProducts)): ?>
                            <p class="empty-message">No product sales data for the selected period.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Product Name</th>
                                            <th>Total Quantity</th>
                                            <th>Total Amount (BIF)</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($topProducts as $row): ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td><?php echo htmlspecialchars($row['name']); ?></td>
                                                <td><?php echo number_format($row['total_quantity'] ?? 0, 0, '.', ','); ?></td>
                                                <td><?php echo number_format($row['total_amount'] ?? 0, 0, '.', ','); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Expenses Section -->
                <div class="section-box <?php echo $activeSection === 'expenses' ? 'active' : ''; ?>" id="section-expenses">
                    <div class="section-content">
                        <?php if (empty($expenses)): ?>
                            <p class="empty-message">No expenses for the selected period.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>ID</th>
                                            <th>Amount (BIF)</th>
                                            <th>Method</th>
                                            <th>Recipient</th>
                                            <th>Accountant</th>
                                            <th>Province</th>
                                            <th>Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($expenses as $row): ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td>#<?php echo $row['id']; ?></td>
                                                <td style="color: #D9534F; font-weight: 600;"><?php echo number_format($row['amount'] ?? 0, 0, '.', ','); ?></td>
                                                <td><?php echo htmlspecialchars(ucfirst($row['payment_method'] ?? '')); ?></td>
                                                <td><?php echo htmlspecialchars($row['recipient_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo htmlspecialchars($row['accountant_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo date('Y-m-d H:i', strtotime($row['created_at'])); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Loan Payments Section -->
                <div class="section-box <?php echo $activeSection === 'loan_payments' ? 'active' : ''; ?>" id="section-loan_payments">
                    <div class="section-content">
                        <?php if (empty($loanPayments)): ?>
                            <p class="empty-message">No loan payments for the selected period.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>ID</th>
                                            <th>Customer</th>
                                            <th>Amount (BIF)</th>
                                            <th>Method</th>
                                            <th>Province</th>
                                            <th>Payment Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($loanPayments as $row): ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td>#<?php echo $row['id']; ?></td>
                                                <td><?php echo htmlspecialchars($row['customer_name'] ?? 'N/A'); ?></td>
                                                <td style="color: #4CAF50; font-weight: 600;"><?php echo number_format($row['payment_amount'] ?? 0, 0, '.', ','); ?></td>
                                                <td><?php echo htmlspecialchars(ucfirst($row['payment_method'] ?? '')); ?></td>
                                                <td><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo date('Y-m-d H:i', strtotime($row['payment_date'])); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Unpaid Loans Section -->
                <div class="section-box <?php echo $activeSection === 'unpaid_loans' ? 'active' : ''; ?>" id="section-unpaid_loans">
                    <div class="section-content">
                        <?php if (empty($unpaidLoans)): ?>
                            <p class="empty-message">No unpaid loans for the selected period.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Invoice Number</th>
                                            <th>Customer</th>
                                            <th>Outstanding (BIF)</th>
                                            <th>Due Date</th>
                                            <th>Status</th>
                                            <th>Province</th>
                                            <th>Created Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($unpaidLoans as $row): ?>
                                            <?php 
                                            $is_late = $row['loan_due_date'] && strtotime($row['loan_due_date']) < time();
                                            $status_color = $is_late ? '#dc3545' : ($row['payment_status'] === 'Pending' ? '#ffc107' : '#6c757d');
                                            ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td><strong><?php echo htmlspecialchars($row['invoice_number'] ?? 'N/A'); ?></strong></td>
                                                <td><?php echo htmlspecialchars($row['customer_name'] ?? 'N/A'); ?></td>
                                                <td style="color: #dc3545; font-weight: 600;"><?php echo number_format($row['loan_outstanding_bif'] ?? 0, 0, '.', ','); ?></td>
                                                <td style="color: <?php echo $status_color; ?>; font-weight: 600;">
                                                    <?php echo $row['loan_due_date'] ? date('Y-m-d', strtotime($row['loan_due_date'])) : 'N/A'; ?>
                                                </td>
                                                <td>
                                                    <span class="status-badge" style="background-color: <?php echo $status_color === '#dc3545' ? '#f8d7da' : ($status_color === '#ffc107' ? '#fff3cd' : '#e2e3e5'); ?>; color: <?php echo $status_color; ?>;">
                                                        <?php echo htmlspecialchars($row['payment_status'] ?? 'N/A'); ?>
                                                    </span>
                                                </td>
                                                <td><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo date('Y-m-d H:i', strtotime($row['created_at'])); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Late Payments Section -->
                <div class="section-box <?php echo $activeSection === 'late_payments' ? 'active' : ''; ?>" id="section-late_payments">
                    <div class="section-content">
                        <?php if (empty($latePayments)): ?>
                            <p class="empty-message">No late payments found for the selected period.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Invoice Number</th>
                                            <th>Customer</th>
                                            <th>Outstanding (BIF)</th>
                                            <th>Due Date</th>
                                            <th>Days Overdue</th>
                                            <th>Province</th>
                                            <th>Created Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($latePayments as $row): ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td><strong><?php echo htmlspecialchars($row['invoice_number'] ?? 'N/A'); ?></strong></td>
                                                <td><?php echo htmlspecialchars($row['customer_name'] ?? 'N/A'); ?></td>
                                                <td style="color: #dc3545; font-weight: 600;"><?php echo number_format($row['loan_outstanding_bif'] ?? 0, 0, '.', ','); ?></td>
                                                <td style="color: #dc3545; font-weight: 600;"><?php echo date('Y-m-d', strtotime($row['loan_due_date'])); ?></td>
                                                <td>
                                                    <span class="status-badge" style="background-color: #f8d7da; color: #dc3545;">
                                                        <?php echo $row['days_overdue'] ?? 0; ?> days
                                                    </span>
                                                </td>
                                                <td><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo date('Y-m-d H:i', strtotime($row['created_at'])); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Customer Loan Status Section -->
                <div class="section-box <?php echo $activeSection === 'customer_loans' ? 'active' : ''; ?>" id="section-customer_loans">
                    <div class="section-content">
                        <?php if (empty($customerLoanStatus)): ?>
                            <p class="empty-message">No loan-eligible customers found.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Customer Name</th>
                                            <th>Loan Limit (BIF)</th>
                                            <th>Outstanding Balance (BIF)</th>
                                            <th>Total Unpaid (BIF)</th>
                                            <th>Late Invoices</th>
                                            <th>Loan Status</th>
                                            <th>Province</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($customerLoanStatus as $row): ?>
                                            <?php 
                                            $unpaid = (int)($row['total_unpaid'] ?? 0);
                                            $late_count = (int)($row['late_invoice_count'] ?? 0);
                                            ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td><strong><?php echo htmlspecialchars($row['full_name'] ?? 'N/A'); ?></strong></td>
                                                <td><?php echo number_format($row['loan_limit_bif'] ?? 0, 0, '.', ','); ?></td>
                                                <td style="color: #dc3545; font-weight: 600;"><?php echo number_format($row['loan_balance_bif'] ?? 0, 0, '.', ','); ?></td>
                                                <td>
                                                    <?php if ($unpaid > 0): ?>
                                                        <span style="color: #dc3545; font-weight: 600;"><?php echo number_format($unpaid, 0, '.', ','); ?></span>
                                                    <?php else: ?>
                                                        <span style="color: #28a745;">0</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if ($late_count > 0): ?>
                                                        <span class="status-badge" style="background-color: #f8d7da; color: #dc3545;"><?php echo $late_count; ?> Late</span>
                                                    <?php else: ?>
                                                        <span style="color: #28a745;">0</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <span class="status-badge" style="background-color: <?php echo strtolower($row['loan_status'] ?? 'inactive') === 'active' ? '#d4edda' : '#fff3cd'; ?>; color: <?php echo strtolower($row['loan_status'] ?? 'inactive') === 'active' ? '#155724' : '#856404'; ?>;">
                                                        <?php echo htmlspecialchars(ucfirst($row['loan_status'] ?? 'inactive')); ?>
                                                    </span>
                                                </td>
                                                <td><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Shift Closures Section -->
                <div class="section-box <?php echo $activeSection === 'shift_closures' ? 'active' : ''; ?>" id="section-shift_closures">
                    <div class="section-content">
                        <?php if (empty($shiftClosures)): ?>
                            <p class="empty-message">No shift closures for the selected period.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Shift Start</th>
                                            <th>Shift End</th>
                                            <th>Caissier_Comptable</th>
                                            <th>Province</th>
                                            <th>Beginning Balance</th>
                                            <th>Deposits</th>
                                            <th>Withdrawals</th>
                                            <th>Expected Balance</th>
                                            <th>Counted Cash</th>
                                            <th>Difference</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($shiftClosures as $row): ?>
                                            <?php 
                                            $difference = isset($row['difference_amount']) ? (int)$row['difference_amount'] : null;
                                            $diff_color = '';
                                            if ($difference !== null) {
                                                $diff_color = $difference < 0 ? '#dc3545' : ($difference > 0 ? '#28a745' : '#6c757d');
                                            }
                                            ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td><?php echo date('Y-m-d H:i', strtotime($row['shift_start'])); ?></td>
                                                <td><?php echo date('Y-m-d H:i', strtotime($row['shift_end'])); ?></td>
                                                <td><?php echo htmlspecialchars($row['cashier_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo htmlspecialchars($row['province_name'] ?? 'N/A'); ?></td>
                                                <td><?php echo number_format($row['beginning_balance'] ?? 0, 0, '.', ','); ?></td>
                                                <td style="color: #4CAF50;"><?php echo number_format($row['deposits'] ?? 0, 0, '.', ','); ?></td>
                                                <td style="color: #D9534F;"><?php echo number_format($row['withdrawals'] ?? 0, 0, '.', ','); ?></td>
                                                <td><strong><?php echo number_format($row['end_balance'] ?? 0, 0, '.', ','); ?></strong></td>
                                                <td><?php echo isset($row['counted_amount']) ? number_format($row['counted_amount'], 0, '.', ',') : 'N/A'; ?></td>
                                                <td style="color: <?php echo $diff_color; ?>; font-weight: 600;">
                                                    <?php echo $difference !== null ? number_format($difference, 0, '.', ',') : 'N/A'; ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Stock Additions Section -->
                <div class="section-box <?php echo $activeSection === 'stock_additions' ? 'active' : ''; ?>" id="section-stock_additions">
                    <div class="section-content">
                        <?php if (empty($stockAdditions)): ?>
                            <p class="empty-message">No stock additions for the selected period.</p>
                        <?php else: ?>
                            <div style="overflow-x: auto;">
                                <table>
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Product Name</th>
                                            <th>Quantity (Crates)</th>
                                            <th>Quantity (Bottles)</th>
                                            <th>Unit Type</th>
                                            <th>Added By</th>
                                            <th>Date Added</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $rowNum = 1; foreach ($stockAdditions as $row): ?>
                                            <?php 
                                            $bottles = $row['unit_type'] === 'crates' ? $row['quantity'] * ($row['crate_quantity'] ?? 1) : $row['quantity'];
                                            ?>
                                            <tr>
                                                <td><?php echo $rowNum++; ?></td>
                                                <td><strong><?php echo htmlspecialchars($row['product_name'] ?? 'N/A'); ?></strong></td>
                                                <td><?php echo number_format($row['quantity'] ?? 0, 0, '.', ','); ?></td>
                                                <td><?php echo number_format($bottles, 0, '.', ','); ?></td>
                                                <td><?php echo htmlspecialchars(ucfirst($row['unit_type'] ?? 'N/A')); ?></td>
                                                <td><?php echo htmlspecialchars($row['added_by_name'] ?? 'Unknown'); ?></td>
                                                <td><?php echo date('Y-m-d H:i', strtotime($row['sent_at'])); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <!-- END MAIN CONTENT -->
        </div>
        <!-- END DASHBOARD CONTAINER -->
    </div>
</div>

<script>
function exportData(type) {
    const startDate = document.querySelector('input[name="start_date"]').value;
    const endDate = document.querySelector('input[name="end_date"]').value;
    const provinceId = document.querySelector('select[name="province_id"]').value;
    
    let url = window.location.pathname + '?export=' + type + '&start_date=' + startDate + '&end_date=' + endDate;
    if (provinceId !== '0') {
        url += '&province_id=' + provinceId;
    }
    
    window.location.href = url;
}
</script>

<?php include $baseDir . '/includes/footer.php'; ?>
