<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/export_helpers.php';

if (!is_logged_in()) {
    header("Location: /masunzu_bar_hotel/modules/login/login.php");
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    die("<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>");
}

$user_id = $_SESSION['user_id'];
$role_id = $_SESSION['role_id'];
$role_name = $_SESSION['role_name'];
$province_id = $_SESSION['province_id'];

// Check authorization: Admins (1,2,3), Accountants (4/7/9), Stock Managers (5), Caissier_Comptables (6), Operation Managers (8), Board
$marketing_role_names = ['Chef Marketing', 'Agent Marketing', 'Marketing Agent', 'Marketing Head'];
$allowed_roles = [1, 2, 3, 4, 5, 6, 7, 8, 9]; // inclusive list for current schema
if (!in_array($role_id, $allowed_roles) && !in_array($role_name, $marketing_role_names, true) && $role_name !== 'Board') {
    header("Location: /masunzu_bar_hotel/dashboards/unauthorized.php");
    exit;
}

// Handle filters
$from_date = filter_input(INPUT_GET, 'from_date', FILTER_UNSAFE_RAW) ?? '';
$to_date = filter_input(INPUT_GET, 'to_date', FILTER_UNSAFE_RAW) ?? '';
$search = filter_input(INPUT_GET, 'search', FILTER_UNSAFE_RAW) ?? '';
$expense_status = filter_input(INPUT_GET, 'expense_status', FILTER_UNSAFE_RAW) ?? '';
$filter_province = filter_input(INPUT_GET, 'province', FILTER_VALIDATE_INT) ?? '';
$filter_region = filter_input(INPUT_GET, 'region', FILTER_VALIDATE_INT) ?? '';
$filter_category = filter_input(INPUT_GET, 'category', FILTER_UNSAFE_RAW) ?? '';
$export = filter_input(INPUT_GET, 'export', FILTER_UNSAFE_RAW) ?? '';

// Fetch categories from database ENUM
$categoryOptions = [];
$stmt = $conn->prepare("SHOW COLUMNS FROM requisition_item WHERE Field = 'category'");
if ($stmt) {
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $stmt->close();
    
    if ($row && isset($row['Type'])) {
        // Extract ENUM values from Type field (e.g., "enum('communication','transport',...)")
        preg_match("/^enum\((.*)\)$/", $row['Type'], $matches);
        if (isset($matches[1])) {
            $enumValues = str_getcsv($matches[1], ',', "'");
            // Map values to labels (convert snake_case to Title Case)
            foreach ($enumValues as $value) {
                $value = trim($value, "'\"");
                $label = ucwords(str_replace('_', ' ', $value));
                // Special cases for better labels
                $labelMap = [
                    'frais_de_mission' => 'Frais de mission',
                    'office_utilities' => 'Bureautique / Utilités',
                ];
                $categoryOptions[$value] = $labelMap[$value] ?? $label;
            }
        }
    }
}

// If no categories found, use defaults
if (empty($categoryOptions)) {
    $categoryOptions = [
        'communication' => 'Communication',
        'transport' => 'Transport',
        'frais_de_mission' => 'Frais de mission',
        'office_utilities' => 'Bureautique / Utilités',
        'maintenance' => 'Maintenance',
        'others' => 'Autres',
    ];
}

// Handle category creation (only for admins and accountants)
$can_create_category = in_array($role_id, [1, 2, 3, 4, 7, 9]); // DG, DGA, Admin, Board, Accountant
$category_created = false;
$category_error = '';

if ($can_create_category && isset($_POST['create_category']) && $_POST['create_category'] === '1') {
    $new_category_label = trim($_POST['category_label'] ?? '');
    
    if (empty($new_category_label)) {
        $category_error = 'Le nom de la catégorie est requis.';
    } else {
        // Auto-generate code from label: lowercase, replace spaces/special chars with underscores, remove accents
        $new_category_value = strtolower($new_category_label);
        // Remove accents
        $new_category_value = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $new_category_value);
        // Replace spaces and special characters with underscores
        $new_category_value = preg_replace('/[^a-z0-9_]+/', '_', $new_category_value);
        // Remove multiple consecutive underscores
        $new_category_value = preg_replace('/_+/', '_', $new_category_value);
        // Remove leading/trailing underscores
        $new_category_value = trim($new_category_value, '_');
        
        if (empty($new_category_value)) {
            $category_error = 'Impossible de générer un code valide à partir du nom.';
        } elseif (isset($categoryOptions[$new_category_value])) {
            $category_error = 'Une catégorie avec ce nom existe déjà.';
        } else {
            // Add new category to ENUM
            $allCategories = array_keys($categoryOptions);
            $allCategories[] = $new_category_value;
            $enumValues = "'" . implode("','", $allCategories) . "'";
            
            $alterQuery = "ALTER TABLE requisition_item MODIFY COLUMN category ENUM($enumValues) NOT NULL";
            if ($conn->query($alterQuery)) {
                $category_created = true;
                $categoryOptions[$new_category_value] = $new_category_label;
                // Reload page to show new category
                header("Location: " . $_SERVER['PHP_SELF'] . "?" . http_build_query(array_merge($_GET, ['category_created' => '1'])));
                exit;
            } else {
                $category_error = 'Erreur lors de l\'ajout de la catégorie: ' . $conn->error;
            }
        }
    }
}

// Build query based on role
$query = "
    SELECT 
        r.id AS requisition_id, 
        r.status AS requisition_status, 
        r.created_at AS requisition_date,
        r.total_amount AS approved_amount,
        COALESCE(r.original_total_amount, r.total_amount) AS initiated_amount,
        r.original_total_amount,
        r.recipient_type,
        r.recipient_external_name,
        r.recipient_external_phone,
        u.full_name AS initiator_name,
        u2.full_name AS recipient_name,
        approver.full_name AS approver_name,
        p.name AS province_name,
        p.id AS province_id,
        reg.name AS region_name,
        e.id AS expense_id,
        e.payment_status,
        e.created_at AS expense_date,
        cashier.full_name AS cashier_name,
        assigned_cashier.full_name AS assigned_cashier_name,
        GROUP_CONCAT(DISTINCT ri.category ORDER BY ri.category) AS categories,
        SUM(ri.quantity * ri.unit_price) AS item_total_calculated
    FROM requisition r
    LEFT JOIN requisition_item ri ON r.id = ri.requisition_id
    JOIN user u ON r.user_id = u.id
    LEFT JOIN user u2 ON r.recipient_id = u2.id
    LEFT JOIN user approver ON r.approved_by = approver.id
    JOIN province p ON r.province_id = p.id
    LEFT JOIN region reg ON p.region_id = reg.id
    LEFT JOIN expense e ON r.id = e.requisition_id
    LEFT JOIN user cashier ON e.cashier_id = cashier.id
    LEFT JOIN user assigned_cashier ON r.assigned_cashier_id = assigned_cashier.id
    WHERE 1=1
";

$params = [];
$types = "";

// Role-based filtering
if (in_array($role_id, [1, 2, 3])) {
    // Admin roles (Board, DG, DGA) - see all provinces
    // No province filter needed
} else if (in_array($role_id, [4, 7, 9])) {
    // Accountant - see all provinces/regions but only approved/paid requisitions
    $query .= " AND r.status IN ('approved', 'paid')";
    // No province filter - accountants can see all depots
} else if (in_array($role_id, [5, 8]) || in_array($role_name, $marketing_role_names, true)) {
    // Stock Manager, Operation Manager - see only their province
    $query .= " AND r.province_id = ?";
    $params[] = $province_id;
    $types .= "i";
}

// Additional filters
if ($from_date) {
    $query .= " AND r.created_at >= ?";
    $params[] = $from_date . ' 00:00:00';
    $types .= "s";
}
if ($to_date) {
    $query .= " AND r.created_at <= ?";
    $params[] = $to_date . ' 23:59:59';
    $types .= "s";
}
if ($search) {
    $query .= " AND (u.full_name LIKE ? OR u2.full_name LIKE ? OR ri.description LIKE ?)";
    $search_term = "%$search%";
    $params[] = $search_term;
    $params[] = $search_term;
    $params[] = $search_term;
    $types .= "sss";
}
if ($expense_status && in_array($role_id, [1, 2, 3])) {
    $query .= " AND r.status = ?";
    $params[] = $expense_status;
    $types .= "s";
}
if ($filter_province && in_array($role_id, [1, 2, 3, 4, 7, 9])) {
    $query .= " AND r.province_id = ?";
    $params[] = $filter_province;
    $types .= "i";
}
if ($filter_region && in_array($role_id, [1, 2, 3, 4, 7, 9])) {
    $query .= " AND reg.id = ?";
    $params[] = $filter_region;
    $types .= "i";
}
if ($filter_category !== '' && $filter_category !== null) {
    $query .= " AND ri.category = ?";
    $params[] = $filter_category;
    $types .= "s";
}

$query .= " GROUP BY r.id, e.id ORDER BY r.created_at DESC";

$stmt = $conn->prepare($query);
if ($types) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$result = $stmt->get_result();
$expenses = $result->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch provinces and regions for filters (admins and accountants)
$provinces = [];
$regions = [];
if (in_array($role_id, [1, 2, 3, 4, 7, 9])) {
    $stmt = $conn->prepare("SELECT id, name FROM province ORDER BY name");
    $stmt->execute();
    $provinces = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    $stmt = $conn->prepare("SELECT id, name FROM region ORDER BY name");
    $stmt->execute();
    $regions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

// Build filter summary for exports
$activeFilters = [];
if ($from_date) {
    $activeFilters[] = 'Date début: ' . date('d/m/Y', strtotime($from_date));
}
if ($to_date) {
    $activeFilters[] = 'Date fin: ' . date('d/m/Y', strtotime($to_date));
}
if ($filter_category && isset($categoryOptions[$filter_category])) {
    $activeFilters[] = 'Catégorie: ' . $categoryOptions[$filter_category];
}
if ($filter_province && !empty($provinces)) {
    foreach ($provinces as $p) {
        if ($p['id'] == $filter_province) {
            $activeFilters[] = 'Dépôt: ' . $p['name'];
            break;
        }
    }
}
if ($filter_region && !empty($regions)) {
    foreach ($regions as $r) {
        if ($r['id'] == $filter_region) {
            $activeFilters[] = 'Région: ' . $r['name'];
            break;
        }
    }
}
if ($expense_status && in_array($role_id, [1, 2, 3])) {
    $statusLabels = [
        'pending' => 'En Attente',
        'approved' => 'Approuvé',
        'paid' => 'Payé',
        'rejected' => 'Rejeté'
    ];
    $activeFilters[] = 'Statut: ' . ($statusLabels[$expense_status] ?? $expense_status);
}
if ($search) {
    $activeFilters[] = 'Recherche: ' . $search;
}
$filterSummary = !empty($activeFilters) ? implode(' | ', $activeFilters) : 'Aucun filtre appliqué';

// Handle CSV/PDF export
if ($export === 'csv') {
    // Add filter information as first rows
    $csvContent = [];
    
    // Header row 1: Title
    $csvContent[] = ['Historique Dépenses / Réquisitions'];
    $csvContent[] = []; // Empty row
    
    // Header row 2: Filter information
    $csvContent[] = ['Filtres appliqués:', $filterSummary];
    $csvContent[] = ['Généré le:', date('d/m/Y à H:i')];
    $csvContent[] = ['Généré par:', $_SESSION['full_name'] ?? 'System'];
    $csvContent[] = []; // Empty row
    
    // Header row 3: Column headers
    $csvContent[] = [
        'ID Réquisition',
        'Catégories',
        'Initiateur',
        'Approbateur',
        'Caissier Désigné',
        'Dépôt',
        'Région',
        'Montant Demandé',
        'Montant Approuvé',
        'Date Demande',
        'Date Paiement',
        'Bénéficiaire',
        'Type Bénéficiaire',
        'Téléphone Bénéficiaire',
        'Statut Réquisition',
        'Statut Paiement'
    ];

    $rows = [];
    foreach ($expenses as $row) {
        $initiated_amount = (int)($row['initiated_amount'] ?? 0);
        $approved_amount = (int)($row['approved_amount'] ?? 0);

        if (isset($row['recipient_type']) && $row['recipient_type'] === 'external') {
            $receiver_name = $row['recipient_external_name'] ?: 'Externe';
            $receiver_phone = $row['recipient_external_phone'] ?: '';
            $receiver_type = 'Externe';
        } else {
            $receiver_name = $row['recipient_name'] ?: 'Interne';
            $receiver_phone = '';
            $receiver_type = 'Interne';
        }

        $rows[] = [
            $row['requisition_id'] ?? '',
            $row['categories'] ?? '',
            $row['initiator_name'] ?? '',
            $row['approver_name'] ?? '',
            $row['assigned_cashier_name'] ?? 'N/A',
            $row['province_name'] ?? '',
            $row['region_name'] ?? '',
            $initiated_amount,
            $approved_amount,
            $row['requisition_date'] ?? '',
            $row['expense_date'] ?? '',
            $receiver_name,
            $receiver_type,
            $receiver_phone,
            $row['requisition_status'] ?? '',
            $row['payment_status'] ?? '',
        ];
    }
    
    // Combine header info and data rows
    $allRows = array_merge($csvContent, $rows);
    
    $filename = 'expenses_history_' . date('Y-m-d_His') . '.csv';
    
    // Custom CSV export with filter info
    send_csv_headers($filename);
    $output = fopen('php://output', 'w');
    if ($output) {
        foreach ($allRows as $row) {
            fputcsv($output, $row);
        }
        fclose($output);
    }
    $conn->close();
    exit;
}

if ($export === 'pdf') {
    $tcpdfPath = __DIR__ . '/../../libs/tcpdf/tcpdf.php';
    if (file_exists($tcpdfPath)) {
        require_once $tcpdfPath;
        // Create PDF in Landscape orientation (L = Landscape, A4 format)
        $pdf = new TCPDF('L', PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
        $pdf->SetCreator('Masunzu Bar Hotel');
        $pdf->SetAuthor($_SESSION['full_name'] ?? 'System');
        $pdf->SetTitle('Historique Dépenses / Réquisitions');
        // Set smaller margins to maximize space for columns
        $pdf->SetMargins(8, 18, 8);
        $pdf->SetHeaderMargin(5);
        $pdf->SetFooterMargin(5);
        $pdf->SetAutoPageBreak(true, 12);
        // Remove default header/footer
        $pdf->setPrintHeader(false);
        $pdf->setPrintFooter(false);
        // Add page in landscape
        $pdf->AddPage('L'); // Explicitly set landscape

        // Build title with time frame
        $title = 'Historique Dépenses / Réquisitions';
        $timeFrame = '';
        if ($from_date && $to_date) {
            $timeFrame = 'Du ' . date('d/m/Y', strtotime($from_date)) . ' au ' . date('d/m/Y', strtotime($to_date));
        } elseif ($from_date) {
            $timeFrame = 'À partir du ' . date('d/m/Y', strtotime($from_date));
        } elseif ($to_date) {
            $timeFrame = 'Jusqu\'au ' . date('d/m/Y', strtotime($to_date));
        } else {
            $timeFrame = 'Toutes les périodes';
        }
        
        // Build filter summary for PDF
        $activeFilters = [];
        if ($from_date) {
            $activeFilters[] = 'Date début: ' . date('d/m/Y', strtotime($from_date));
        }
        if ($to_date) {
            $activeFilters[] = 'Date fin: ' . date('d/m/Y', strtotime($to_date));
        }
        if ($filter_category && isset($categoryOptions[$filter_category])) {
            $activeFilters[] = 'Catégorie: ' . $categoryOptions[$filter_category];
        }
        if ($filter_province && !empty($provinces)) {
            foreach ($provinces as $p) {
                if ($p['id'] == $filter_province) {
                    $activeFilters[] = 'Dépôt: ' . $p['name'];
                    break;
                }
            }
        }
        if ($filter_region && !empty($regions)) {
            foreach ($regions as $r) {
                if ($r['id'] == $filter_region) {
                    $activeFilters[] = 'Région: ' . $r['name'];
                    break;
                }
            }
        }
        if ($expense_status && in_array($role_id, [1, 2, 3])) {
            $statusLabels = [
                'pending' => 'En Attente',
                'approved' => 'Approuvé',
                'paid' => 'Payé',
                'rejected' => 'Rejeté'
            ];
            $activeFilters[] = 'Statut: ' . ($statusLabels[$expense_status] ?? $expense_status);
        }
        if ($search) {
            $activeFilters[] = 'Recherche: ' . htmlspecialchars($search, ENT_QUOTES, 'UTF-8');
        }
        $filterSummary = !empty($activeFilters) ? implode(' | ', $activeFilters) : 'Aucun filtre appliqué';
        
        $html = '<style>
            .header { background-color: #4B2F1F; color: #F4F0E4; padding: 8px; margin-bottom: 8px; }
            .header h1 { margin: 0; font-size: 16px; font-weight: bold; }
            .header p { margin: 4px 0 0; font-size: 10px; }
            .info-box { background-color: #F9F9F9; padding: 6px; margin-bottom: 8px; border: 1px solid #DDD; font-size: 9px; }
            .info-box strong { color: #4B2F1F; }
            table { width: 100%; border-collapse: collapse; font-size: 7px; table-layout: fixed; }
            th { background-color: #4B2F1F; color: #F4F0E4; padding: 5px 3px; text-align: left; font-weight: bold; font-size: 7px; word-wrap: break-word; }
            td { padding: 4px 3px; border: 1px solid #DDD; font-size: 7px; word-wrap: break-word; overflow: hidden; }
            tr:nth-child(even) { background-color: #F9F9F9; }
            .amount { text-align: right; }
            .status { padding: 2px 4px; border-radius: 2px; font-size: 6px; }
        </style>';
        
        $html .= '<div class="header">';
        $html .= '<h1>' . htmlspecialchars($title, ENT_QUOTES, 'UTF-8') . '</h1>';
        $html .= '<p>Période: ' . htmlspecialchars($timeFrame, ENT_QUOTES, 'UTF-8') . ' | Généré le ' . date('d/m/Y à H:i') . ' par ' . htmlspecialchars($_SESSION['full_name'] ?? 'System', ENT_QUOTES, 'UTF-8') . '</p>';
        $html .= '</div>';
        
        // Summary info with all filters
        $total_count = count($expenses);
        $total_approved = array_sum(array_map(function($e) { return (int)($e['approved_amount'] ?? 0); }, $expenses));
        $html .= '<div class="info-box">';
        $html .= '<strong>Total Réquisitions:</strong> ' . $total_count . ' | ';
        $html .= '<strong>Montant Total Approuvé:</strong> ' . number_format($total_approved, 0, ',', ' ') . ' BIF';
        $html .= '</div>';
        
        // Filter summary box
        if (!empty($activeFilters)) {
            $html .= '<div class="info-box" style="background-color: #E8F4F8; border-color: #4B2F1F;">';
            $html .= '<strong>Filtres appliqués:</strong><br>';
            $html .= htmlspecialchars($filterSummary, ENT_QUOTES, 'UTF-8');
            $html .= '</div>';
        }
        
        $html .= '<table>';
        $html .= '<thead><tr>';
        $html .= '<th style="width: 4%;">ID</th>';
        $html .= '<th style="width: 7.5%;">Catégories</th>';
        $html .= '<th style="width: 7.5%;">Initiateur</th>';
        $html .= '<th style="width: 7.5%;">Approbateur</th>';
        $html .= '<th style="width: 7%;">Caissier Désigné</th>';
        $html .= '<th style="width: 6%;">Dépôt</th>';
        $html .= '<th style="width: 6%;">Région</th>';
        $html .= '<th style="width: 7%;" class="amount">Montant Demandé</th>';
        $html .= '<th style="width: 7%;" class="amount">Montant Approuvé</th>';
        $html .= '<th style="width: 8%;">Bénéficiaire</th>';
        $html .= '<th style="width: 5%;">Type</th>';
        $html .= '<th style="width: 6%;">Statut</th>';
        $html .= '<th style="width: 6%;">Statut Paiement</th>';
        $html .= '<th style="width: 6.5%;">Date Demande</th>';
        $html .= '<th style="width: 6.5%;">Date Paiement</th>';
        $html .= '</tr></thead><tbody>';

        foreach ($expenses as $row) {
            if (isset($row['recipient_type']) && $row['recipient_type'] === 'external') {
                $receiver_name = $row['recipient_external_name'] ?: 'Externe';
                $receiver_type = 'Externe';
                $receiver_phone = $row['recipient_external_phone'] ?? '';
                if ($receiver_phone) {
                    $receiver_name .= ' (' . $receiver_phone . ')';
                }
            } else {
                $receiver_name = $row['recipient_name'] ?: 'Interne';
                $receiver_type = 'Interne';
            }
            
            $initiated_amount = (int)($row['initiated_amount'] ?? 0);
            $approved_amount = (int)($row['approved_amount'] ?? 0);
            $status = strtolower($row['requisition_status'] ?? '');
            $payment_status = strtolower($row['payment_status'] ?? 'pending');
            
            $req_date = $row['requisition_date'] ? date('d/m/Y', strtotime($row['requisition_date'])) : '—';
            $pay_date = $row['expense_date'] ? date('d/m/Y', strtotime($row['expense_date'])) : '—';

            $html .= '<tr>';
            $html .= '<td>#' . htmlspecialchars((string)($row['requisition_id'] ?? ''), ENT_QUOTES, 'UTF-8') . '</td>';
            $html .= '<td>' . htmlspecialchars($row['categories'] ?? '-', ENT_QUOTES, 'UTF-8') . '</td>';
            $html .= '<td>' . htmlspecialchars($row['initiator_name'] ?? '', ENT_QUOTES, 'UTF-8') . '</td>';
            $html .= '<td>' . htmlspecialchars($row['approver_name'] ?: 'En Attente', ENT_QUOTES, 'UTF-8') . '</td>';
            $html .= '<td>' . htmlspecialchars($row['assigned_cashier_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8') . '</td>';
            $html .= '<td>' . htmlspecialchars($row['province_name'] ?? '', ENT_QUOTES, 'UTF-8') . '</td>';
            $html .= '<td>' . htmlspecialchars($row['region_name'] ?? '', ENT_QUOTES, 'UTF-8') . '</td>';
            $html .= '<td class="amount">' . number_format($initiated_amount, 0, ',', ' ') . '</td>';
            $html .= '<td class="amount">' . number_format($approved_amount, 0, ',', ' ') . '</td>';
            $html .= '<td>' . htmlspecialchars($receiver_name, ENT_QUOTES, 'UTF-8') . '</td>';
            $html .= '<td>' . htmlspecialchars($receiver_type, ENT_QUOTES, 'UTF-8') . '</td>';
            $html .= '<td><span class="status">' . htmlspecialchars(ucfirst($status), ENT_QUOTES, 'UTF-8') . '</span></td>';
            $html .= '<td><span class="status">' . htmlspecialchars(ucfirst($payment_status), ENT_QUOTES, 'UTF-8') . '</span></td>';
            $html .= '<td>' . htmlspecialchars($req_date, ENT_QUOTES, 'UTF-8') . '</td>';
            $html .= '<td>' . htmlspecialchars($pay_date, ENT_QUOTES, 'UTF-8') . '</td>';
            $html .= '</tr>';
        }

        $html .= '</tbody></table>';
        $pdf->writeHTML($html, true, false, true, false, '');
        $pdf->Output('expenses_history_' . date('Y-m-d_His') . '.pdf', 'I');
        $conn->close();
        exit;
    }
}

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Historique Dépenses - Masunzu Bar Hotel</title>
    <link rel="stylesheet" href="/masunzu_bar_hotel/assets/css/style.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        html {
            margin: 0 !important;
            padding: 0 !important;
            width: 100% !important;
            max-width: 100% !important;
            overflow-x: hidden !important;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 0 !important;
            padding: 0 !important;
            background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%);
            min-height: 100vh;
            width: 100% !important;
            max-width: 100% !important;
            box-sizing: border-box;
            overflow-x: hidden !important;
            display: block !important;
            align-items: stretch !important;
            justify-content: flex-start !important;
        }
        .container {
            width: 100% !important;
            max-width: 100% !important;
            min-width: 100% !important;
            margin: 0 !important;
            padding: 0 !important;
            background-color: #FFFFFF;
            box-sizing: border-box;
            overflow-x: hidden;
            position: relative;
        }
        h2 {
            color: #4B2F1F;
            margin: 0 0 10px 0;
            padding: 15px 15px 0 15px;
            font-size: 28px;
        }
        .info-text {
            color: #666;
            margin: 0 0 20px 0;
            padding: 0 15px;
            font-size: 14px;
        }
        .filters {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin: 0 0 25px 0;
            padding: 15px;
            background: #F9F9F9;
        }
        .filter-group {
            display: flex;
            flex-direction: column;
        }
        .filter-group label {
            color: #4B2F1F;
            font-weight: 600;
            margin-bottom: 6px;
            font-size: 13px;
            text-transform: uppercase;
        }
        .filter-group input, 
        .filter-group select {
            padding: 10px;
            border: 1px solid #DDD;
            font-size: 14px;
            transition: border-color 0.3s ease;
        }
        .filter-group input[list] {
            width: 100%;
        }
        .category-option:hover {
            background-color: #F4A261;
            color: #4B2F1F;
        }
        .category-option.highlight {
            background-color: #F4A261;
            color: #4B2F1F;
        }
        .modal {
            display: none;
            position: fixed;
            z-index: 2000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0,0,0,0.5);
        }
        .modal-content {
            background-color: #FFFFFF;
            margin: 10% auto;
            padding: 25px;
            border: 2px solid #4B2F1F;
            border-radius: 8px;
            width: 90%;
            max-width: 500px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.3);
        }
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #4B2F1F;
        }
        .modal-header h3 {
            margin: 0;
            color: #4B2F1F;
            font-size: 20px;
        }
        .close {
            color: #4B2F1F;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            line-height: 20px;
        }
        .close:hover {
            color: #F4A261;
        }
        .form-group {
            margin-bottom: 15px;
        }
        .form-group label {
            display: block;
            margin-bottom: 5px;
            color: #4B2F1F;
            font-weight: 600;
            font-size: 14px;
        }
        .form-group input {
            width: 100%;
            padding: 10px;
            border: 1px solid #DDD;
            border-radius: 4px;
            font-size: 14px;
            box-sizing: border-box;
        }
        .form-group input:focus {
            outline: none;
            border-color: #F4A261;
            box-shadow: 0 0 0 3px rgba(244, 162, 97, 0.1);
        }
        .form-help {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
        }
        .error-message {
            background-color: #F8D7DA;
            color: #721C24;
            padding: 10px;
            border-radius: 4px;
            margin-bottom: 15px;
            font-size: 14px;
        }
        .success-message {
            background-color: #D4EDDA;
            color: #155724;
            padding: 10px;
            border-radius: 4px;
            margin-bottom: 15px;
            font-size: 14px;
        }
        .modal-actions {
            display: flex;
            gap: 10px;
            justify-content: flex-end;
            margin-top: 20px;
        }
        .btn-modal {
            padding: 10px 20px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-weight: 600;
            font-size: 14px;
        }
        .btn-modal-primary {
            background: linear-gradient(135deg, #4B2F1F 0%, #6B4F2F 100%);
            color: #F4F0E4;
        }
        .btn-modal-secondary {
            background: #E0E0E0;
            color: #4B2F1F;
        }
        .filter-group input:focus, 
        .filter-group select:focus {
            outline: none;
            border-color: #F4A261;
            box-shadow: 0 0 0 3px rgba(244, 162, 97, 0.1);
        }
        .btn-group {
            display: flex;
            gap: 10px;
            grid-column: 1 / -1;
            justify-content: flex-end;
        }
        button[type="submit"],
        button[type="reset"] {
            padding: 10px 20px;
            border: none;
            cursor: pointer;
            font-weight: 600;
            font-size: 14px;
            transition: all 0.3s ease;
        }
        button[type="submit"] {
            background: linear-gradient(135deg, #4B2F1F 0%, #6B4F2F 100%);
            color: #F4F0E4;
        }
        button[type="submit"]:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(75, 47, 31, 0.3);
        }
        button[type="reset"] {
            background: #E0E0E0;
            color: #4B2F1F;
        }
        button[type="reset"]:hover {
            background: #D0D0D0;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
            gap: 12px;
            margin: 0 15px 20px 15px;
        }
        .stat-card {
            background: linear-gradient(135deg, #F4A261 0%, #F4B580 100%);
            padding: 15px;
            text-align: center;
            color: #4B2F1F;
        }
        .stat-card p {
            margin: 0;
        }
        .stat-card .label {
            font-size: 12px;
            opacity: 0.8;
            text-transform: uppercase;
            font-weight: 600;
        }
        .stat-card .value {
            font-size: 22px;
            font-weight: bold;
            margin-top: 5px;
        }
        .table-container {
            overflow-x: auto;
            margin-top: 20px;
            width: 100%;
            padding: 0 0 15px 0;
        }
        .table-container table {
            padding: 0;
        }
        table {
            width: 100%;
            border-collapse: collapse;
        }
        th, td {
            padding: 14px;
            text-align: left;
            border-bottom: 1px solid #E0E0E0;
        }
        .table-container th:first-child,
        .table-container td:first-child {
            padding-left: 15px;
        }
        .table-container th:last-child,
        .table-container td:last-child {
            padding-right: 15px;
        }
        th {
            background-color: #4B2F1F;
            color: #F4F0E4;
            font-weight: 600;
            font-size: 13px;
            text-transform: uppercase;
        }
        tr:nth-child(even) {
            background-color: #F9F9F9;
        }
        tr:hover {
            background-color: #F0F0F0;
        }
        .status-badge {
            display: inline-block;
            padding: 6px 12px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
        }
        .status-pending {
            background-color: #FFF3CD;
            color: #856404;
        }
        .status-approved {
            background-color: #D4EDDA;
            color: #155724;
        }
        .status-paid {
            background-color: #D1ECF1;
            color: #0C5460;
        }
        .status-rejected {
            background-color: #F8D7DA;
            color: #721C24;
        }
        .status-processed {
            background-color: #E7D4F5;
            color: #5A2D81;
        }
        .no-data {
            text-align: center;
            color: #999;
            padding: 40px 15px;
            font-size: 16px;
        }
        .amount {
            font-weight: 600;
            color: #4B2F1F;
        }
        @media (max-width: 768px) {
            .container {
                padding: 0;
            }
            h2 {
                padding: 15px 15px 0 15px;
            }
            .info-text {
                padding: 0 15px;
            }
            .stats-grid {
                margin: 0 15px 20px 15px;
                grid-template-columns: repeat(2, 1fr);
            }
            .filters {
                grid-template-columns: 1fr;
                padding: 15px;
            }
            .btn-group {
                grid-column: auto;
            }
            table {
                font-size: 13px;
            }
            th, td {
                padding: 10px;
            }
        }
    </style>
</head>
<body>
    <?php include __DIR__ . '/../../includes/header.php'; ?>
    <div class="container">
        <h2>💼 Historique Dépenses</h2>
        <p class="info-text">Voir et gérer les réquisitions de dépenses</p>

        <!-- Statistics -->
        <div class="stats-grid">
            <div class="stat-card">
                <p class="label">Total Réquisitions</p>
                <p class="value"><?php echo count($expenses); ?></p>
            </div>
            <div class="stat-card">
                <p class="label">Montant Total</p>
                <p class="value"><?php 
                    $total_approved = array_sum(array_map(function($e) { return (int)($e['approved_amount'] ?? $e['total_amount'] ?? 0); }, $expenses));
                    echo number_format($total_approved, 0, '.', ','); 
                ?> BIF</p>
            </div>
        </div>

        <?php if (isset($_GET['category_created']) && $_GET['category_created'] === '1'): ?>
            <div class="success-message" style="margin: 15px; padding: 15px; background-color: #D4EDDA; color: #155724; border-radius: 4px;">
                ✅ Catégorie créée avec succès!
            </div>
        <?php endif; ?>

        <!-- Category Creation Modal -->
        <?php if ($can_create_category): ?>
        <div id="categoryModal" class="modal">
            <div class="modal-content">
                <div class="modal-header">
                    <h3>➕ Créer une Nouvelle Catégorie</h3>
                    <span class="close" onclick="closeCategoryModal()">&times;</span>
                </div>
                <?php if ($category_error): ?>
                    <div class="error-message"><?php echo htmlspecialchars($category_error, ENT_QUOTES); ?></div>
                <?php endif; ?>
                <form method="POST" action="" id="categoryForm">
                    <input type="hidden" name="create_category" value="1">
                    <?php 
                    // Preserve current filters in form
                    foreach ($_GET as $key => $value) {
                        if ($key !== 'create_category' && $key !== 'category_label') {
                            echo '<input type="hidden" name="' . htmlspecialchars($key, ENT_QUOTES) . '" value="' . htmlspecialchars($value, ENT_QUOTES) . '">';
                        }
                    }
                    ?>
                    <div class="form-group">
                        <label for="category_label">Nom de la Catégorie *</label>
                        <input type="text" id="category_label" name="category_label" required 
                               placeholder="ex: Marketing, Formation, Équipement, etc."
                               value="<?php echo htmlspecialchars($_POST['category_label'] ?? '', ENT_QUOTES); ?>"
                               oninput="updateCategoryCode(this.value)">
                        <div class="form-help">Le code sera généré automatiquement à partir du nom</div>
                    </div>
                    <div class="form-group">
                        <label for="category_code_preview">Code généré (automatique):</label>
                        <input type="text" id="category_code_preview" readonly 
                               style="background-color: #F5F5F5; color: #666; cursor: not-allowed;"
                               placeholder="Le code apparaîtra ici...">
                        <div class="form-help" style="font-size: 11px; color: #999;">Ce code sera utilisé en interne (non modifiable)</div>
                    </div>
                    <div class="modal-actions">
                        <button type="button" class="btn-modal btn-modal-secondary" onclick="closeCategoryModal()">Annuler</button>
                        <button type="submit" class="btn-modal btn-modal-primary">Créer la Catégorie</button>
                    </div>
                </form>
            </div>
        </div>
        <?php endif; ?>

        <!-- Filters Form -->
        <form class="filters" method="GET">
            <div class="filter-group">
                <label for="search">Rechercher (Nom/Description):</label>
                <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($search, ENT_QUOTES); ?>" placeholder="Rechercher...">
            </div>
            <div class="filter-group">
                <label for="from_date">Date Début:</label>
                <input type="date" id="from_date" name="from_date" value="<?php echo htmlspecialchars($from_date, ENT_QUOTES); ?>">
            </div>
            <div class="filter-group">
                <label for="to_date">Date Fin:</label>
                <input type="date" id="to_date" name="to_date" value="<?php echo htmlspecialchars($to_date, ENT_QUOTES); ?>">
            </div>
            <div class="filter-group">
                <label for="category">Catégorie: <?php if ($can_create_category): ?><button type="button" onclick="openCategoryModal()" style="margin-left: 5px; padding: 4px 8px; background: #4B2F1F; color: #F4F0E4; border: none; border-radius: 3px; cursor: pointer; font-size: 11px; font-weight: 600;">+ Nouvelle</button><?php endif; ?></label>
                <div style="position: relative;">
                    <input type="text" id="category-search" placeholder="Tapez pour rechercher une catégorie..." autocomplete="off" style="width: 100%; padding: 10px; border: 1px solid #DDD; font-size: 14px;" onkeyup="showCategorySuggestions(this.value)" onfocus="showCategorySuggestions(this.value)" onblur="setTimeout(function(){ hideCategorySuggestions(); }, 200);">
                    <input type="hidden" id="category" name="category" value="<?php echo htmlspecialchars($filter_category, ENT_QUOTES); ?>">
                    <div id="category-suggestions" style="display: none; position: absolute; top: 100%; left: 0; right: 0; background: white; border: 1px solid #DDD; border-top: none; max-height: 200px; overflow-y: auto; z-index: 1000; box-shadow: 0 4px 6px rgba(0,0,0,0.1);">
                        <div class="category-option" data-value="" onclick="selectCategory('', 'Toutes les catégories')" style="padding: 10px; cursor: pointer; border-bottom: 1px solid #EEE;">
                            <strong>Toutes les catégories</strong>
                        </div>
                        <?php foreach ($categoryOptions as $catValue => $catLabel): ?>
                            <div class="category-option" data-value="<?php echo htmlspecialchars($catValue, ENT_QUOTES); ?>" data-label="<?php echo htmlspecialchars($catLabel, ENT_QUOTES); ?>" onclick="selectCategory('<?php echo htmlspecialchars($catValue, ENT_QUOTES); ?>', '<?php echo htmlspecialchars($catLabel, ENT_QUOTES); ?>')" style="padding: 10px; cursor: pointer; border-bottom: 1px solid #EEE;">
                                <?php echo htmlspecialchars($catLabel, ENT_QUOTES); ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php if (in_array($role_id, [1, 2, 3, 4, 7, 9])): ?>
                <div class="filter-group">
                    <label for="province">Dépôt:</label>
                    <select id="province" name="province">
                        <option value="">Tous les Dépôts</option>
                        <?php foreach ($provinces as $prov): ?>
                            <option value="<?php echo $prov['id']; ?>" <?php echo $filter_province == $prov['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($prov['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="filter-group">
                    <label for="region">Région:</label>
                    <select id="region" name="region">
                        <option value="">Toutes les Régions</option>
                        <?php foreach ($regions as $reg): ?>
                            <option value="<?php echo $reg['id']; ?>" <?php echo (string)$filter_region === (string)$reg['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($reg['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php if (in_array($role_id, [1, 2, 3])): ?>
                    <div class="filter-group">
                        <label for="expense_status">Statut:</label>
                        <select id="expense_status" name="expense_status">
                            <option value="">Tous les Statuts</option>
                            <option value="pending" <?php echo $expense_status === 'pending' ? 'selected' : ''; ?>>En Attente</option>
                            <option value="approved" <?php echo $expense_status === 'approved' ? 'selected' : ''; ?>>Approuvé</option>
                            <option value="paid" <?php echo $expense_status === 'paid' ? 'selected' : ''; ?>>Payé</option>
                            <option value="rejected" <?php echo $expense_status === 'rejected' ? 'selected' : ''; ?>>Rejeté</option>
                        </select>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
            <div class="btn-group">
                <button type="reset">Réinitialiser</button>
                <button type="submit">Appliquer Filtres</button>
                <button type="submit" name="export" value="csv" style="background:#2d6a4f; color:#fff;">Exporter CSV</button>
                <button type="submit" name="export" value="pdf" style="background:#1d3557; color:#fff;">Exporter PDF</button>
            </div>
        </form>

        <!-- Table -->
        <?php if (empty($expenses)): ?>
            <div class="no-data">📭 Aucune dépense trouvée correspondant à vos critères</div>
        <?php else: ?>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Initiateur</th>
                            <th>Destinataire</th>
                            <th>Dépôt</th>
                            <th>Région</th>
                            <th>Catégories</th>
                            <th>Montant</th>
                            <th>Statut</th>
                            <th>Approbateur</th>
                            <th>Caissier Désigné</th>
                            <th>Statut Paiement</th>
                            <th>Date Demande</th>
                            <th>Date Paiement</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($expenses as $expense): ?>
                            <tr>
                                <td>#<?php echo htmlspecialchars($expense['requisition_id']); ?></td>
                                <td><?php echo htmlspecialchars($expense['initiator_name']); ?></td>
                                <td>
                                    <?php 
                                    if (isset($expense['recipient_type']) && $expense['recipient_type'] === 'external') {
                                        $external_name = htmlspecialchars($expense['recipient_external_name'] ?? 'Externe');
                                        $external_phone = htmlspecialchars($expense['recipient_external_phone'] ?? '');
                                        echo $external_name . ($external_phone ? ' (' . $external_phone . ')' : '');
                                    } else {
                                        echo htmlspecialchars($expense['recipient_name'] ?? 'N/A');
                                    }
                                    ?>
                                </td>
                                <td><?php echo htmlspecialchars($expense['province_name'] ?? 'N/A'); ?></td>
                                <td><?php echo htmlspecialchars($expense['region_name'] ?? 'N/A'); ?></td>
                                <td><?php echo htmlspecialchars($expense['categories'] ?? '-', ENT_QUOTES); ?></td>
                                <td class="amount">
                                    <?php 
                                    $initiated_amount = (int)($expense['initiated_amount'] ?? $expense['approved_amount'] ?? 0);
                                    $approved_amount = (int)($expense['approved_amount'] ?? 0);
                                    $amount_changed = $initiated_amount !== $approved_amount;
                                    ?>
                                    <div style="display: flex; flex-direction: column; gap: 4px;">
                                        <div>
                                            <strong>Initiation:</strong> <?php echo number_format($initiated_amount, 0, ',', '.'); ?> BIF
                                        </div>
                                        <div style="<?php echo $amount_changed ? 'color: #C62828; font-weight: 600;' : 'color: #357A38;'; ?>">
                                            <strong>Approuvé:</strong> <?php echo number_format($approved_amount, 0, ',', '.'); ?> BIF
                                            <?php if ($amount_changed): ?>
                                                <span style="font-size: 11px; color: #C62828; margin-left: 5px;">(Modifié)</span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <?php
                                    $status = strtolower($expense['requisition_status']);
                                    $status_fr = ($status === 'pending') ? 'En Attente' : (($status === 'approved') ? 'Approuvé' : (($status === 'paid') ? 'Payé' : (($status === 'rejected') ? 'Rejeté' : ucfirst(str_replace('_', ' ', $status)))));
                                    ?>
                                    <span class="status-badge status-<?php echo $status; ?>">
                                        <?php echo htmlspecialchars($status_fr); ?>
                                    </span>
                                </td>
                                <td><?php 
                                $approver = $expense['approver_name'] ?? '';
                                echo htmlspecialchars($approver !== '' ? $approver : 'En Attente'); 
                                ?></td>
                                <td><?php echo htmlspecialchars($expense['assigned_cashier_name'] ?? 'N/A'); ?></td>
                                <td>
                                    <?php if ($expense['payment_status']): ?>
                                        <?php
                                        $payment_status = strtolower($expense['payment_status']);
                                        $payment_status_fr = ($payment_status === 'pending') ? 'En Attente' : (($payment_status === 'processed') ? 'Traité' : (($payment_status === 'paid') ? 'Payé' : ucfirst($payment_status)));
                                        ?>
                                        <span class="status-badge status-<?php echo $payment_status; ?>">
                                            <?php echo htmlspecialchars($payment_status_fr); ?>
                                        </span>
                                    <?php else: ?>
                                        <span class="status-badge status-pending">En Attente</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo $expense['requisition_date'] ? date('d M Y', strtotime($expense['requisition_date'])) : '—'; ?></td>
                                <td><?php echo !empty($expense['expense_date']) ? date('d M Y', strtotime($expense['expense_date'])) : '—'; ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
    <?php include __DIR__ . '/../../includes/footer.php'; ?>
    <script>
        const categoryOptions = <?php echo json_encode($categoryOptions); ?>;
        let selectedCategoryValue = '<?php echo htmlspecialchars($filter_category, ENT_QUOTES); ?>';
        
        function showCategorySuggestions(searchTerm) {
            const suggestionsDiv = document.getElementById('category-suggestions');
            const options = suggestionsDiv.querySelectorAll('.category-option');
            const searchLower = (searchTerm || '').toLowerCase().trim();
            
            let hasMatches = false;
            
            options.forEach(function(option) {
                const value = option.getAttribute('data-value');
                const label = option.getAttribute('data-label') || option.textContent.trim();
                const labelLower = label.toLowerCase();
                
                // Always show "Toutes les catégories" if search is empty
                if (value === '') {
                    option.style.display = searchTerm === '' ? 'block' : 'block';
                    hasMatches = true;
                } else if (searchTerm === '' || labelLower.includes(searchLower) || value.toLowerCase().includes(searchLower)) {
                    option.style.display = 'block';
                    hasMatches = true;
                    
                    // Highlight matching text
                    if (searchTerm && labelLower.includes(searchLower)) {
                        option.classList.add('highlight');
                    } else {
                        option.classList.remove('highlight');
                    }
                } else {
                    option.style.display = 'none';
                    option.classList.remove('highlight');
                }
            });
            
            if (hasMatches) {
                suggestionsDiv.style.display = 'block';
            } else {
                suggestionsDiv.style.display = 'none';
            }
        }
        
        function hideCategorySuggestions() {
            const suggestionsDiv = document.getElementById('category-suggestions');
            suggestionsDiv.style.display = 'none';
        }
        
        function selectCategory(value, label) {
            const searchInput = document.getElementById('category-search');
            const hiddenInput = document.getElementById('category');
            
            selectedCategoryValue = value;
            hiddenInput.value = value;
            
            if (value === '') {
                searchInput.value = '';
            } else {
                searchInput.value = label;
            }
            
            hideCategorySuggestions();
        }
        
        // Set initial value on page load
        document.addEventListener('DOMContentLoaded', function() {
            const searchInput = document.getElementById('category-search');
            const hiddenInput = document.getElementById('category');
            
            if (selectedCategoryValue && categoryOptions[selectedCategoryValue]) {
                searchInput.value = categoryOptions[selectedCategoryValue];
            } else if (selectedCategoryValue === '') {
                searchInput.value = '';
            }
            
            // Handle keyboard navigation
            let highlightedIndex = -1;
            const suggestionsDiv = document.getElementById('category-suggestions');
            
            searchInput.addEventListener('keydown', function(e) {
                const visibleOptions = Array.from(suggestionsDiv.querySelectorAll('.category-option')).filter(opt => opt.style.display !== 'none');
                
                if (e.key === 'ArrowDown') {
                    e.preventDefault();
                    highlightedIndex = (highlightedIndex + 1) % visibleOptions.length;
                    visibleOptions.forEach((opt, idx) => {
                        opt.classList.toggle('highlight', idx === highlightedIndex);
                    });
                    if (visibleOptions[highlightedIndex]) {
                        visibleOptions[highlightedIndex].scrollIntoView({ block: 'nearest' });
                    }
                } else if (e.key === 'ArrowUp') {
                    e.preventDefault();
                    highlightedIndex = highlightedIndex <= 0 ? visibleOptions.length - 1 : highlightedIndex - 1;
                    visibleOptions.forEach((opt, idx) => {
                        opt.classList.toggle('highlight', idx === highlightedIndex);
                    });
                    if (visibleOptions[highlightedIndex]) {
                        visibleOptions[highlightedIndex].scrollIntoView({ block: 'nearest' });
                    }
                } else if (e.key === 'Enter') {
                    e.preventDefault();
                    if (highlightedIndex >= 0 && visibleOptions[highlightedIndex]) {
                        const option = visibleOptions[highlightedIndex];
                        const value = option.getAttribute('data-value');
                        const label = option.getAttribute('data-label') || option.textContent.trim();
                        selectCategory(value, label);
                    }
                } else if (e.key === 'Escape') {
                    hideCategorySuggestions();
                }
            });
        });
        
        function updateCategoryCode(label) {
            const previewField = document.getElementById('category_code_preview');
            if (!previewField) return;
            
            if (!label || label.trim() === '') {
                previewField.value = '';
                return;
            }
            
            // Generate code from label: lowercase, remove accents, replace special chars with underscores
            let code = label.toLowerCase();
            
            // Remove accents (basic approach)
            code = code.normalize('NFD').replace(/[\u0300-\u036f]/g, '');
            
            // Replace spaces and special characters with underscores
            code = code.replace(/[^a-z0-9_]+/g, '_');
            
            // Remove multiple consecutive underscores
            code = code.replace(/_+/g, '_');
            
            // Remove leading/trailing underscores
            code = code.replace(/^_+|_+$/g, '');
            
            previewField.value = code || '';
        }
        
        function openCategoryModal() {
            document.getElementById('categoryModal').style.display = 'block';
            // Initialize code preview if there's already a value
            const labelInput = document.getElementById('category_label');
            if (labelInput) {
                updateCategoryCode(labelInput.value);
            }
        }
        
        function closeCategoryModal() {
            document.getElementById('categoryModal').style.display = 'none';
            // Clear form when closing
            const form = document.getElementById('categoryForm');
            if (form) {
                form.reset();
                const previewField = document.getElementById('category_code_preview');
                if (previewField) {
                    previewField.value = '';
                }
            }
        }
        
        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('categoryModal');
            if (event.target == modal) {
                closeCategoryModal();
            }
        }
    </script>
</body>
</html>
