<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

// Initialize database connection
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in view_credit_payments.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Restrict to Admin (role_id = 1), DG (role_id = 2) and DGA (role_id = 3)
$stmt = $conn->prepare("SELECT u.id, u.full_name, r.name, r.id AS role_id FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role_result = $stmt->get_result();
$role = $role_result->fetch_assoc();
$stmt->close();
if (!$role || !in_array($role['role_id'], [1, 2, 3])) {
    error_log("Unauthorized access to view_credit_payments.php by user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
    $conn->close();
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $role['name'])) . "_dashboard.php");
    exit;
}

// Fetch customers with loan balances and repayment history
$customers = [];
$stmt = $conn->prepare("
    SELECT
        c.id,
        c.full_name,
        c.loan_limit_bif,
        c.loan_balance_bif,
        c.loan_status,
        c.can_pay_by_loan,
        (
            SELECT COALESCE(SUM(lp.payment_amount), 0)
            FROM loan_payment lp
            WHERE lp.customer_id = c.id
        ) AS total_repaid
    FROM customer c
    WHERE c.loan_balance_bif > 0 OR c.can_pay_by_loan = 1
    GROUP BY c.id, c.full_name, c.loan_limit_bif, c.loan_balance_bif, c.loan_status, c.can_pay_by_loan
    ORDER BY c.full_name
");
$stmt->execute();
$customers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

include __DIR__ . '/../../includes/header.php';
?>
<div class="credit-payments-container" style="padding: 20px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 70vh;">
    <div class="credit-payments-content" style="background-color: #F4F0E4; padding: 30px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); max-width: 1000px; margin: 0 auto;">
        <h2 style="color: #4B2F1F; font-size: 28px; margin-bottom: 15px;">Loan Exposure Tracking</h2>
        <p style="color: #000000; font-size: 16px;">Overview of customers approved for loans and their reimbursement progress.</p>
        <?php if (empty($customers)): ?>
            <p style="color: #FF0000; margin-top: 20px;">No customers currently have loan balances.</p>
        <?php else: ?>
            <table style="width: 100%; border-collapse: collapse; margin-top: 15px;">
                <thead>
                    <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                        <th style="padding: 10px; text-align: left;">Full Name</th>
                        <th style="padding: 10px; text-align: right;">Loan Limit (BIF)</th>
                        <th style="padding: 10px; text-align: right;">Outstanding Loan (BIF)</th>
                        <th style="padding: 10px; text-align: right;">Total Repaid (BIF)</th>
                        <th style="padding: 10px; text-align: right;">Remaining Limit (BIF)</th>
                        <th style="padding: 10px; text-align: left;">Loan Status</th>
                        <th style="padding: 10px; text-align: center;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($customers as $customer): ?>
                        <tr style="background-color: #FFFFFF;">
                            <td style="padding: 10px;"><?php echo htmlspecialchars($customer['full_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 10px; text-align: right;"><?php echo number_format((int)($customer['loan_limit_bif'] ?? 0), 0, ',', '.'); ?></td>
                            <td style="padding: 10px; text-align: right;"><?php echo number_format((int)($customer['loan_balance_bif'] ?? 0), 0, ',', '.'); ?></td>
                            <td style="padding: 10px; text-align: right;"><?php echo number_format((int)($customer['total_repaid'] ?? 0), 0, ',', '.'); ?></td>
                            <td style="padding: 10px; text-align: right;"><?php
                                $remaining = max(0, (int)($customer['loan_limit_bif'] ?? 0) - (int)($customer['loan_balance_bif'] ?? 0));
                                echo $customer['loan_limit_bif'] > 0 ? number_format($remaining, 0, ',', '.') : 'N/A';
                            ?></td>
                            <td style="padding: 10px;"><?php echo htmlspecialchars(ucfirst($customer['loan_status'] ?? 'inactive'), ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 10px; text-align: center;">
                                <?php if ((int)($customer['loan_balance_bif'] ?? 0) > 0): ?>
                                    <a href="/masunzu_bar_hotel/modules/cashiers/loan_repayments.php?customer_id=<?php echo htmlspecialchars($customer['id'], ENT_QUOTES, 'UTF-8'); ?>" style="background-color: #4CAF50; color: #FFFFFF; padding: 8px 15px; border-radius: 5px; text-decoration: none;">Record Repayment</a>
                                <?php else: ?>
                                    <span style="color: #2C1810; font-size: 14px;">No outstanding loan</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
<?php $conn->close(); ?>
