<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$base_dir = '/opt/lampp/htdocs/masunzu_bar_hotel';
require_once $base_dir . '/includes/auth.php';
require_once $base_dir . '/config/database.php';
require_once $base_dir . '/includes/functions.php';

try {
    require_login();

    $conn = get_db_connection();
    if (!$conn) {
        throw new Exception("Connexion à la base de données impossible");
    }

    // Allowed roles for viewing product history
    $allowed_roles = ['DG', 'DGA', 'Admin', 'Operation Manager', 'Stock Manager', 'Senior Accountant', 'Junior Accountant', 'Marketing Head', 'Chef Marketing', 'Marketing Agent', 'Agent Marketing'];
    
    if (!in_array($_SESSION['role_name'], $allowed_roles)) {
        error_log("ERROR: Unauthorized access to product_history by user_id={$_SESSION['user_id']}, role={$_SESSION['role_name']}");
        header("Location: /masunzu_bar_hotel/dashboards/unauthorized.php");
        exit;
    }

    // Get product ID from query string
    $product_id = isset($_GET['id']) ? (int)$_GET['id'] : null;

    if (!$product_id) {
        throw new Exception("Aucun identifiant produit fourni");
    }

    // Fetch product details
    $stmt = $conn->prepare("SELECT id, name, volume_cl, crate_quantity, price_per_crate, created_by, created_at FROM product WHERE id = ?");
    $stmt->bind_param("i", $product_id);
    $stmt->execute();
    $product = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$product) {
        throw new Exception("Produit introuvable");
    }

    // Get creator information
    $creator_name = "Système";
    if ($product['created_by']) {
        $stmt = $conn->prepare("SELECT full_name FROM user WHERE id = ?");
        $stmt->bind_param("i", $product['created_by']);
        $stmt->execute();
        $creator = $stmt->get_result()->fetch_assoc();
        if ($creator) {
            $creator_name = $creator['full_name'];
        }
        $stmt->close();
    }

    // Fetch all price change history
    $history = [];
    
    // Get initial entry from product table
    $initial_entry = [
        'type' => 'creation',
        'changed_by_name' => $creator_name,
        'old_price' => null,
        'new_price' => $product['price_per_crate'],
        'changed_at' => $product['created_at'],
        'change_reason' => 'Produit créé'
    ];
    $history[] = $initial_entry;

    // Check if price history table exists before querying
    $priceHistoryAvailable = false;
    $tableCheck = $conn->query("SHOW TABLES LIKE 'product_price_history'");
    if ($tableCheck && $tableCheck->num_rows > 0) {
        $priceHistoryAvailable = true;
    }
    if ($tableCheck) {
        $tableCheck->free();
    }

    if ($priceHistoryAvailable) {
        // Determine correct column names for history table (handles legacy schema)
        $oldPriceColumn = 'old_price';
        $newPriceColumn = 'new_price';
        $changeReasonSelect = 'NULL';
        $changeReasonColumn = false;
        $columnMap = [
            'old_price' => ['old_price', 'old_price_per_crate'],
            'new_price' => ['new_price', 'new_price_per_crate']
        ];

        foreach ($columnMap as $key => $candidates) {
            $found = false;
            foreach ($candidates as $candidate) {
                $check = $conn->query("SHOW COLUMNS FROM product_price_history LIKE '{$candidate}'");
                if ($check && $check->num_rows > 0) {
                    if ($key === 'old_price') {
                        $oldPriceColumn = $candidate;
                    } else {
                        $newPriceColumn = $candidate;
                    }
                    $found = true;
                    $check->free();
                    break;
                }
                if ($check) {
                    $check->free();
                }
            }

            if (!$found) {
                $priceHistoryAvailable = false;
                error_log("product_history.php: Missing {$key} column in product_price_history table");
                break;
            }
        }

        // Optional change_reason column
        $reasonCheck = $conn->query("SHOW COLUMNS FROM product_price_history LIKE 'change_reason'");
        if ($reasonCheck && $reasonCheck->num_rows > 0) {
            $changeReasonColumn = true;
            $changeReasonSelect = 'pph.change_reason';
        }
        if ($reasonCheck) {
            $reasonCheck->free();
        }
    }

    if ($priceHistoryAvailable) {
        // Get all price change records
        $stmt = $conn->prepare("
            SELECT 
                pph.id,
                pph.product_id,
                pph.changed_by,
                pph.{$oldPriceColumn} AS old_price,
                pph.{$newPriceColumn} AS new_price,
                {$changeReasonSelect} AS change_reason,
                pph.changed_at,
                u.full_name
            FROM product_price_history pph
            LEFT JOIN user u ON pph.changed_by = u.id
            WHERE pph.product_id = ?
            ORDER BY pph.changed_at DESC
        ");
        $stmt->bind_param("i", $product_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        while ($row = $result->fetch_assoc()) {
            $row['type'] = 'change';
            $row['changed_by_name'] = $row['full_name'] ?? 'Utilisateur inconnu';
            $history[] = $row;
        }
        $stmt->close();
    }

    // Sort by date descending (most recent first)
    usort($history, function($a, $b) {
        return strtotime($b['changed_at']) - strtotime($a['changed_at']);
    });

} catch (Exception $e) {
    $error_message = $e->getMessage();
    error_log("ERROR in product_history.php: " . $error_message);
}

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Historique des prix du produit - <?php echo htmlspecialchars($product['name'] ?? 'Produit'); ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #4B2F1F;
            --accent: #F4A261;
            --light: #F4F0E4;
            --dark: #2E1A12;
            --success: #27ae60;
            --error: #e74c3c;
            --border: #ddd;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, var(--light) 0%, #faf8f3 100%);
            color: var(--dark);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            overflow: hidden;
        }

        .header {
            background: linear-gradient(135deg, var(--primary) 0%, var(--dark) 100%);
            color: var(--light);
            padding: 30px;
            text-align: center;
        }

        .header h1 {
            font-size: 28px;
            margin-bottom: 10px;
        }

        .header p {
            font-size: 14px;
            opacity: 0.9;
        }

        .content {
            padding: 30px;
        }

        .product-info {
            background: linear-gradient(135deg, rgba(75, 47, 31, 0.08) 0%, rgba(244, 162, 97, 0.08) 100%);
            border-left: 4px solid var(--accent);
            padding: 20px;
            margin-bottom: 30px;
            border-radius: 4px;
        }

        .product-info h3 {
            color: var(--primary);
            margin-bottom: 12px;
            font-size: 16px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .info-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }

        .info-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px 0;
            border-bottom: 1px solid rgba(244, 162, 97, 0.2);
        }

        .info-item:last-child {
            border-bottom: none;
        }

        .info-label {
            font-weight: 600;
            color: var(--primary);
            font-size: 12px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .info-value {
            font-weight: 500;
            color: var(--dark);
        }

        .price-highlight {
            color: var(--accent);
            font-weight: 700;
            font-size: 16px;
        }

        .history-timeline {
            position: relative;
        }

        .timeline-item {
            display: grid;
            grid-template-columns: 140px 1fr;
            gap: 20px;
            margin-bottom: 25px;
            padding: 0;
            position: relative;
        }

        .timeline-item::before {
            content: '';
            position: absolute;
            left: 69px;
            top: 40px;
            bottom: -25px;
            width: 2px;
            background: linear-gradient(180deg, var(--accent) 0%, rgba(244, 162, 97, 0.3) 100%);
        }

        .timeline-item:last-child::before {
            display: none;
        }

        .timeline-marker {
            position: relative;
            z-index: 1;
        }

        .timeline-dot {
            width: 16px;
            height: 16px;
            background: var(--accent);
            border: 3px solid white;
            border-radius: 50%;
            position: absolute;
            left: 62px;
            top: 8px;
            box-shadow: 0 0 0 3px var(--accent);
        }

        .timeline-dot.creation {
            background: var(--primary);
            box-shadow: 0 0 0 3px var(--primary);
        }

        .timeline-date {
            font-size: 12px;
            font-weight: 600;
            color: var(--primary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-top: 8px;
        }

        .timeline-content {
            background: linear-gradient(135deg, rgba(75, 47, 31, 0.03) 0%, rgba(244, 162, 97, 0.03) 100%);
            border-left: 3px solid var(--accent);
            padding: 15px;
            border-radius: 4px;
            margin-top: -2px;
        }

        .timeline-content.creation {
            border-left-color: var(--primary);
        }

        .timeline-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 12px;
        }

        .timeline-title {
            font-weight: 600;
            color: var(--dark);
            font-size: 14px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .timeline-user {
            font-size: 12px;
            background: var(--accent);
            color: white;
            padding: 4px 10px;
            border-radius: 12px;
            font-weight: 600;
        }

        .timeline-reason {
            font-size: 13px;
            color: var(--primary);
            margin-bottom: 10px;
            font-style: italic;
        }

        .price-details {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 12px;
            margin-top: 10px;
        }

        .price-item {
            background: white;
            padding: 10px;
            border-radius: 4px;
            border: 1px solid var(--border);
        }

        .price-item label {
            font-size: 11px;
            font-weight: 600;
            color: var(--primary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            display: block;
            margin-bottom: 5px;
        }

        .price-change {
            display: flex;
            gap: 8px;
            align-items: center;
        }

        .old-price {
            font-size: 12px;
            color: var(--error);
            text-decoration: line-through;
        }

        .new-price {
            font-size: 14px;
            font-weight: 700;
            color: var(--success);
        }

        .price-arrow {
            color: var(--accent);
            font-weight: bold;
        }

        .no-changes {
            text-align: center;
            padding: 40px 20px;
            color: var(--primary);
        }

        .no-changes p {
            font-size: 14px;
            margin-bottom: 5px;
        }

        .buttons {
            display: flex;
            gap: 10px;
            margin-top: 30px;
            flex-wrap: wrap;
        }

        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 4px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            transition: all 0.3s ease;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--dark);
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(75, 47, 31, 0.3);
        }

        .btn-secondary {
            background: var(--accent);
            color: white;
        }

        .btn-secondary:hover {
            background: #f39450;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(244, 162, 97, 0.3);
        }

        .alert {
            padding: 15px 20px;
            border-radius: 4px;
            margin-bottom: 20px;
            font-size: 14px;
        }

        .alert-error {
            background: #ffe6e6;
            border-left: 4px solid var(--error);
            color: var(--error);
        }

        .alert-error::before {
            content: '❌ ';
            font-weight: bold;
        }

        @media (max-width: 768px) {
            .info-grid {
                grid-template-columns: 1fr;
            }

            .price-details {
                grid-template-columns: 1fr;
            }

            .timeline-item {
                grid-template-columns: 1fr;
                gap: 10px;
            }

            .timeline-item::before {
                left: 7px;
            }

            .timeline-dot {
                left: 0;
            }

            .timeline-date {
                margin-top: 0;
                margin-left: 0;
            }

            .content {
                padding: 20px;
            }

            .header {
                padding: 20px;
            }

            .header h1 {
                font-size: 20px;
            }

            .buttons {
                flex-direction: column;
            }

            .btn {
                width: 100%;
                text-align: center;
            }
        }

        .format-number {
            color: var(--accent);
            font-weight: 600;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>📊 Historique des prix du produit</h1>
            <p>Historique complet de toutes les modifications de prix</p>
        </div>

        <div class="content">
            <?php if (isset($error_message)): ?>
                <div class="alert alert-error">
                    <?php echo htmlspecialchars($error_message); ?>
                </div>
                <div class="buttons">
                </div>
            <?php else: ?>
                <!-- Product Information -->
                <div class="product-info">
                    <h3>📦 Informations actuelles du produit</h3>
                    <div class="info-grid">
                        <div class="info-item">
                            <span class="info-label">Nom du produit</span>
                            <span class="info-value"><?php echo htmlspecialchars($product['name']); ?></span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Volume</span>
                            <span class="info-value"><?php echo $product['volume_cl']; ?> cl</span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Quantité par caisse</span>
                            <span class="info-value"><?php echo $product['crate_quantity']; ?> unités</span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Créé le</span>
                            <span class="info-value"><?php echo date('d/m/Y H:i:s', strtotime($product['created_at'])); ?></span>
                        </div>
                        <div class="info-item">
                            <span class="info-label">Prix actuel / caisse</span>
                            <span class="info-value price-highlight"><?php echo number_format($product['price_per_crate']); ?> FBU</span>
                        </div>
                    </div>
                </div>

                <!-- Price Change History Timeline -->
                <h2 style="color: var(--primary); margin-bottom: 20px; font-size: 18px; text-transform: uppercase; letter-spacing: 0.5px;">⏱️ Historique des changements de prix</h2>

                <?php if (count($history) > 1): ?>
                    <div class="history-timeline">
                        <?php foreach ($history as $entry): ?>
                            <div class="timeline-item">
                                <div class="timeline-marker">
                                    <div class="timeline-dot <?php echo $entry['type']; ?>"></div>
                                    <div class="timeline-date">
                                        <?php echo date('d/m/Y', strtotime($entry['changed_at'])); ?>
                                    </div>
                                </div>
                                <div class="timeline-content <?php echo $entry['type']; ?>">
                                    <div class="timeline-header">
                                        <span class="timeline-title">
                                            <?php echo ($entry['type'] === 'creation') ? '✨ Produit créé' : '✏️ Prix mis à jour'; ?>
                                        </span>
                                        <span class="timeline-user"><?php echo htmlspecialchars($entry['changed_by_name']); ?></span>
                                    </div>
                                    
                                    <?php if ($entry['change_reason']): ?>
                                        <div class="timeline-reason"><?php echo htmlspecialchars($entry['change_reason']); ?></div>
                                    <?php endif; ?>

                                    <div style="font-size: 12px; color: var(--primary); margin-bottom: 10px; font-weight: 600;">
                                        ⏰ <?php echo date('d/m/Y · H:i:s', strtotime($entry['changed_at'])); ?>
                                    </div>

                                    <div class="price-details">
                                        <div class="price-item">
                                            <label>🏷️ Prix unitaire</label>
                                            <div class="price-change">
                                                <?php if ($entry['old_price'] !== null): ?>
                                                    <span class="old-price"><?php echo number_format($entry['old_price']); ?></span>
                                                    <span class="price-arrow">→</span>
                                                <?php endif; ?>
                                                <span class="new-price"><?php echo number_format($entry['new_price']); ?> FBU</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="no-changes">
                        <p>✨ Aucun changement de prix pour le moment</p>
                        <p style="font-size: 12px; color: #999;">Ce produit conserve son prix initial depuis sa création.</p>
                    </div>
                <?php endif; ?>

                <!-- Navigation Buttons -->
                <div class="buttons">
                    <a href="/masunzu_bar_hotel/modules/operation/view_product.php?id=<?php echo $product_id; ?>" class="btn btn-secondary">📊 Voir la fiche produit</a>
                </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
