<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// Restrict to Stock Manager role
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in receiving_stock_history.php");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$stmt = $conn->prepare("SELECT r.name FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$role = $stmt->get_result()->fetch_assoc()['name'] ?? '';
$stmt->close();
if ($role !== 'Stock Manager') {
    error_log("Unauthorized access to receiving_stock_history.php by user {$_SESSION['full_name']} (Role: $role)");
    header("Location: /masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $role)) . "_dashboard.php");
    exit;
}

// Fetch province_id
$province_id = $_SESSION['province_id'] ?? null;
if ($province_id === null || $province_id === '') {
    $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $province_id = $stmt->get_result()->fetch_assoc()['province_id'] ?? null;
    $_SESSION['province_id'] = $province_id;
    $stmt->close();
}
$province_id = $_SESSION['province_id'] ?? null;
if ($province_id === null || $province_id === '') {
    error_log("No province_id for user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Error: No province assigned. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}
$province_id = (int)$province_id;

// Fetch receiving history grouped by shipment_id
$stmt = $conn->prepare("
    SELECT 
        t.shipment_id,
        MAX(t.sent_at) AS sent_at,
        MAX(t.received_at) AS received_at,
        MAX(t.estimated_arrival) AS estimated_arrival,
        MAX(t.driver_name) AS driver_name,
        MAX(t.driver_mobile) AS driver_mobile,
        MAX(t.plate_number) AS plate_number,
        MAX(t.status) AS status,
        MAX(u1.full_name) AS sender_name,
        MAX(u1.id) AS sender_id,
        MAX(u2.full_name) AS receiver_name,
        MAX(u2.id) AS receiver_id,
        MAX(pr.name) AS province_name,
        COUNT(DISTINCT t.product_id) AS product_count,
        SUM(t.quantity) AS total_quantity,
        SUM(t.quantity * COALESCE(p.price_per_crate, 0)) AS total_value
    FROM stock_transfer t
    LEFT JOIN user u1 ON t.operation_manager_id = u1.id
    LEFT JOIN user u2 ON t.stock_manager_id = u2.id
    LEFT JOIN product p ON t.product_id = p.id
    LEFT JOIN province pr ON t.province_id = pr.id
    WHERE t.stock_manager_id = ? AND t.status = 'confirmed'
    GROUP BY t.shipment_id
    ORDER BY MAX(t.received_at) DESC
");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$receptions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch all products for each shipment (for modal details)
$shipment_products = [];
if (!empty($receptions) && count($receptions) > 0) {
    $shipment_ids = array_column($receptions, 'shipment_id');
    if (!empty($shipment_ids)) {
        $placeholders = str_repeat('?,', count($shipment_ids) - 1) . '?';
        $stmt = $conn->prepare("
            SELECT 
                t.shipment_id,
                t.product_id,
                p.name AS product_name,
                t.quantity,
                p.crate_quantity,
                p.price_per_crate,
                t.quantity * p.price_per_crate AS item_value
            FROM stock_transfer t
            JOIN product p ON t.product_id = p.id
            WHERE t.shipment_id IN ($placeholders) AND t.status = 'confirmed'
            ORDER BY t.shipment_id, p.name
        ");
        $types = str_repeat('i', count($shipment_ids));
        $stmt->bind_param($types, ...$shipment_ids);
        $stmt->execute();
        $products_result = $stmt->get_result();
        while ($row = $products_result->fetch_assoc()) {
            $shipment_products[$row['shipment_id']][] = $row;
        }
        $stmt->close();
    }
}

$conn->close();
include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
?>
<div class="container" style="padding: 20px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh;">
    <div class="content" style="background-color: #F4F0E4; padding: 30px; border-radius: 12px; box-shadow: 0 6px 12px rgba(0,0,0,0.15); max-width: 1000px; margin: 30px auto;">
        <h2 style="color: #4B2F1F; font-size: 28px; margin-bottom: 15px; font-weight: 600; text-align: center;">Receiving Stock History</h2>
        <p style="color: #4B2F1F; font-size: 16px; text-align: center;">Stock receptions for <?php echo htmlspecialchars($_SESSION['full_name'] ?? 'Unknown', ENT_QUOTES, 'UTF-8'); ?> (Stock Manager)</p>
        <?php if (empty($receptions)): ?>
            <p style="color: #4B2F1F; font-size: 16px; text-align: center;">No stock receptions found.</p>
        <?php else: ?>
            <div style="overflow-x: auto;">
                <table style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 8px; overflow: hidden; margin-bottom: 20px;">
                    <thead>
                        <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                            <th style="padding: 15px; text-align: left; width: 10%;">Transfert ID</th>
                            <th style="padding: 15px; text-align: left; width: 15%;">Sender</th>
                            <th style="padding: 15px; text-align: left; width: 15%;">Receiver</th>
                            <th style="padding: 15px; text-align: center; width: 10%;">Products</th>
                            <th style="padding: 15px; text-align: center; width: 10%;">Total Crates</th>
                            <th style="padding: 15px; text-align: right; width: 12%;">Total Value</th>
                            <th style="padding: 15px; text-align: left; width: 12%;">Sent At</th>
                            <th style="padding: 15px; text-align: left; width: 12%;">Received At</th>
                            <th style="padding: 15px; text-align: center; width: 10%;">Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($receptions as $reception): ?>
                            <tr style="border-bottom: 1px solid #F4F0E4;">
                                <td style="padding: 15px; text-align: left; font-weight: 600;"><?php echo htmlspecialchars($reception['shipment_id'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars($reception['sender_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo htmlspecialchars($reception['receiver_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 15px; text-align: center;"><?php echo number_format($reception['product_count'], 0, ',', '.'); ?></td>
                                <td style="padding: 15px; text-align: center; font-weight: 600;"><?php echo number_format($reception['total_quantity'], 0, ',', '.'); ?></td>
                                <td style="padding: 15px; text-align: right; font-weight: 600;"><?php echo number_format($reception['total_value'], 0, ',', '.'); ?> BIF</td>
                                <td style="padding: 15px; text-align: left;"><?php echo date('Y-m-d H:i', strtotime($reception['sent_at'])); ?></td>
                                <td style="padding: 15px; text-align: left;"><?php echo $reception['received_at'] ? date('Y-m-d H:i', strtotime($reception['received_at'])) : 'N/A'; ?></td>
                                <td style="padding: 15px; text-align: center;">
                                    <button onclick="showTransferDetails(<?php echo htmlspecialchars($reception['shipment_id'], ENT_QUOTES, 'UTF-8'); ?>)" 
                                            style="background-color: #4B2F1F; color: #F4F0E4; padding: 8px 16px; border: none; border-radius: 6px; cursor: pointer; font-size: 14px; font-weight: 500; transition: all 0.3s ease;">
                                        View
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
    <!-- Modal for Transfer Details -->
    <div id="transferModal" style="display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.5);">
        <div style="background-color: #F4F0E4; margin: 5% auto; padding: 30px; border-radius: 12px; width: 90%; max-width: 800px; box-shadow: 0 8px 16px rgba(0,0,0,0.3); max-height: 80vh; overflow-y: auto;">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; border-bottom: 2px solid #4B2F1F; padding-bottom: 15px;">
                <h2 style="color: #4B2F1F; font-size: 24px; font-weight: 600; margin: 0;">Transfert Details</h2>
                <span onclick="closeTransferModal()" style="color: #4B2F1F; font-size: 28px; font-weight: bold; cursor: pointer; transition: color 0.3s ease;">&times;</span>
            </div>
            <div id="transferModalContent">
                <!-- Content will be loaded here -->
            </div>
            <div style="margin-top: 20px; text-align: right;">
                <button onclick="closeTransferModal()" style="background-color: #4B2F1F; color: #F4F0E4; padding: 10px 24px; border: none; border-radius: 6px; cursor: pointer; font-size: 16px; font-weight: 500; transition: all 0.3s ease;">
                    Close
                </button>
            </div>
        </div>
    </div>

    <style>
        table { min-width: 600px; }
        button:hover {
            background-color: #F4A261 !important;
            color: #4B2F1F !important;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.3);
        }
        .modal-detail-row {
            display: flex;
            justify-content: space-between;
            padding: 12px 0;
            border-bottom: 1px solid #E0E0E0;
        }
        .modal-detail-label {
            font-weight: 600;
            color: #4B2F1F;
            width: 40%;
        }
        .modal-detail-value {
            color: #333;
            width: 60%;
            text-align: right;
        }
        .modal-product-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }
        .modal-product-table th {
            background-color: #4B2F1F;
            color: #F4F0E4;
            padding: 12px;
            text-align: left;
        }
        .modal-product-table td {
            padding: 10px 12px;
            border-bottom: 1px solid #E0E0E0;
        }
        .modal-product-table tr:hover {
            background-color: #F9F9F9;
        }
        @media (max-width: 768px) {
            .container { padding: 15px; }
            .content { padding: 20px; max-width: 100%; }
            h2 { font-size: 24px; }
            h3 { font-size: 20px; }
            p { font-size: 14px; }
            table { font-size: 14px; }
            #transferModal > div { width: 95%; margin: 10% auto; padding: 20px; }
        }
        @media (max-width: 480px) {
            h2 { font-size: 20px; }
            h3 { font-size: 18px; }
            p { font-size: 12px; }
            table { font-size: 12px; }
            .modal-detail-row { flex-direction: column; }
            .modal-detail-label, .modal-detail-value { width: 100%; }
            .modal-detail-value { text-align: left; margin-top: 5px; }
        }
    </style>

    <script>
        // Transfer data for modal
        const transferData = <?php echo json_encode($receptions); ?>;
        const shipmentProducts = <?php echo json_encode($shipment_products); ?>;

        function showTransferDetails(shipmentId) {
            const transfer = transferData.find(t => t.shipment_id == shipmentId);
            if (!transfer) {
                alert('Transfer details not found');
                return;
            }

            const products = shipmentProducts[shipmentId] || [];
            let productsHtml = '';
            if (products.length > 0) {
                productsHtml = '<h3 style="color: #4B2F1F; margin-top: 20px; margin-bottom: 10px; font-size: 18px;">Products:</h3>';
                productsHtml += '<table class="modal-product-table">';
                productsHtml += '<thead><tr><th>Product</th><th>Quantity (Crates)</th><th>Price per Crate</th><th>Total Value</th></tr></thead>';
                productsHtml += '<tbody>';
                products.forEach(product => {
                    const itemValue = (product.quantity || 0) * (product.price_per_crate || 0);
                    productsHtml += `<tr>
                        <td>${escapeHtml(product.product_name || 'N/A')}</td>
                        <td style="text-align: center;">${formatNumber(product.quantity || 0)}</td>
                        <td style="text-align: right;">${formatNumber(product.price_per_crate || 0)} BIF</td>
                        <td style="text-align: right; font-weight: 600;">${formatNumber(itemValue)} BIF</td>
                    </tr>`;
                });
                productsHtml += '</tbody></table>';
            }

            const modalContent = `
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Transfert ID:</div>
                    <div class="modal-detail-value">${escapeHtml(transfer.shipment_id)}</div>
                </div>
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Sender:</div>
                    <div class="modal-detail-value">${escapeHtml(transfer.sender_name || 'N/A')}</div>
                </div>
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Receiver:</div>
                    <div class="modal-detail-value">${escapeHtml(transfer.receiver_name || 'N/A')}</div>
                </div>
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Province:</div>
                    <div class="modal-detail-value">${escapeHtml(transfer.province_name || 'N/A')}</div>
                </div>
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Driver Name:</div>
                    <div class="modal-detail-value">${escapeHtml(transfer.driver_name || 'N/A')}</div>
                </div>
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Driver Mobile:</div>
                    <div class="modal-detail-value">${escapeHtml(transfer.driver_mobile || 'N/A')}</div>
                </div>
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Plate Number:</div>
                    <div class="modal-detail-value">${escapeHtml(transfer.plate_number || 'N/A')}</div>
                </div>
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Sent At:</div>
                    <div class="modal-detail-value">${formatDateTime(transfer.sent_at)}</div>
                </div>
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Estimated Arrival:</div>
                    <div class="modal-detail-value">${transfer.estimated_arrival ? formatDateTime(transfer.estimated_arrival) : 'N/A'}</div>
                </div>
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Received At:</div>
                    <div class="modal-detail-value">${transfer.received_at ? formatDateTime(transfer.received_at) : 'N/A'}</div>
                </div>
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Status:</div>
                    <div class="modal-detail-value"><strong>${escapeHtml(transfer.status ? transfer.status.charAt(0).toUpperCase() + transfer.status.slice(1) : 'N/A')}</strong></div>
                </div>
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Number of Products:</div>
                    <div class="modal-detail-value">${formatNumber(transfer.product_count || 0)}</div>
                </div>
                <div class="modal-detail-row">
                    <div class="modal-detail-label">Total Crates:</div>
                    <div class="modal-detail-value"><strong>${formatNumber(transfer.total_quantity || 0)}</strong></div>
                </div>
                <div class="modal-detail-row" style="border-bottom: 2px solid #4B2F1F; padding-bottom: 15px; margin-bottom: 15px;">
                    <div class="modal-detail-label">Total Value:</div>
                    <div class="modal-detail-value" style="font-size: 18px; font-weight: 700; color: #4B2F1F;">${formatNumber(transfer.total_value || 0)} BIF</div>
                </div>
                ${productsHtml}
            `;

            document.getElementById('transferModalContent').innerHTML = modalContent;
            document.getElementById('transferModal').style.display = 'block';
        }

        function closeTransferModal() {
            document.getElementById('transferModal').style.display = 'none';
        }

        // Close modal when clicking outside of it
        window.onclick = function(event) {
            const modal = document.getElementById('transferModal');
            if (event.target == modal) {
                closeTransferModal();
            }
        }

        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        function formatNumber(num) {
            return Number(num).toLocaleString('en-US').replace(/,/g, '.');
        }

        function formatDateTime(dateString) {
            if (!dateString) return 'N/A';
            const date = new Date(dateString);
            return date.toLocaleString('en-US', {
                year: 'numeric',
                month: '2-digit',
                day: '2-digit',
                hour: '2-digit',
                minute: '2-digit'
            }).replace(',', '');
        }
    </script>
</div>
<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
