<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/cash_helpers.php';

try {
    require_login();

    $role_name = $_SESSION['role_name'] ?? '';
    $role_id   = $_SESSION['role_id'] ?? 0;

    // Restrict to Accountant role only - viewing only, no payment functionality
    // Caissier_Comptables should use modules/cashiers/pay_requisitions.php instead
    if ($role_name !== 'Accountant') {
        $dashboard = strtolower(str_replace(' ', '_', $role_name)) . '_dashboard.php';
        header('Location: /masunzu_bar_hotel/dashboards/' . $dashboard);
        exit;
    }

    require_once __DIR__ . '/../../includes/header.php';

    $conn = get_db_connection();
    if (!$conn) {
        throw new Exception('Échec de la connexion à la base de données: ' . mysqli_connect_error());
    }

    // Ensure expense.payment_method column exists
    $column_check = $conn->query("SHOW COLUMNS FROM expense LIKE 'payment_method'");
    if (!$column_check) {
        throw new Exception('Failed to inspect expense table: ' . $conn->error);
    }
    if ($column_check->num_rows === 0) {
        if (!$conn->query("ALTER TABLE expense ADD COLUMN payment_method ENUM('cash','bank_transfer','mobile') DEFAULT 'cash'")) {
            throw new Exception('Failed to add payment_method column: ' . $conn->error);
        }
    }
    if ($column_check instanceof mysqli_result) {
        $column_check->free();
    }

    // Check if requisition.department exists (optional feature)
    $department_exists = false;
    $department_check = $conn->query("SHOW COLUMNS FROM requisition LIKE 'department'");
    if (!$department_check) {
        throw new Exception('Failed to inspect requisition table: ' . $conn->error);
    }
    if ($department_check->num_rows > 0) {
        $department_exists = true;
    }
    if ($department_check instanceof mysqli_result) {
        $department_check->free();
    }

    // Ensure we have province in session
    if (!isset($_SESSION['province_id']) || empty($_SESSION['province_id'])) {
        $stmt = $conn->prepare('SELECT province_id FROM user WHERE id = ?');
        $stmt->bind_param('i', $_SESSION['user_id']);
        $stmt->execute();
        $province_row = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        $_SESSION['province_id'] = $province_row['province_id'] ?? null;
    }
    $province_id = $_SESSION['province_id'];
    if (!$province_id) {
        throw new Exception('Aucune province assignée à ce comptable.');
    }

    // TCPDF availability check
    $tcpdf_path = __DIR__ . '/../../libs/tcpdf/tcpdf.php';
    $composer_autoload = __DIR__ . '/../../vendor/autoload.php';
    $tcpdf_available = false;
    if (file_exists($composer_autoload)) {
        require_once $composer_autoload;
        if (class_exists('TCPDF')) $tcpdf_available = true;
    } elseif (file_exists($tcpdf_path)) {
        require_once $tcpdf_path;
        if (class_exists('TCPDF')) $tcpdf_available = true;
    }

    $errors = [];
    $success = '';

    // =========================
    // Payment functionality removed - Accountants can only view requisitions
    // Payments must be made by Caissier_Comptables using modules/cashiers/pay_requisitions.php
    // =========================
    if (false && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['pay_requisition'])) {
        $requisition_id   = filter_input(INPUT_POST, 'requisition_id', FILTER_VALIDATE_INT);
        $cash_receiver_id = filter_input(INPUT_POST, 'cash_receiver_id', FILTER_VALIDATE_INT);
        $payment_method   = trim(filter_input(INPUT_POST, 'payment_method', FILTER_UNSAFE_RAW) ?? 'cash');
        $payment_comments = trim(filter_input(INPUT_POST, 'payment_comments', FILTER_UNSAFE_RAW) ?? '');

        if (!$requisition_id)   { $errors[] = 'Réquisition sélectionnée invalide.'; }
        if (!$cash_receiver_id) { $errors[] = 'Sélectionnez un récepteur d\'espèces.'; }
        if ($payment_comments === '') { $errors[] = 'Fournissez des commentaires de paiement.'; }
        if (!in_array($payment_method, ['cash', 'bank_transfer', 'mobile'], true)) {
            $errors[] = 'Méthode de paiement fournie invalide.';
        }

        if (empty($errors)) {
            // Fetch requisition for this province and ensure approved
            $stmt = $conn->prepare('
                SELECT r.id, r.total_amount, r.status, r.province_id, r.recipient_id,
                       r.recipient_type, r.recipient_external_name, r.recipient_external_phone,
                       r.created_at, u.full_name AS initiator_name, recip.full_name AS recipient_name, r.comments
                FROM requisition r
                LEFT JOIN user u ON r.user_id = u.id
                LEFT JOIN user recip ON r.recipient_id = recip.id
                WHERE r.id = ? AND r.province_id = ? AND r.status = "approved"
            ');
            $stmt->bind_param('ii', $requisition_id, $province_id);
            $stmt->execute();
            $requisition = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            if (!$requisition) {
                $errors[] = 'Réquisition introuvable ou non approuvée.';
            } else {
                $total_amount = (float)($requisition['total_amount'] ?? 0);

                // NOTE: We no longer rely on accountant_balance for funding this requisition.
                // The payment is treated as coming from the cashier's cash account (cashier shift),
                // while province_cash_account is debited at the province level via adjustProvinceCash().
                {
                    // Validate receiver is in same province
                    $stmt = $conn->prepare('SELECT id, full_name FROM user WHERE id = ? AND province_id = ?');
                    $stmt->bind_param('ii', $cash_receiver_id, $province_id);
                    $stmt->execute();
                    $receiver_row = $stmt->get_result()->fetch_assoc();
                    $stmt->close();

                    if (!$receiver_row) {
                        $errors[] = 'Le récepteur sélectionné n\'est pas valide pour cette province.';
                    } else {
                        // Requisition items
                        $stmt = $conn->prepare('SELECT ri.category, ri.description, ri.quantity, ri.unit_price 
                                                FROM requisition_item ri WHERE ri.requisition_id = ?');
                        $stmt->bind_param('i', $requisition_id);
                        $stmt->execute();
                        $items = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
                        $stmt->close();

                        // Generate unique receipt number
                        $receipt_number = '';
                        do {
                            $receipt_number = 'REQ-' . date('Ymd') . '-' . mt_rand(1000, 9999);
                            $check_stmt = $conn->prepare('SELECT id FROM expense WHERE receipt_number = ? LIMIT 1');
                            $check_stmt->bind_param('s', $receipt_number);
                            $check_stmt->execute();
                            $exists = $check_stmt->get_result()->num_rows > 0;
                            $check_stmt->close();
                        } while ($exists);

                        $conn->begin_transaction();
                        try {
                            // 1) Insert expense (proof of payment)
                        $receiver_user_id = null;
                        $receiver_external_name = null;
                        $receiver_external_phone = null;
                        if (($requisition['recipient_type'] ?? 'internal') === 'external') {
                            $receiver_external_name = $requisition['recipient_external_name'] ?? '';
                            $receiver_external_phone = $requisition['recipient_external_phone'] ?? '';
                        } else {
                            $receiver_user_id = (int)($requisition['recipient_id'] ?? 0) ?: null;
                        }

                        $stmt = $conn->prepare('
                            INSERT INTO expense 
                                (requisition_id, cashier_id, amount, receipt_number, recipient_signature, accountant_id, approval_status, payment_status, payment_method, receiver_user_id, receiver_external_name, receiver_external_phone, created_at) 
                            VALUES (?, ?, ?, ?, ?, ?, "approved", "processed", ?, ?, ?, ?, NOW())
                        ');
                        if (!$stmt) {
                            throw new Exception("Query preparation failed for expense insert: " . $conn->error);
                        }
                        $stmt->bind_param('iidssisiss',
                            $requisition_id,
                            $cash_receiver_id,
                            $total_amount,
                            $receipt_number,
                            $payment_comments,
                            $_SESSION['user_id'],
                            $payment_method,
                            $receiver_user_id,
                            $receiver_external_name,
                            $receiver_external_phone
                        );
                            $stmt->execute();
                            $expense_id = $stmt->insert_id;
                            $stmt->close();

                            // 2) Mark requisition as paid
                            $stmt = $conn->prepare('UPDATE requisition SET status = "paid" WHERE id = ?');
                            $stmt->bind_param('i', $requisition_id);
                            $stmt->execute();
                            if ($stmt->affected_rows === 0) {
                                throw new Exception('Échec du marquage de la réquisition comme payée.');
                            }
                            $stmt->close();

                            // 3) ✅ Province cash: DEBIT & ledger write
                            adjustProvinceCash(
                                $conn,
                                $province_id,
                                $total_amount,
                                'REQUISITION_PAYMENT',
                                'REQUISITION',
                                $requisition_id,
                                intval($_SESSION['user_id']),
                                'Requisition Payment ID ' . $requisition_id . ' (Receipt ' . $receipt_number . ')'
                            );

                            // 4) Log action
                            if (($requisition['recipient_type'] ?? 'internal') === 'external') {
                                $externalName = trim((string)($requisition['recipient_external_name'] ?? ''));
                                $externalPhone = trim((string)($requisition['recipient_external_phone'] ?? ''));
                                $recipientSummary = 'external recipient ' . ($externalName !== '' ? $externalName : 'unknown');
                                if ($externalPhone !== '') {
                                    $recipientSummary .= ' (' . $externalPhone . ')';
                                }
                            } else {
                                $internalName = trim((string)($requisition['recipient_name'] ?? ''));
                                $recipientSummary = 'internal recipient ' . ($internalName !== '' ? $internalName : 'unknown');
                            }
                            log_action(
                                $_SESSION['user_id'],
                                'pay_requisition',
                                'Paid requisition ID ' . $requisition_id . ' for ' . $total_amount . ' BIF with receipt ' . $receipt_number . ' to ' . $recipientSummary
                            );

                            // NOTE: No notifications to accountants when requisition is paid
                            // Accountants perform the action themselves, so no notification needed
                            // Notifications are only for amount adjustments on approval (handled in approve_expenses.php)

                            // 6) PDF receipt (optional)
                            if ($tcpdf_available) {
                                // Fetch shared company info with depot name
                                $company_info = mbh_get_company_info_for_receipt($conn, $province_id);

                                $pdf = new TCPDF();
                                $pdf->SetCreator('Masunzu Bar Hotel');
                                $pdf->SetAuthor($_SESSION['full_name'] ?? 'Accountant');
                                $pdf->SetTitle('Requisition Payment Receipt ' . $receipt_number);
                                $pdf->AddPage();
                                $pdf->SetFont('helvetica', '', 12);

                                $html = '<h1 style="color:#4B2F1F;">Reçu de Paiement de Réquisition</h1>';
                                $html .= '<p><strong>Numéro de Reçu :</strong> ' . htmlspecialchars($receipt_number, ENT_QUOTES, 'UTF-8') . '</p>';
                                $html .= '<p><strong>Date :</strong> ' . date('Y-m-d H:i') . '</p>';

                                $html .= '<h3 style="color:#4B2F1F;">Informations de l\'Entreprise</h3>';
                                $companyNameWithDepot = htmlspecialchars($company_info['company_name'] ?? 'Masunzu Bar Hotel', ENT_QUOTES, 'UTF-8');
                                if (!empty($company_info['depot_name'])) {
                                    $companyNameWithDepot .= ' - ' . htmlspecialchars($company_info['depot_name'], ENT_QUOTES, 'UTF-8');
                                }
                                $html .= '<p><strong>Nom :</strong> ' . $companyNameWithDepot . '</p>';
                                $html .= '<p><strong>Adresse :</strong> ' . htmlspecialchars($company_info['address'] ?? '', ENT_QUOTES, 'UTF-8') . '</p>';
                                $html .= '<p><strong>Téléphone 1 :</strong> ' . htmlspecialchars($company_info['phone1'] ?? '', ENT_QUOTES, 'UTF-8') . '</p>';
                                $html .= '<p><strong>Téléphone 2 :</strong> ' . htmlspecialchars($company_info['phone2'] ?? '', ENT_QUOTES, 'UTF-8') . '</p>';
                                $html .= '<p><strong>Email :</strong> ' . htmlspecialchars($company_info['email'] ?? '', ENT_QUOTES, 'UTF-8') . '</p>';
                                $html .= '<p><strong>NIF :</strong> ' . htmlspecialchars($company_info['nif'] ?? '', ENT_QUOTES, 'UTF-8') . '</p>';
                                $html .= '<p><strong>RC :</strong> ' . htmlspecialchars($company_info['rc'] ?? '', ENT_QUOTES, 'UTF-8') . '</p>';

                                $html .= '<h3 style="color:#4B2F1F;">Détails de la Réquisition</h3>';
                                $html .= '<p><strong>ID Réquisition :</strong> ' . $requisition['id'] . '</p>';
                                $html .= '<p><strong>Initiateur :</strong> ' . htmlspecialchars($requisition['initiator_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8') . '</p>';
                                $html .= '<p><strong>Créé le :</strong> ' . htmlspecialchars($requisition['created_at'], ENT_QUOTES, 'UTF-8') . '</p>';
                                $html .= '<p><strong>Commentaires :</strong> ' . htmlspecialchars($requisition['comments'] ?? 'Aucun', ENT_QUOTES, 'UTF-8') . '</p>';
                                $html .= '<p><strong>Récepteur d\'Espèces :</strong> ' . htmlspecialchars($receiver_row['full_name'], ENT_QUOTES, 'UTF-8') . '</p>';
                                $payment_method_map = ['cash' => 'Espèces', 'bank_transfer' => 'Virement Bancaire', 'mobile' => 'Mobile Money'];
                                $payment_method_text = $payment_method_map[$payment_method] ?? ucfirst($payment_method);
                                $html .= '<p><strong>Méthode de Paiement :</strong> ' . htmlspecialchars($payment_method_text, ENT_QUOTES, 'UTF-8') . '</p>';
                                $html .= '<p><strong>Commentaires de Paiement :</strong> ' . htmlspecialchars($payment_comments, ENT_QUOTES, 'UTF-8') . '</p>';

                                $html .= '<h3 style="color:#4B2F1F;">Articles de la Réquisition</h3>';
                                $html .= '<table border="1" cellpadding="5"><tr style="background-color:#4B2F1F;color:#F4F0E4;"><th>Catégorie</th><th>Description</th><th>Quantité</th><th>Prix Unitaire (BIF)</th><th>Total (BIF)</th></tr>';
                                foreach ($items as $item) {
                                    $item_total = (float)$item['quantity'] * (float)$item['unit_price'];
                                    $html .= '<tr><td>' . htmlspecialchars(ucwords(str_replace('_', ' ', $item['category'])), ENT_QUOTES, 'UTF-8') . '</td><td>' . htmlspecialchars($item['description'], ENT_QUOTES, 'UTF-8') . '</td><td>' . htmlspecialchars($item['quantity'], ENT_QUOTES, 'UTF-8') . '</td><td>' . number_format((float)$item['unit_price'], 2, '.', ',') . '</td><td>' . number_format($item_total, 2, '.', ',') . '</td></tr>';
                                }
                                $html .= '<tr><td colspan="4" style="text-align:right;"><strong>Total Payé</strong></td><td>' . number_format($total_amount, 2, '.', ',') . '</td></tr>';
                                $html .= '</table>';

                                $pdf->writeHTML($html);
                                $conn->commit(); // commit before output
                                $pdf->Output('requisition_receipt_' . $receipt_number . '.pdf', 'D');
                                exit;
                            }

                            // If no PDF or after rendering completes
                            $conn->commit();
                            $success = 'Réquisition payée avec succès. Numéro de reçu : ' . $receipt_number . ($tcpdf_available ? ' (Téléchargement démarré)' : ' (PDF indisponible)');

                        } catch (Exception $e) {
                            $conn->rollback();
                            $errors[] = $e->getMessage();
                        }
                    }
                }
            }
        }
    }

    // =====================
    // List approved items
    // =====================
    $requisitions = [];
    $department_select = $department_exists ? "COALESCE(r.department, 'N/A') AS department" : "'N/A' AS department";
    $requisition_query = "
        SELECT r.id, r.total_amount, r.created_at, r.recipient_id, r.recipient_type,
               r.recipient_external_name, r.recipient_external_phone, r.comments, {$department_select}, 
               r.approved_by, u.full_name AS initiator_name, u2.full_name AS recipient_name, 
               approver.full_name AS approver_name
        FROM requisition r
        LEFT JOIN user u ON r.user_id = u.id
        LEFT JOIN user u2 ON r.recipient_id = u2.id
        LEFT JOIN user approver ON r.approved_by = approver.id
        WHERE r.province_id = ? AND r.status = 'approved'
        ORDER BY r.created_at DESC";
    $stmt = $conn->prepare($requisition_query);
    $stmt->bind_param('i', $province_id);
    $stmt->execute();
    $requisitions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // Requisition items for displayed list
    $requisition_items = [];
    if (!empty($requisitions)) {
        $ids = array_column($requisitions, 'id');
        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        $types = str_repeat('i', count($ids));
        $item_stmt = $conn->prepare('SELECT requisition_id, category, description, quantity, unit_price FROM requisition_item WHERE requisition_id IN (' . $placeholders . ')');
        $item_stmt->bind_param($types, ...$ids);
        $item_stmt->execute();
        $item_results = $item_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $item_stmt->close();
        foreach ($item_results as $row) {
            $requisition_items[$row['requisition_id']][] = $row;
        }
    }

    // Cash receivers list removed - not needed for view-only access

    $conn->close();

} catch (Exception $e) {
    error_log('ERROR in accountant/pay_requisitions.php: ' . $e->getMessage());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur : " . htmlspecialchars($e->getMessage()) . "</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

?>
<div style="padding: 20px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh;">
    <div style="max-width: 1100px; margin: 20px auto; background-color: #F4F0E4; padding: 25px; border-radius: 12px; box-shadow: 0 10px 30px rgba(0,0,0,0.15);">
        <h2 style="color: #4B2F1F; margin-bottom: 10px;">Voir Réquisitions Approuvées</h2>
        <p style="color: #000000; font-size: 15px; margin-bottom: 20px;">Voir les réquisitions approuvées pour votre province. Les paiements doivent être effectués par les caissiers en utilisant leur solde de session.</p>

        <?php if (!empty($success)): ?>
            <div style="background-color: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 15px; border-radius: 8px; margin-bottom: 20px;"><?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?></div>
        <?php endif; ?>

        <?php if (!empty($errors)): ?>
            <div style="background-color: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; border-radius: 8px; margin-bottom: 20px;">
                <?php foreach ($errors as $error): ?>
                    <p style="margin: 0; font-size: 14px;"><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <?php if (empty($requisitions)): ?>
            <p style="color: #4B2F1F; font-size: 16px; text-align: center;">Aucune réquisition approuvée disponible pour paiement.</p>
        <?php else: ?>
            <div style="overflow-x: auto;">
                <table style="width: 100%; border-collapse: collapse;">
                    <thead>
                        <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                            <th style="padding: 12px; text-align: left;">Réquisition</th>
                            <th style="padding: 12px; text-align: left;">Initiateur</th>
                            <th style="padding: 12px; text-align: left;">Destinataire</th>
                            <th style="padding: 12px; text-align: left;">Approbateur</th>
                            <th style="padding: 12px; text-align: left;">Total</th>
                            <th style="padding: 12px; text-align: left;">Créé</th>
                            <th style="padding: 12px; text-align: left;">Articles</th>
                            <th style="padding: 12px; text-align: right;">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($requisitions as $requisition): ?>
                            <?php
                                $items_for_requisition = $requisition_items[$requisition['id']] ?? [];
                                $detail_items = [];
                                foreach ($items_for_requisition as $item) {
                                    $detail_items[] = [
                                        'category'   => $item['category'] ?? '',
                                        'description'=> $item['description'] ?? '',
                                        'quantity'   => $item['quantity'] ?? 0,
                                        'unit_price' => $item['unit_price'] ?? 0,
                                        'total'      => (float)($item['quantity'] ?? 0) * (float)($item['unit_price'] ?? 0)
                                    ];
                                }
                                $recipient_display = ($requisition['recipient_type'] ?? 'internal') === 'external'
                                    ? trim(($requisition['recipient_external_name'] ?? '') . (($requisition['recipient_external_phone'] ?? '') !== '' ? ' (' . $requisition['recipient_external_phone'] . ')' : ''))
                                    : ($requisition['recipient_name'] ?? 'Unknown');
                                $detail_payload = [
                                    'id'           => (int)($requisition['id'] ?? 0),
                                    'initiator'    => $requisition['initiator_name'] ?? 'Unknown',
                                    'recipient'    => $recipient_display,
                                    'recipient_type' => $requisition['recipient_type'] ?? 'internal',
                                    'recipient_external_name' => $requisition['recipient_external_name'] ?? '',
                                    'recipient_external_phone' => $requisition['recipient_external_phone'] ?? '',
                                    'recipient_user_id' => (int)($requisition['recipient_id'] ?? 0),
                                    'approver'     => $requisition['approver_name'] ?? 'Pending',
                                    'department'   => $requisition['department'] ?? 'N/A',
                                    'total_amount' => (float)($requisition['total_amount'] ?? 0),
                                    'created_at'   => $requisition['created_at'] ?? '',
                                    'comments'     => $requisition['comments'] ?? '',
                                    'items'        => $detail_items
                                ];
                                $detail_json = htmlspecialchars(json_encode($detail_payload, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP), ENT_QUOTES, 'UTF-8');
                                $payment_payload = [
                                    'id' => (int)($requisition['id'] ?? 0),
                                    'total_amount' => (float)($requisition['total_amount'] ?? 0),
                                    'recipient_type' => $requisition['recipient_type'] ?? 'internal',
                                    'recipient_id' => (int)($requisition['recipient_id'] ?? 0),
                                    'recipient_display' => $recipient_display,
                                    'recipient_external_name' => $requisition['recipient_external_name'] ?? '',
                                    'recipient_external_phone' => $requisition['recipient_external_phone'] ?? ''
                                ];
                                $payment_json = htmlspecialchars(json_encode($payment_payload, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP), ENT_QUOTES, 'UTF-8');
                            ?>
                            <tr style="border-bottom: 1px solid #E0E0E0; background-color: #FFFFFF;">
                                <td style="padding: 12px;">#<?php echo htmlspecialchars($requisition['id'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px;"><?php echo htmlspecialchars($requisition['initiator_name'] ?? 'Unknown', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px;"><?php echo htmlspecialchars($recipient_display, ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px;"><?php echo htmlspecialchars($requisition['approver_name'] ?? 'Pending', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px; font-weight: bold; color: #4B2F1F;"><?php echo number_format((float)$requisition['total_amount'], 0, '.', ','); ?> BIF</td>
                                <td style="padding: 12px;"><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($requisition['created_at'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px;">
                                    <?php if (!empty($items_for_requisition)): ?>
                                        <ul style="margin: 0; padding-left: 18px; color: #4B2F1F;">
                                            <?php foreach ($items_for_requisition as $item): ?>
                                                <li><?php echo htmlspecialchars(ucwords(str_replace('_', ' ', $item['category'])), ENT_QUOTES, 'UTF-8'); ?> - <?php echo htmlspecialchars($item['description'], ENT_QUOTES, 'UTF-8'); ?> (<?php echo htmlspecialchars($item['quantity'], ENT_QUOTES, 'UTF-8'); ?> x <?php echo number_format((float)$item['unit_price'], 0, '.', ','); ?>)</li>
                                            <?php endforeach; ?>
                                        </ul>
                                    <?php else: ?>
                                        <span style="color: #999;">Aucun article</span>
                                    <?php endif; ?>
                                </td>
                                <td style="padding: 12px; text-align: right;">
                                    <div style="display: inline-flex; gap: 8px;">
                                        <button data-details="<?php echo $detail_json; ?>" class="view-details" style="background: #FFFFFF; color: #4B2F1F; border: 1px solid #4B2F1F; padding: 10px 16px; border-radius: 6px; cursor: pointer;">Voir</button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Payment functionality removed - Accountants can only view requisitions -->

<!-- Detail Modal -->
<div id="detail-modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.55); backdrop-filter: blur(4px); z-index: 1000;">
    <div style="background: #F4F0E4; border-radius: 12px; max-width: 650px; width: 92%; margin: 60px auto; padding: 24px; position: relative; box-shadow: 0 20px 60px rgba(0,0,0,0.35);">
        <button id="close-detail" style="position: absolute; top: 12px; right: 12px; background: none; border: none; font-size: 24px; color: #4B2F1F; cursor: pointer;">×</button>
        <h3 style="color: #4B2F1F; margin-bottom: 18px;">Détails Réquisition</h3>
        <div id="detail-meta" style="display: flex; flex-direction: column; gap: 6px; margin-bottom: 18px; color: #4B2F1F;"></div>
        <div style="max-height: 260px; overflow-y: auto; border: 1px solid rgba(75,47,31,0.15); border-radius: 10px;">
            <table style="width: 100%; border-collapse: collapse;">
                <thead>
                    <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                        <th style="padding: 10px; text-align: left;">Catégorie</th>
                        <th style="padding: 10px; text-align: left;">Description</th>
                        <th style="padding: 10px; text-align: left;">Qté</th>
                        <th style="padding: 10px; text-align: left;">Prix Unitaire (BIF)</th>
                        <th style="padding: 10px; text-align: left;">Total (BIF)</th>
                    </tr>
                </thead>
                <tbody id="detail-items-body">
                    <tr>
                        <td colspan="5" style="padding: 12px; text-align: center; color: #4B2F1F;">Aucun article à afficher.</td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
const detailModal = document.getElementById('detail-modal');
const closeDetail = document.getElementById('close-detail');
const detailMeta = document.getElementById('detail-meta');
const detailItemsBody = document.getElementById('detail-items-body');

document.querySelectorAll('.view-details').forEach(button => {
    button.addEventListener('click', () => {
        const payload = button.getAttribute('data-details');
        try {
            const details = JSON.parse(payload);
            detailMeta.innerHTML = `
                <div><strong>Réquisition:</strong> #${details.id}</div>
                <div><strong>Initiateur:</strong> ${details.initiator}</div>
                <div><strong>Destinataire:</strong> ${details.recipient}</div>
                <div><strong>Approbateur:</strong> ${details.approver}</div>
                <div><strong>Département:</strong> ${details.department}</div>
                <div><strong>Total:</strong> ${Number(details.total_amount).toLocaleString(undefined, { minimumFractionDigits: 0 })} BIF</div>
                <div><strong>Créé:</strong> ${details.created_at}</div>
                <div><strong>Commentaires:</strong> ${details.comments || 'Aucun'}</div>
            `;
            if (Array.isArray(details.items) && details.items.length > 0) {
                detailItemsBody.innerHTML = details.items.map(item => {
                    return `
                        <tr>
                            <td style="padding: 10px;">${(item.category || '').replace(/_/g, ' ')}</td>
                            <td style="padding: 10px;">${item.description || ''}</td>
                            <td style="padding: 10px;">${item.quantity ?? 0}</td>
                            <td style="padding: 10px;">${Number(item.unit_price || 0).toLocaleString(undefined, { minimumFractionDigits: 0 })} BIF</td>
                            <td style="padding: 10px;">${Number(item.total || 0).toLocaleString(undefined, { minimumFractionDigits: 0 })} BIF</td>
                        </tr>
                    `;
                }).join('');
            } else {
                detailItemsBody.innerHTML = '<tr><td colspan="5" style="padding: 12px; text-align: center; color: #4B2F1F;">Aucun article disponible.</td></tr>';
            }
            detailModal.style.display = 'block';
        } catch (error) {
            console.error('Failed to parse requisition details', error);
        }
    });
});

closeModal?.addEventListener('click', () => { modal.style.display = 'none'; });
closeDetail?.addEventListener('click', () => { detailModal.style.display = 'none'; });

modal.addEventListener('click', e => { if (e.target === modal) modal.style.display = 'none'; });
detailModal.addEventListener('click', e => { if (e.target === detailModal) detailModal.style.display = 'none'; });
</script>

<?php include __DIR__ . '/../../includes/footer.php'; ?>
