<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../config/database.php';

require_login();

$allowedRoles = ['Accountant', 'Admin', 'DG', 'DGA', 'Board'];
$roleName = $_SESSION['role_name'] ?? '';
if (!in_array($roleName, $allowedRoles, true)) {
    header('Location: /masunzu_bar_hotel/index.php');
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    http_response_code(500);
    echo "<p style='color:#b71c1c;text-align:center;padding:20px;'>Connexion base de données indisponible.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

$today = date('Y-m-d');
$startDate = ($_GET['start_date'] ?? '') !== '' ? $_GET['start_date'] : $today;
$endDate = ($_GET['end_date'] ?? '') !== '' ? $_GET['end_date'] : $today;
$userFilter = isset($_GET['user_id']) ? (int)$_GET['user_id'] : 0;
$typeFilter = trim($_GET['transaction_type'] ?? '');
$exportCsv = isset($_GET['export']) && $_GET['export'] === 'csv';

$filters = [];
$types = '';
$params = [];

$filters[] = 't.created_at BETWEEN ? AND ?';
$types .= 'ss';
$params[] = $startDate . ' 00:00:00';
$params[] = $endDate . ' 23:59:59';

if ($userFilter > 0) {
    $filters[] = 't.user_id = ?';
    $types .= 'i';
    $params[] = $userFilter;
}

if ($typeFilter !== '') {
    $filters[] = 't.transaction_type = ?';
    $types .= 's';
    $params[] = $typeFilter;
}

$where = $filters ? 'WHERE ' . implode(' AND ', $filters) : '';

// unified view: cashier_account + bank_account_transaction
$sql = "
    SELECT * FROM (
        SELECT
            ca.id AS row_id,
            'cashier_account' AS source,
            ca.transaction_type,
            ca.direction,
            ca.amount,
            ca.reference_type,
            ca.reference_id,
            ca.description,
            ca.created_at,
            ca.created_by AS user_id,
            u.full_name AS user_name,
            r.name AS user_role,
            p.name AS province_name,
            NULL AS bank_account,
            NULL AS slip_number,
            NULL AS slip_file,
            COALESCE(pi.paid_amount, ui.paid_amount, (ui.subtotal_amount + ui.tax_amount)) AS invoice_amount
        FROM cashier_account ca
        LEFT JOIN user u ON ca.created_by = u.id
        LEFT JOIN role r ON u.role_id = r.id
        LEFT JOIN province p ON u.province_id = p.id
        LEFT JOIN paid_invoices pi ON ca.reference_type = 'INVOICE' AND ca.reference_id = pi.id
        LEFT JOIN unpaid_invoices ui ON ca.reference_type = 'INVOICE' AND ca.reference_id = ui.id
        UNION ALL
        SELECT
            bat.id AS row_id,
            'bank_account_transaction' AS source,
            bat.transaction_type,
            bat.direction,
            bat.amount,
            bat.reference_type,
            bat.reference_id,
            bat.description,
            bat.created_at,
            bat.created_by AS user_id,
            u2.full_name AS user_name,
            r2.name AS user_role,
            p2.name AS province_name,
            ba.account_number AS bank_account,
            bd.slip_number AS slip_number,
            bd.slip_file_path AS slip_file,
            NULL AS invoice_amount
        FROM bank_account_transaction bat
        LEFT JOIN user u2 ON bat.created_by = u2.id
        LEFT JOIN role r2 ON u2.role_id = r2.id
        LEFT JOIN province p2 ON u2.province_id = p2.id
        LEFT JOIN bank_account ba ON bat.bank_account_id = ba.id
        LEFT JOIN bank_deposit bd ON bat.reference_type = 'CASH_DEPOSIT' AND bat.reference_id = bd.id
    ) t
    $where
    ORDER BY t.created_at DESC, t.row_id DESC
    LIMIT 1000
";

$stmt = $conn->prepare($sql);
if (!$stmt) {
    error_log("Failed to prepare transactions query: " . $conn->error);
    http_response_code(500);
    echo "<p style='color:#b71c1c;text-align:center;padding:20px;'>Impossible de charger les transactions.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

if ($types !== '') {
    $bind = [];
    foreach ($params as $k => $v) {
        $bind[$k] = &$params[$k];
    }
    array_unshift($bind, $types);
    call_user_func_array([$stmt, 'bind_param'], $bind);
}

if (!$stmt->execute()) {
    error_log("Failed to execute transactions query: " . $stmt->error);
    $stmt->close();
    http_response_code(500);
    echo "<p style='color:#b71c1c;text-align:center;padding:20px;'>Impossible d'exécuter la requête.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

$transactions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Users for filter dropdown
$userOptions = [];
$res = $conn->query("SELECT id, full_name FROM user ORDER BY full_name");
if ($res) {
    $userOptions = $res->fetch_all(MYSQLI_ASSOC);
    $res->close();
}

$conn->close();

if ($exportCsv) {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=transactions_' . date('Ymd_His') . '.csv');
    $out = fopen('php://output', 'w');
    fputcsv($out, ['Date', 'Utilisateur', 'Rôle', 'Province', 'Type', 'Direction', 'Montant', 'Référence', 'Description', 'Compte Banque', 'Slip']);
    foreach ($transactions as $t) {
        $amountDisplay = ($t['transaction_type'] === 'INVOICE_PAYMENT' && (float)$t['amount'] === 0.0)
            ? (float)($t['invoice_amount'] ?? 0)
            : (float)$t['amount'];
        fputcsv($out, [
            $t['created_at'],
            $t['user_name'] ?? 'N/A',
            $t['user_role'] ?? 'N/A',
            $t['province_name'] ?? 'N/A',
            $t['transaction_type'],
            $t['direction'],
            $amountDisplay,
            ($t['reference_type'] ? $t['reference_type'] . ':' . $t['reference_id'] : ''),
            $t['description'],
            $t['bank_account'] ?? '',
            $t['slip_number'] ?? ''
        ]);
    }
    fclose($out);
    exit;
}

include __DIR__ . '/../../includes/header.php';
?>
<style>
.tx-page { padding: 20px; }
.tx-filters { background: #fff; padding: 16px; border-radius: 10px; box-shadow: 0 4px 10px rgba(0,0,0,0.05); margin-bottom: 16px; }
.tx-grid { display: grid; grid-template-columns: repeat(auto-fit,minmax(200px,1fr)); gap: 12px; }
.tx-grid label { font-weight: 600; color: #4B2F1F; margin-bottom: 4px; display: block; }
.tx-grid input, .tx-grid select { width: 100%; padding: 10px; border: 1px solid #D7CCC8; border-radius: 8px; }
.tx-actions { margin-top: 10px; display: flex; gap: 10px; flex-wrap: wrap; }
.tx-actions button, .tx-actions a { background: #4B2F1F; color: #F4F0E4; border: none; border-radius: 8px; padding: 10px 16px; font-weight: 600; text-decoration: none; cursor: pointer; }
.tx-actions a.secondary { background: #B0BEC5; color: #263238; }
.tx-table { width: 100%; border-collapse: collapse; background: #fff; border-radius: 10px; overflow: hidden; box-shadow: 0 4px 14px rgba(0,0,0,0.07); }
.tx-table th, .tx-table td { padding: 10px 12px; text-align: left; font-size: 13px; }
.tx-table th { background: #4B2F1F; color: #F4F0E4; }
.tx-table tr:nth-child(even) { background: #F9F5F0; }
.badge-dir { padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 700; }
.badge-credit { background: #E8F5E9; color: #2E7D32; }
.badge-debit { background: #FFEBEE; color: #C62828; }
.slip-btn { background: #1976D2; color: #fff; padding: 6px 10px; border-radius: 6px; text-decoration: none; font-weight: 600; }
</style>

<div class="tx-page">
    <h2 style="color:#4B2F1F;margin-top:0;">Transactions utilisateurs</h2>
    <div class="tx-filters">
        <form method="get">
            <div class="tx-grid">
                <div>
                    <label for="start_date">Date début</label>
                    <input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($startDate, ENT_QUOTES, 'UTF-8'); ?>">
                </div>
                <div>
                    <label for="end_date">Date fin</label>
                    <input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($endDate, ENT_QUOTES, 'UTF-8'); ?>">
                </div>
                <div>
                    <label for="user_id">Utilisateur</label>
                    <select id="user_id" name="user_id">
                        <option value="0">Tous</option>
                        <?php foreach ($userOptions as $u): ?>
                            <option value="<?php echo (int)$u['id']; ?>" <?php echo $userFilter === (int)$u['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($u['full_name'], ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div>
                    <label for="transaction_type">Type</label>
                    <select id="transaction_type" name="transaction_type">
                        <option value="">Tous</option>
                        <?php
                        $typesList = [
                            'SHIFT_START','SHIFT_END','INVOICE_PAYMENT','LOAN_REPAYMENT','REQUISITION_PAYMENT',
                            'EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','ADJUSTMENT','CASH_TRANSFER_RECEIVED',
                            'BANK_DEPOSIT','REVERSAL'
                        ];
                        foreach ($typesList as $opt) {
                            $sel = $typeFilter === $opt ? 'selected' : '';
                            echo '<option value="' . htmlspecialchars($opt, ENT_QUOTES, 'UTF-8') . "\" {$sel}>" . htmlspecialchars($opt, ENT_QUOTES, 'UTF-8') . '</option>';
                        }
                        ?>
                    </select>
                </div>
            </div>
            <div class="tx-actions">
                <button type="submit">Filtrer</button>
                <a class="secondary" href="/masunzu_bar_hotel/modules/accounting/transactions_overview.php">Réinitialiser</a>
                <a href="<?php echo htmlspecialchars('/masunzu_bar_hotel/modules/accounting/transactions_overview.php?' . http_build_query(array_merge($_GET, ['export' => 'csv'])), ENT_QUOTES, 'UTF-8'); ?>" class="secondary">Exporter CSV</a>
            </div>
        </form>
    </div>

    <div style="overflow-x:auto;">
        <table class="tx-table">
            <thead>
                <tr>
                    <th>Date</th>
                    <th>Utilisateur</th>
                    <th>Rôle</th>
                    <th>Province</th>
                    <th>Type</th>
                    <th>Direction</th>
                    <th>Montant</th>
                    <th>Référence</th>
                    <th>Description</th>
                    <th>Slip</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($transactions)): ?>
                    <tr><td colspan="10" style="text-align:center;padding:16px;">Aucune transaction.</td></tr>
                <?php else: ?>
                    <?php foreach ($transactions as $t): ?>
                        <?php
                            $displayAmount = ($t['transaction_type'] === 'INVOICE_PAYMENT' && (float)($t['amount'] ?? 0) === 0.0)
                                ? (float)($t['invoice_amount'] ?? 0)
                                : (float)($t['amount'] ?? 0);
                        ?>
                        <tr>
                            <td><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($t['created_at'])), ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($t['user_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($t['user_role'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($t['province_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($t['transaction_type'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td>
                                <?php $dir = strtoupper($t['direction'] ?? ''); ?>
                                <span class="badge-dir <?php echo $dir === 'CREDIT' ? 'badge-credit' : 'badge-debit'; ?>">
                                    <?php echo htmlspecialchars($dir, ENT_QUOTES, 'UTF-8'); ?>
                                </span>
                            </td>
                            <td><?php echo number_format($displayAmount, 0, ',', '.'); ?></td>
                            <td><?php echo htmlspecialchars($t['reference_type'] ? $t['reference_type'] . ':' . $t['reference_id'] : '', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($t['description'] ?? '', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td>
                                <?php if (!empty($t['slip_file'])): ?>
                                    <a class="slip-btn" href="<?php echo htmlspecialchars($t['slip_file'], ENT_QUOTES, 'UTF-8'); ?>" target="_blank">Voir slip</a>
                                <?php elseif (!empty($t['slip_number'])): ?>
                                    <?php echo htmlspecialchars($t['slip_number'], ENT_QUOTES, 'UTF-8'); ?>
                                <?php else: ?>
                                    —
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<?php include __DIR__ . '/../../includes/footer.php'; ?>
