<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/export_helpers.php';

require_login();

$allowedRoles = ['Accountant', 'Admin', 'DG', 'DGA', 'Board'];
$roleName = $_SESSION['role_name'] ?? '';
if (!in_array($roleName, $allowedRoles, true)) {
    header("Location: /masunzu_bar_hotel/dashboards/unauthorized.php");
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in view_bank_deposits.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Handle CSV export
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    $startDateInput = $_GET['start_date'] ?? date('Y-m-01');
    $endDateInput = $_GET['end_date'] ?? date('Y-m-d');
    
    $startDate = DateTimeImmutable::createFromFormat('Y-m-d', $startDateInput) ?: new DateTimeImmutable('today');
    $endDate = DateTimeImmutable::createFromFormat('Y-m-d', $endDateInput) ?: new DateTimeImmutable('today');
    if ($startDate > $endDate) {
        [$startDate, $endDate] = [$endDate, $startDate];
    }
    
    $rangeStart = $startDate->format('Y-m-d 00:00:00');
    $rangeEnd = $endDate->format('Y-m-d 23:59:59');
    
    // Query to get deposits with all required information
    $exportQuery = "
        SELECT 
            bd.id,
            bd.deposit_date,
            bd.amount,
            bd.slip_number,
            bd.slip_file_path,
            u.full_name AS depositor_name,
            p.name AS depot_name,
            r.name AS region_name,
            ba.account_number,
            ba.currency
        FROM bank_deposit bd
        LEFT JOIN user u ON bd.user_id = u.id
        LEFT JOIN province p ON bd.province_id = p.id
        LEFT JOIN region r ON p.region_id = r.id
        LEFT JOIN bank_account ba ON bd.bank_account_id = ba.id
        WHERE bd.deposit_date BETWEEN ? AND ?
        ORDER BY bd.deposit_date DESC, bd.id DESC
    ";
    
    $stmt = $conn->prepare($exportQuery);
    if ($stmt) {
        $stmt->bind_param('ss', $rangeStart, $rangeEnd);
        $stmt->execute();
        $deposits = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
        
        // Calculate balance before and after for each deposit
        $exportData = [];
        foreach ($deposits as $deposit) {
            $depositId = (int)$deposit['id'];
            $depositDate = $deposit['deposit_date'];
            $amount = (float)$deposit['amount'];
            
            // Get balance before (sum of all transactions before this deposit)
            $beforeQuery = "
                SELECT COALESCE(SUM(CASE WHEN direction = 'CREDIT' THEN amount ELSE -amount END), 0) AS balance
                FROM bank_account_transaction
                WHERE bank_account_id = (SELECT bank_account_id FROM bank_deposit WHERE id = ?)
                AND created_at < ?
            ";
            $beforeStmt = $conn->prepare($beforeQuery);
            $balanceBefore = 0;
            if ($beforeStmt) {
                $beforeStmt->bind_param('is', $depositId, $depositDate);
                $beforeStmt->execute();
                $beforeResult = $beforeStmt->get_result()->fetch_assoc();
                $balanceBefore = (float)($beforeResult['balance'] ?? 0);
                $beforeStmt->close();
            }
            
            $balanceAfter = $balanceBefore + $amount;
            
            $exportData[] = [
                date('Y-m-d H:i', strtotime($depositDate)),
                $deposit['depositor_name'] ?? 'N/A',
                $deposit['depot_name'] ?? 'N/A',
                $deposit['region_name'] ?? 'N/A',
                number_format($amount, 2, '.', ''),
                number_format($balanceBefore, 2, '.', ''),
                number_format($balanceAfter, 2, '.', ''),
                $deposit['slip_number'] ?? 'N/A',
                $deposit['account_number'] ?? 'N/A',
                $deposit['currency'] ?? 'BIF'
            ];
        }
        
        $headers = [
            'Date',
            'Déposant (Nom)',
            'Dépôt',
            'Région',
            'Montant (BIF)',
            'Solde Avant',
            'Solde Après',
            'Numéro Bordereau',
            'Compte Bancaire',
            'Devise'
        ];
        
        send_csv_headers('depots_bancaires_' . $startDate->format('Ymd') . '_' . $endDate->format('Ymd') . '.csv');
        $output = fopen('php://output', 'w');
        fputcsv($output, $headers);
        foreach ($exportData as $row) {
            fputcsv($output, $row);
        }
        fclose($output);
        exit;
    }
}

// Get date filters
$startDateInput = $_GET['start_date'] ?? date('Y-m-01');
$endDateInput = $_GET['end_date'] ?? date('Y-m-d');

$startDate = DateTimeImmutable::createFromFormat('Y-m-d', $startDateInput) ?: new DateTimeImmutable('today');
$endDate = DateTimeImmutable::createFromFormat('Y-m-d', $endDateInput) ?: new DateTimeImmutable('today');
if ($startDate > $endDate) {
    [$startDate, $endDate] = [$endDate, $startDate];
}

$rangeStart = $startDate->format('Y-m-d 00:00:00');
$rangeEnd = $endDate->format('Y-m-d 23:59:59');

// Query to get deposits with all required information
$query = "
    SELECT 
        bd.id,
        bd.deposit_date,
        bd.amount,
        bd.slip_number,
        bd.slip_file_path,
        bd.bank_account_id,
        u.full_name AS depositor_name,
        p.name AS depot_name,
        r.name AS region_name,
        ba.account_number,
        ba.currency
    FROM bank_deposit bd
    LEFT JOIN user u ON bd.user_id = u.id
    LEFT JOIN province p ON bd.province_id = p.id
    LEFT JOIN region r ON p.region_id = r.id
    LEFT JOIN bank_account ba ON bd.bank_account_id = ba.id
    WHERE bd.deposit_date BETWEEN ? AND ?
    ORDER BY bd.deposit_date DESC, bd.id DESC
    LIMIT 1000
";

$stmt = $conn->prepare($query);
if (!$stmt) {
    error_log("Failed to prepare deposits query: " . $conn->error);
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

$stmt->bind_param('ss', $rangeStart, $rangeEnd);
$stmt->execute();
$deposits = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Calculate balance before and after for each deposit
$depositsWithBalance = [];
foreach ($deposits as $deposit) {
    $depositId = (int)$deposit['id'];
    $depositDate = $deposit['deposit_date'];
    $amount = (float)$deposit['amount'];
    $bankAccountId = (int)($deposit['bank_account_id'] ?? 0);
    
    // Get balance before (sum of all transactions before this deposit)
    $balanceBefore = 0;
    if ($bankAccountId > 0) {
        $beforeQuery = "
            SELECT COALESCE(SUM(CASE WHEN direction = 'CREDIT' THEN amount ELSE -amount END), 0) AS balance
            FROM bank_account_transaction
            WHERE bank_account_id = ?
            AND created_at < ?
        ";
        $beforeStmt = $conn->prepare($beforeQuery);
        if ($beforeStmt) {
            $beforeStmt->bind_param('is', $bankAccountId, $depositDate);
            $beforeStmt->execute();
            $beforeResult = $beforeStmt->get_result()->fetch_assoc();
            $balanceBefore = (float)($beforeResult['balance'] ?? 0);
            $beforeStmt->close();
        }
    }
    
    $balanceAfter = $balanceBefore + $amount;
    
    $deposit['balance_before'] = $balanceBefore;
    $deposit['balance_after'] = $balanceAfter;
    $depositsWithBalance[] = $deposit;
}

$totalAmount = 0;
foreach ($depositsWithBalance as $dep) {
    $totalAmount += (float)($dep['amount'] ?? 0);
}

include __DIR__ . '/../../includes/header.php';
?>

<style>
.view-deposits-page {
    max-width: 1600px;
    margin: 0 auto;
    padding: 25px 20px 60px;
    background: #F4F0E4;
    min-height: calc(100vh - 70px);
}

.page-header {
    display: flex;
    flex-wrap: wrap;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 25px;
}

.page-header h2 {
    color: #4B2F1F;
    margin: 0;
    font-size: 28px;
}

.filters-card {
    background-color: #FFFFFF;
    border-radius: 12px;
    padding: 20px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.08);
    margin-bottom: 20px;
}

.filters-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
    margin-bottom: 15px;
}

.filters-grid label {
    display: block;
    font-weight: 600;
    color: #4B2F1F;
    margin-bottom: 6px;
    font-size: 14px;
}

.filters-grid input {
    width: 100%;
    padding: 10px;
    border-radius: 8px;
    border: 1px solid #D7CCC8;
    font-size: 14px;
    color: #4B2F1F;
    box-sizing: border-box;
}

.filters-actions {
    display: flex;
    gap: 10px;
    flex-wrap: wrap;
}

.filters-actions button {
    background-color: #4B2F1F;
    color: #F5F5F5;
    border: none;
    border-radius: 8px;
    padding: 10px 20px;
    font-weight: 600;
    cursor: pointer;
    font-size: 14px;
}

.filters-actions button.export-btn {
    background-color: #2E7D32;
}

.filters-actions button:hover {
    opacity: 0.9;
}

.summary-card {
    background: linear-gradient(135deg, #4CAF50 0%, #2E7D32 100%);
    border-radius: 12px;
    padding: 18px;
    color: #FFFFFF;
    box-shadow: 0 6px 18px rgba(76, 175, 80, 0.25);
    margin-bottom: 20px;
}

.summary-card p {
    margin: 0;
    opacity: 0.9;
    font-size: 14px;
}

.summary-card h3 {
    margin: 10px 0 0;
    font-size: 26px;
    font-weight: 700;
}

.deposits-table-wrapper {
    background-color: #FFFFFF;
    border-radius: 12px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    overflow: hidden;
}

.deposits-table-wrapper table {
    width: 100%;
    border-collapse: collapse;
}

.deposits-table-wrapper th,
.deposits-table-wrapper td {
    padding: 12px 14px;
    text-align: left;
    font-size: 14px;
    border-bottom: 1px solid #E0E0E0;
}

.deposits-table-wrapper th {
    background-color: #4B2F1F;
    color: #F4F0E4;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    font-weight: 600;
    position: sticky;
    top: 0;
    z-index: 10;
}

.deposits-table-wrapper tr:nth-child(even) {
    background-color: #F9F5F0;
}

.deposits-table-wrapper tr:hover {
    background-color: #FFF3E0;
}

.badge-depot {
    background-color: #FF7043;
    color: #FFFFFF;
    padding: 3px 8px;
    border-radius: 999px;
    font-size: 12px;
    display: inline-block;
}

.badge-region {
    background-color: #42A5F5;
    color: #FFFFFF;
    padding: 3px 8px;
    border-radius: 999px;
    font-size: 12px;
    display: inline-block;
}

.slip-view-btn {
    background: linear-gradient(135deg, #4B2F1F, #6B4F2F);
    color: #FFFFFF;
    border: none;
    border-radius: 6px;
    padding: 6px 14px;
    cursor: pointer;
    font-size: 13px;
    font-weight: 600;
    margin-right: 8px;
}

.slip-view-btn:hover {
    opacity: 0.9;
}

.slip-modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.65);
    display: none;
    align-items: center;
    justify-content: center;
    z-index: 9999;
}

.slip-modal.active {
    display: flex;
}

.slip-modal-content {
    width: 90%;
    max-width: 900px;
    background: #fff;
    border-radius: 12px;
    padding: 20px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.25);
    display: flex;
    flex-direction: column;
    gap: 16px;
    max-height: 90vh;
}

.slip-modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-bottom: 1px solid #eee;
    padding-bottom: 8px;
}

.slip-modal-header h4 {
    margin: 0;
    color: #4B2F1F;
}

.slip-modal-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: #4B2F1F;
}

.slip-preview-frame,
.slip-preview-image {
    width: 100%;
    border: none;
    border-radius: 10px;
}

.slip-preview-frame {
    height: 70vh;
}

.slip-preview-image {
    max-height: 70vh;
    object-fit: contain;
}

.slip-download-link {
    align-self: flex-end;
    display: inline-flex;
    align-items: center;
    gap: 8px;
    font-weight: 600;
    color: #1976D2;
    text-decoration: none;
}

@media (max-width: 768px) {
    .filters-grid {
        grid-template-columns: 1fr;
    }
    .deposits-table-wrapper {
        overflow-x: auto;
    }
}
</style>

<div class="view-deposits-page">
    <div class="page-header">
        <h2>Dépôts Bancaires - Transactions</h2>
    </div>

    <div class="filters-card">
        <form method="get" id="filtersForm">
            <div class="filters-grid">
                <div>
                    <label for="start_date">Date Début</label>
                    <input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($startDate->format('Y-m-d'), ENT_QUOTES, 'UTF-8'); ?>">
                </div>
                <div>
                    <label for="end_date">Date Fin</label>
                    <input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($endDate->format('Y-m-d'), ENT_QUOTES, 'UTF-8'); ?>">
                </div>
            </div>
            <div class="filters-actions">
                <button type="submit">Filtrer</button>
                <button type="submit" name="export" value="csv" class="export-btn">Exporter CSV</button>
                <a href="/masunzu_bar_hotel/modules/accounting/view_bank_deposits.php" style="background-color: #B0BEC5; color: #263238; padding: 10px 20px; border-radius: 8px; text-decoration: none; font-weight: 600; display: inline-block;">Réinitialiser</a>
            </div>
        </form>
    </div>

    <div class="summary-card">
        <p>Total des dépôts sur la période</p>
        <h3><?php echo number_format($totalAmount, 0, ',', '.'); ?> BIF</h3>
        <p style="margin-top: 8px;"><?php echo count($depositsWithBalance); ?> dépôt(s) enregistré(s)</p>
    </div>

    <div class="deposits-table-wrapper">
        <table>
            <thead>
                <tr>
                    <th>Date</th>
                    <th>Déposant</th>
                    <th>Dépôt</th>
                    <th>Région</th>
                    <th>Montant (BIF)</th>
                    <th>Solde Avant</th>
                    <th>Solde Après</th>
                    <th>Bordereau</th>
                    <th>Compte Bancaire</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($depositsWithBalance)): ?>
                    <tr>
                        <td colspan="9" style="text-align:center;color:#6d4c41;padding:30px;">Aucun dépôt trouvé pour cette période.</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($depositsWithBalance as $deposit): ?>
                        <tr>
                            <td><?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($deposit['deposit_date'] ?? 'now')), ENT_QUOTES, 'UTF-8'); ?></td>
                            <td><?php echo htmlspecialchars($deposit['depositor_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                            <td>
                                <span class="badge-depot"><?php echo htmlspecialchars($deposit['depot_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></span>
                            </td>
                            <td>
                                <span class="badge-region"><?php echo htmlspecialchars($deposit['region_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></span>
                            </td>
                            <td style="font-weight: 600; color: #2E7D32;"><?php echo number_format((float)($deposit['amount'] ?? 0), 0, ',', '.'); ?> BIF</td>
                            <td><?php echo number_format((float)($deposit['balance_before'] ?? 0), 0, ',', '.'); ?> BIF</td>
                            <td style="font-weight: 600; color: #1E88E5;"><?php echo number_format((float)($deposit['balance_after'] ?? 0), 0, ',', '.'); ?> BIF</td>
                            <td>
                                <?php if (!empty($deposit['slip_file_path'])): ?>
                                    <button type="button"
                                            class="slip-view-btn"
                                            data-slip-url="<?php echo htmlspecialchars($deposit['slip_file_path'], ENT_QUOTES, 'UTF-8'); ?>"
                                            data-slip-label="<?php echo htmlspecialchars($deposit['slip_number'] ?? 'Bordereau #' . $deposit['id'], ENT_QUOTES, 'UTF-8'); ?>">
                                        Voir Bordereau
                                    </button>
                                    <br>
                                    <small style="color: #666; margin-top: 4px; display: block;"><?php echo htmlspecialchars($deposit['slip_number'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></small>
                                <?php else: ?>
                                    <?php echo htmlspecialchars($deposit['slip_number'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php echo htmlspecialchars($deposit['account_number'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?>
                                <?php if (!empty($deposit['currency'])): ?>
                                    <br><small style="color: #666;"><?php echo htmlspecialchars($deposit['currency'], ENT_QUOTES, 'UTF-8'); ?></small>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Bankslip Modal -->
<div id="slipModal" class="slip-modal">
    <div class="slip-modal-content">
        <div class="slip-modal-header">
            <h4 id="slipModalTitle">Bordereau</h4>
            <button type="button" class="slip-modal-close" aria-label="Fermer">&times;</button>
        </div>
        <iframe class="slip-preview-frame" src="" title="Prévisualisation bordereau" hidden></iframe>
        <img class="slip-preview-image" src="" alt="Prévisualisation bordereau" hidden>
        <a class="slip-download-link" href="#" target="_blank" rel="noopener">⬇ Télécharger le fichier</a>
    </div>
</div>

<script>
(function() {
    const modal = document.getElementById('slipModal');
    if (!modal) {
        return;
    }
    const modalTitle = document.getElementById('slipModalTitle');
    const frame = modal.querySelector('.slip-preview-frame');
    const image = modal.querySelector('.slip-preview-image');
    const downloadLink = modal.querySelector('.slip-download-link');
    const closeBtn = modal.querySelector('.slip-modal-close');

    function closeModal() {
        modal.classList.remove('active');
        frame.src = '';
        image.src = '';
        frame.hidden = true;
        image.hidden = true;
    }

    document.querySelectorAll('.slip-view-btn').forEach((btn) => {
        btn.addEventListener('click', () => {
            const url = btn.getAttribute('data-slip-url');
            if (!url) {
                return;
            }
            const label = btn.getAttribute('data-slip-label') || 'Bordereau';
            modalTitle.textContent = label;
            downloadLink.href = url;

            const extension = (url.split('.').pop() || '').toLowerCase();
            if (extension === 'pdf') {
                frame.src = url;
                frame.hidden = false;
                image.hidden = true;
            } else {
                image.src = url;
                image.hidden = false;
                frame.hidden = true;
            }

            modal.classList.add('active');
        });
    });

    closeBtn.addEventListener('click', closeModal);
    modal.addEventListener('click', (event) => {
        if (event.target === modal) {
            closeModal();
        }
    });
    document.addEventListener('keyup', (event) => {
        if (event.key === 'Escape' && modal.classList.contains('active')) {
            closeModal();
        }
    });
})();
</script>

<?php include __DIR__ . '/../../includes/footer.php'; ?>
<?php $conn->close(); ?>







