<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

$allowedRoles = ['Admin', 'DG', 'DGA', 'Board', 'Accountant'];
$roleName = $_SESSION['role_name'] ?? '';
if (!in_array($roleName, $allowedRoles, true)) {
    header("Location: /masunzu_bar_hotel/dashboards/admin_dashboard.php");
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in view_transactions.php: " . mysqli_connect_error());
    include __DIR__ . '/../../includes/header.php';
    echo "<div style='padding: 40px; text-align: center;'><p style='color: #C62828;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p></div>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

$startDateInput = $_GET['start_date'] ?? date('Y-m-d');
$endDateInput = $_GET['end_date'] ?? date('Y-m-d');
$searchTerm = trim($_GET['search'] ?? '');
$typeFilter = $_GET['type'] ?? '';

$startDateObj = DateTimeImmutable::createFromFormat('Y-m-d', $startDateInput) ?: new DateTimeImmutable('today');
$endDateObj = DateTimeImmutable::createFromFormat('Y-m-d', $endDateInput) ?: new DateTimeImmutable('today');
if ($startDateObj > $endDateObj) {
    [$startDateObj, $endDateObj] = [$endDateObj, $startDateObj];
}
$rangeStart = $startDateObj->setTime(0, 0, 0)->format('Y-m-d H:i:s');
$rangeEnd = $endDateObj->setTime(23, 59, 59)->format('Y-m-d H:i:s');

$query = "
    SELECT data.* FROM (
        SELECT ct.transfer_date AS created_at,
               ct.amount,
               'Transfert Reçu' AS transaction_type,
               ct.id AS reference,
               CONCAT('Transfert d''espèces #', ct.id) AS description,
               COALESCE(cashier.full_name, 'Caissier inconnu') AS user_name,
               'Caissier' AS user_role
        FROM cash_transfer ct
        LEFT JOIN user cashier ON ct.cashier_id = cashier.id
        WHERE ct.status = 'received'
        UNION ALL
        SELECT bd.deposit_date AS created_at,
               bd.amount,
               'Dépôt Bancaire' AS transaction_type,
               bd.slip_number AS reference,
               CONCAT('Bordereau ', COALESCE(bd.slip_number, 'N/A')) AS description,
               COALESCE(u.full_name, 'Comptable inconnu') AS user_name,
               'Comptable' AS user_role
        FROM bank_deposit bd
        LEFT JOIN user u ON bd.user_id = u.id
        UNION ALL
        SELECT e.created_at AS created_at,
               e.amount,
               'Dépense' AS transaction_type,
               e.receipt_number AS reference,
               CONCAT('Réquisition #', COALESCE(e.requisition_id, 'N/A')) AS description,
               COALESCE(acc.full_name, 'Comptable inconnu') AS user_name,
               'Comptable' AS user_role
        FROM expense e
        LEFT JOIN user acc ON e.accountant_id = acc.id
        UNION ALL
        SELECT lp.payment_date AS created_at,
               lp.payment_amount AS amount,
               'Remboursement Crédit' AS transaction_type,
               lp.invoice_id AS reference,
               CONCAT('Paiement crédit (commande #', COALESCE(lp.invoice_id, 'N/A'), ')') AS description,
               COALESCE(u.full_name, 'Caissier inconnu') AS user_name,
               'Caissier' AS user_role
        FROM loan_payment lp
        LEFT JOIN user u ON lp.created_by = u.id
        UNION ALL
        SELECT pi.paid_at AS created_at,
               pi.paid_amount AS amount,
               'Vente à crédit' AS transaction_type,
               pi.id AS reference,
               CONCAT('Vente à crédit : ', pi.invoice_number) AS description,
               COALESCE(u.full_name, 'Caissier inconnu') AS user_name,
               'Caissier' AS user_role
        FROM paid_invoices pi
        LEFT JOIN user u ON pi.created_by = u.id
        WHERE pi.payment_details LIKE '%Type Paiement: Crédit%'
    ) data
    WHERE data.created_at BETWEEN ? AND ?
";
$types = 'ss';
$params = [$rangeStart, $rangeEnd];

if ($searchTerm !== '') {
    $searchWildcard = "%$searchTerm%";
    $query .= " AND (
        data.user_name LIKE ?
        OR data.description LIKE ?
        OR data.transaction_type LIKE ?
        OR data.reference LIKE ?
    )";
    $types .= 'ssss';
    $params = array_merge($params, [$searchWildcard, $searchWildcard, $searchWildcard, $searchWildcard]);
}

if ($typeFilter !== '') {
    $query .= " AND data.transaction_type = ?";
    $types .= 's';
    $params[] = $typeFilter;
}

$query .= " ORDER BY data.created_at DESC";
$stmt = $conn->prepare($query);
if (!$stmt) {
    error_log("Failed to prepare transactions query: " . $conn->error);
    include __DIR__ . '/../../includes/header.php';
    echo "<div style='padding: 40px; text-align: center;'><p style='color: #C62828;'>Erreur lors du chargement des transactions.</p></div>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}
$stmt->bind_param($types, ...$params);
$stmt->execute();
$transactions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();
$conn->close();

// Calculate totals
$totalAmount = array_sum(array_column($transactions, 'amount'));

// Handle CSV Export
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="transactions_' . $startDateObj->format('Y-m-d') . '_' . $endDateObj->format('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    // Add BOM for Excel UTF-8 compatibility
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // Header row
    fputcsv($output, ['Date', 'Utilisateur', 'Rôle', 'Type', 'Référence', 'Description', 'Montant (BIF)'], ';');
    
    // Data rows
    foreach ($transactions as $transaction) {
        fputcsv($output, [
            date('Y-m-d H:i:s', strtotime($transaction['created_at'])),
            $transaction['user_name'] ?? 'N/A',
            $transaction['user_role'] ?? 'N/A',
            $transaction['transaction_type'] ?? 'N/A',
            $transaction['reference'] ?? 'N/A',
            $transaction['description'] ?? 'N/A',
            number_format((float)($transaction['amount'] ?? 0), 0, '', '')
        ], ';');
    }
    
    // Total row
    fputcsv($output, ['', '', '', '', '', 'TOTAL', number_format($totalAmount, 0, '', '')], ';');
    
    fclose($output);
    exit;
}

// Transaction types for filter
$transactionTypes = [
    'Transfert Reçu',
    'Dépôt Bancaire',
    'Dépense',
    'Remboursement Crédit',
    'Vente à crédit'
];

include __DIR__ . '/../../includes/header.php';
?>
<div style="padding: 20px; background: #F4F0E4; min-height: 70vh;">
    <!-- Header -->
    <div style="display: flex; justify-content: space-between; align-items: flex-start; flex-wrap: wrap; gap: 16px; margin-bottom: 20px;">
        <div>
            <h1 style="margin: 0 0 8px; color: #4B2F1F; font-size: 28px; font-weight: 700;">Transactions globales</h1>
            <p style="margin: 0; color: #6B4F2F; font-size: 14px;">Toutes les opérations des caissiers et comptables, triées du plus récent au plus ancien.</p>
        </div>
        <div style="display: flex; gap: 12px; align-items: center;">
            <div style="background: linear-gradient(135deg, #4B2F1F 0%, #6B4F2F 100%); color: #F4F0E4; padding: 12px 20px; border-radius: 10px; text-align: center;">
                <p style="margin: 0; font-size: 11px; text-transform: uppercase; letter-spacing: 0.05em; opacity: 0.8;">Total Montant</p>
                <p style="margin: 4px 0 0; font-size: 20px; font-weight: 700;"><?php echo number_format($totalAmount, 0, ',', '.'); ?> BIF</p>
            </div>
            <div style="background: #fff; border: 2px solid #4B2F1F; color: #4B2F1F; padding: 12px 20px; border-radius: 10px; text-align: center;">
                <p style="margin: 0; font-size: 11px; text-transform: uppercase; letter-spacing: 0.05em; opacity: 0.7;">Transactions</p>
                <p style="margin: 4px 0 0; font-size: 20px; font-weight: 700;"><?php echo number_format(count($transactions)); ?></p>
            </div>
        </div>
    </div>

    <!-- Filters -->
    <form method="GET" style="display: grid; grid-template-columns: repeat(6, 1fr); gap: 12px; background: #fff; padding: 20px; border-radius: 10px; box-shadow: 0 2px 8px rgba(0,0,0,0.06); margin-bottom: 20px;">
        <div>
            <label style="display: block; font-size: 12px; font-weight: 600; color: #4B2F1F; margin-bottom: 5px;">Date début</label>
            <input type="date" name="start_date" value="<?php echo htmlspecialchars($startDateObj->format('Y-m-d'), ENT_QUOTES, 'UTF-8'); ?>" 
                   style="width: 100%; padding: 10px; border: 1px solid #B08968; border-radius: 6px; font-size: 14px; box-sizing: border-box;">
        </div>
        <div>
            <label style="display: block; font-size: 12px; font-weight: 600; color: #4B2F1F; margin-bottom: 5px;">Date fin</label>
            <input type="date" name="end_date" value="<?php echo htmlspecialchars($endDateObj->format('Y-m-d'), ENT_QUOTES, 'UTF-8'); ?>" 
                   style="width: 100%; padding: 10px; border: 1px solid #B08968; border-radius: 6px; font-size: 14px; box-sizing: border-box;">
        </div>
        <div>
            <label style="display: block; font-size: 12px; font-weight: 600; color: #4B2F1F; margin-bottom: 5px;">Type</label>
            <select name="type" style="width: 100%; padding: 10px; border: 1px solid #B08968; border-radius: 6px; font-size: 14px; box-sizing: border-box;">
                <option value="">Tous les types</option>
                <?php foreach ($transactionTypes as $type): ?>
                    <option value="<?php echo htmlspecialchars($type, ENT_QUOTES, 'UTF-8'); ?>" <?php echo $typeFilter === $type ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars($type, ENT_QUOTES, 'UTF-8'); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div>
            <label style="display: block; font-size: 12px; font-weight: 600; color: #4B2F1F; margin-bottom: 5px;">Recherche</label>
            <input type="search" name="search" placeholder="Nom, description, référence..." value="<?php echo htmlspecialchars($searchTerm, ENT_QUOTES, 'UTF-8'); ?>" 
                   style="width: 100%; padding: 10px; border: 1px solid #B08968; border-radius: 6px; font-size: 14px; box-sizing: border-box;">
        </div>
        <div style="display: flex; align-items: flex-end; gap: 8px;">
            <button type="submit" style="flex: 1; background: #4B2F1F; color: #FFF; border: none; border-radius: 6px; padding: 11px; font-weight: 600; cursor: pointer; font-size: 14px;">
                Filtrer
            </button>
        </div>
        <div style="display: flex; align-items: flex-end; gap: 8px;">
            <a href="?start_date=<?php echo htmlspecialchars($startDateObj->format('Y-m-d'), ENT_QUOTES, 'UTF-8'); ?>&end_date=<?php echo htmlspecialchars($endDateObj->format('Y-m-d'), ENT_QUOTES, 'UTF-8'); ?>&search=<?php echo htmlspecialchars($searchTerm, ENT_QUOTES, 'UTF-8'); ?>&type=<?php echo htmlspecialchars($typeFilter, ENT_QUOTES, 'UTF-8'); ?>&export=csv" 
               style="flex: 1; display: flex; align-items: center; justify-content: center; gap: 6px; background: #2E7D32; color: #FFF; border: none; border-radius: 6px; padding: 11px; font-weight: 600; text-decoration: none; font-size: 14px;">
                <span>📥</span> Exporter CSV
            </a>
        </div>
    </form>

    <!-- Table -->
    <div style="background: #FFF; border-radius: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.06); overflow: hidden;">
        <div style="padding: 16px 20px; border-bottom: 1px solid #F0F0F0; display: flex; justify-content: space-between; align-items: center; background: #FDFBF7;">
            <span style="font-weight: 600; color: #4B2F1F;">
                <?php echo number_format(count($transactions)); ?> transaction<?php echo count($transactions) === 1 ? '' : 's'; ?> trouvée<?php echo count($transactions) === 1 ? '' : 's'; ?>
            </span>
            <span style="color: #6B4F2F; font-size: 13px;">
                Période: <?php echo htmlspecialchars($startDateObj->format('d/m/Y'), ENT_QUOTES, 'UTF-8'); ?> - <?php echo htmlspecialchars($endDateObj->format('d/m/Y'), ENT_QUOTES, 'UTF-8'); ?>
            </span>
        </div>
        <div style="overflow-x: auto;">
            <table style="width: 100%; border-collapse: collapse; font-size: 14px; min-width: 900px;">
                <thead>
                    <tr style="background: #4B2F1F; color: #FFF;">
                        <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Date</th>
                        <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Utilisateur</th>
                        <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Rôle</th>
                        <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Type</th>
                        <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Référence</th>
                        <th style="padding: 14px 16px; text-align: left; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Description</th>
                        <th style="padding: 14px 16px; text-align: right; font-weight: 600; font-size: 12px; text-transform: uppercase; letter-spacing: 0.03em;">Montant (BIF)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($transactions)): ?>
                        <tr>
                            <td colspan="7" style="padding: 60px 20px; text-align: center;">
                                <div style="font-size: 48px; margin-bottom: 10px;">📋</div>
                                <p style="color: #4B2F1F; font-size: 16px; font-weight: 600; margin: 0 0 5px;">Aucune transaction trouvée</p>
                                <p style="color: #6B4F2F; font-size: 13px; margin: 0;">Essayez de modifier vos filtres de recherche.</p>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($transactions as $transaction): 
                            $typeColor = match($transaction['transaction_type']) {
                                'Transfert Reçu' => ['bg' => '#e3f2fd', 'color' => '#1565C0'],
                                'Dépôt Bancaire' => ['bg' => '#e8f5e9', 'color' => '#2E7D32'],
                                'Dépense' => ['bg' => '#ffebee', 'color' => '#C62828'],
                                'Remboursement Crédit' => ['bg' => '#fff3e0', 'color' => '#E65100'],
                                'Vente à crédit' => ['bg' => '#f3e5f5', 'color' => '#7B1FA2'],
                                default => ['bg' => '#f5f5f5', 'color' => '#616161']
                            };
                        ?>
                            <tr style="border-bottom: 1px solid #F0F0F0; transition: background-color 0.2s;" 
                                onmouseover="this.style.backgroundColor='#FDFBF7'" 
                                onmouseout="this.style.backgroundColor='transparent'">
                                <td style="padding: 12px 16px; color: #4B2F1F;">
                                    <div style="font-weight: 500;"><?php echo htmlspecialchars(date('d/m/Y', strtotime($transaction['created_at'])), ENT_QUOTES, 'UTF-8'); ?></div>
                                    <div style="font-size: 12px; color: #6B4F2F;"><?php echo htmlspecialchars(date('H:i', strtotime($transaction['created_at'])), ENT_QUOTES, 'UTF-8'); ?></div>
                                </td>
                                <td style="padding: 12px 16px; font-weight: 600; color: #2C1810;"><?php echo htmlspecialchars($transaction['user_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px 16px; color: #6B4F2F;"><?php echo htmlspecialchars($transaction['user_role'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px 16px;">
                                    <span style="display: inline-block; padding: 4px 10px; border-radius: 4px; font-size: 12px; font-weight: 600; background: <?php echo $typeColor['bg']; ?>; color: <?php echo $typeColor['color']; ?>;">
                                        <?php echo htmlspecialchars($transaction['transaction_type'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?>
                                    </span>
                                </td>
                                <td style="padding: 12px 16px; color: #4B2F1F; font-family: monospace;"><?php echo htmlspecialchars($transaction['reference'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px 16px; color: #4B2F1F; max-width: 250px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;" title="<?php echo htmlspecialchars($transaction['description'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?>">
                                    <?php echo htmlspecialchars($transaction['description'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?>
                                </td>
                                <td style="padding: 12px 16px; text-align: right; font-weight: 700; color: #2C1810; font-size: 15px;">
                                    <?php echo number_format((float)($transaction['amount'] ?? 0), 0, ',', '.'); ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        <!-- Total Row -->
                        <tr style="background: #4B2F1F; color: #F4F0E4;">
                            <td colspan="6" style="padding: 14px 16px; font-weight: 700; text-align: right; font-size: 15px;">TOTAL</td>
                            <td style="padding: 14px 16px; text-align: right; font-weight: 700; font-size: 17px;">
                                <?php echo number_format($totalAmount, 0, ',', '.'); ?> BIF
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
