<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../config/database.php';

require_login();

$can_manage_prices = in_array($_SESSION['role_name'] ?? '', ['DG', 'DGA', 'Board', 'Admin'], true);
$can_view_history = in_array($_SESSION['role_name'] ?? '', ['DG', 'DGA', 'Board', 'Admin', 'Operation Manager', 'Stock Manager', 'Senior Accountant', 'Junior Accountant', 'Accountant', 'Marketing Head', 'Chef Marketing', 'Marketing Agent', 'Agent Marketing'], true);
$errors = [];
$successMessages = [];

$conn = get_db_connection();
if (!$conn) {
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Ensure product_price_history table exists
$createTableSql = "
    CREATE TABLE IF NOT EXISTS product_price_history (
        id INT AUTO_INCREMENT PRIMARY KEY,
        product_id INT NOT NULL,
        old_price INT NOT NULL,
        new_price INT NOT NULL,
        changed_by INT NOT NULL,
        changed_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_product (product_id),
        INDEX idx_changed_at (changed_at),
        FOREIGN KEY (product_id) REFERENCES product(id) ON DELETE CASCADE,
        FOREIGN KEY (changed_by) REFERENCES user(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";
if (!$conn->query($createTableSql)) {
    error_log("Failed to ensure product_price_history table: " . $conn->error);
}

// Ensure user_notification table exists
$createNotificationTableSql = "
    CREATE TABLE IF NOT EXISTS user_notification (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        message VARCHAR(255) NOT NULL,
        link VARCHAR(255) DEFAULT NULL,
        is_read TINYINT(1) NOT NULL DEFAULT 0,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user_read (user_id, is_read),
        CONSTRAINT fk_notification_user_catalog FOREIGN KEY (user_id) REFERENCES user(id) ON DELETE CASCADE
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
";
if (!$conn->query($createNotificationTableSql)) {
    error_log("Failed to ensure user_notification table: " . $conn->error);
}

if ($can_manage_prices && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_product'])) {
    $product_id = filter_input(INPUT_POST, 'product_id', FILTER_VALIDATE_INT);
    $price_per_crate = filter_input(INPUT_POST, 'price_per_crate', FILTER_VALIDATE_INT);
    $alert_crates = filter_input(INPUT_POST, 'low_stock_alert_crates', FILTER_VALIDATE_INT, ['options' => ['default' => 0]]);
    $alert_units = filter_input(INPUT_POST, 'low_stock_alert_units', FILTER_VALIDATE_INT, ['options' => ['default' => 0]]);

    if (!$product_id) {
        $errors[] = 'Produit invalide sélectionné.';
    }
    if ($price_per_crate === null || $price_per_crate < 0) {
        $errors[] = 'Le prix par caisse doit être zéro ou supérieur.';
    }

    if (empty($errors)) {
        // Get current price before update
        $stmt_get = $conn->prepare('SELECT price_per_crate, name FROM product WHERE id = ?');
        $old_price = null;
        $product_name = '';
        if ($stmt_get) {
            $stmt_get->bind_param('i', $product_id);
            $stmt_get->execute();
            $result = $stmt_get->get_result();
            if ($row = $result->fetch_assoc()) {
                $old_price = (int)$row['price_per_crate'];
                $product_name = $row['name'];
            }
            $stmt_get->close();
        }

        $conn->begin_transaction();
        try {
            $alert_crates = max(0, (int)$alert_crates);
            $alert_units = max(0, (int)$alert_units);
            $stmt_update = $conn->prepare('UPDATE product SET price_per_crate = ?, low_stock_alert_crates = ?, low_stock_alert_units = ? WHERE id = ?');
            if ($stmt_update) {
                $stmt_update->bind_param('iiii', $price_per_crate, $alert_crates, $alert_units, $product_id);
                if ($stmt_update->execute()) {
                    // Record price change in history if price actually changed
                    if ($old_price !== null && $old_price != $price_per_crate) {
                        $stmt_history = $conn->prepare('INSERT INTO product_price_history (product_id, old_price, new_price, changed_by) VALUES (?, ?, ?, ?)');
                        if ($stmt_history) {
                            $changed_by = (int)$_SESSION['user_id'];
                            $stmt_history->bind_param('iiii', $product_id, $old_price, $price_per_crate, $changed_by);
                            $stmt_history->execute();
                            $stmt_history->close();

                            // Notify all users about price change
                            try {
                                $users_stmt = $conn->query("SELECT id FROM user WHERE id IS NOT NULL");
                                if ($users_stmt === false) {
                                    error_log("Failed to query users for notification: " . $conn->error);
                                } else {
                                    $notification_message = sprintf(
                                        'Le prix du produit "%s" a changé: %s BIF → %s BIF par caisse',
                                        $product_name,
                                        number_format($old_price, 0, ',', '.'),
                                        number_format($price_per_crate, 0, ',', '.')
                                    );
                                    $notification_link = '/masunzu_bar_hotel/modules/inventory/product_catalog.php';
                                    
                                    $notif_stmt = $conn->prepare('INSERT INTO user_notification (user_id, message, link, created_at) VALUES (?, ?, ?, NOW())');
                                    if ($notif_stmt === false) {
                                        error_log("Failed to prepare notification statement: " . $conn->error);
                                        if ($users_stmt) $users_stmt->free();
                                    } else {
                                        $notification_count = 0;
                                        $users_fetched = 0;
                                        while ($user = $users_stmt->fetch_assoc()) {
                                            $users_fetched++;
                                            $user_id = (int)$user['id'];
                                            if ($user_id > 0) {
                                                $notif_stmt->bind_param('iss', $user_id, $notification_message, $notification_link);
                                                if ($notif_stmt->execute()) {
                                                    $notification_count++;
                                                } else {
                                                    error_log("Failed to send notification to user {$user_id}: " . $notif_stmt->error);
                                                }
                                            }
                                        }
                                        $notif_stmt->close();
                                        if ($users_stmt) $users_stmt->free();
                                        
                                        if ($notification_count > 0) {
                                            error_log("✓ Price change notifications sent to {$notification_count} users for product: {$product_name}");
                                        } else {
                                            error_log("⚠ No notifications sent. Users fetched: {$users_fetched}");
                                        }
                                    }
                                }
                            } catch (Exception $e) {
                                error_log("Exception while sending notifications: " . $e->getMessage());
                            }
                        }
                    }

                    $conn->commit();
                    $successMessages[] = 'Produit mis à jour avec succès.';
                    log_action($_SESSION['user_id'], 'update_product_prices', "Updated pricing for product ID {$product_id}: {$old_price} → {$price_per_crate} BIF");
                } else {
                    throw new Exception('Failed to update product: ' . $stmt_update->error);
                }
                $stmt_update->close();
            } else {
                throw new Exception('Failed to prepare product update: ' . $conn->error);
            }
        } catch (Exception $e) {
            $conn->rollback();
            $errors[] = $e->getMessage();
        }
    }
}

// Ensure audit columns exist
$columnCheck = $conn->query("SHOW COLUMNS FROM product LIKE 'created_by'");
if ($columnCheck && $columnCheck->num_rows === 0) {
    $conn->query("ALTER TABLE product ADD COLUMN created_by INT NULL AFTER price_per_crate");
}
if ($columnCheck instanceof mysqli_result) {
    $columnCheck->free();
}
$columnCheck = $conn->query("SHOW COLUMNS FROM product LIKE 'created_at'");
if ($columnCheck && $columnCheck->num_rows === 0) {
    $conn->query("ALTER TABLE product ADD COLUMN created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP AFTER created_by");
}
if ($columnCheck instanceof mysqli_result) {
    $columnCheck->free();
}

$products = [];
$stmt = $conn->prepare("
    SELECT p.id, p.name, p.volume_cl, p.crate_quantity, p.expiration_date, p.price_per_crate,
           p.low_stock_alert_crates, p.low_stock_alert_units, p.created_at, u.full_name AS created_by_name
    FROM product p
    LEFT JOIN user u ON p.created_by = u.id
    ORDER BY name
");
if ($stmt) {
    $stmt->execute();
    $products = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
} else {
    error_log('Failed to prepare product query: ' . $conn->error);
}

$conn->close();

if (isset($_GET['download_csv'])) {
    $filename = 'product_catalog_' . date('Y-m-d_H-i-s') . '.csv';
    
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = fopen('php://output', 'w');
    
    fputcsv($output, ['#', 'Name', 'Volume (cl)', 'Crate Quantity', 'Expiration Date', 'Price / Crate (BIF)', 'Low Stock Alert (Crates)', 'Low Stock Alert (Bottles)', 'Created By', 'Created At']);
    
    foreach ($products as $index => $product) {
        fputcsv($output, [
            $index + 1,
            $product['name'],
            $product['volume_cl'],
            $product['crate_quantity'],
            $product['expiration_date'] ?? 'N/A',
            $product['price_per_crate'],
            ($product['low_stock_alert_crates'] ?? 0) > 0 ? $product['low_stock_alert_crates'] : '',
            ($product['low_stock_alert_units'] ?? 0) > 0 ? $product['low_stock_alert_units'] : '',
            $product['created_by_name'] ?? 'Unknown',
            $product['created_at'] ? date('Y-m-d H:i', strtotime($product['created_at'])) : 'N/A'
        ]);
    }
    
    fclose($output);
    exit;
}

include __DIR__ . '/../../includes/header.php';
?>
<div style="padding: 20px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 70vh; width: 100%; margin: 0; box-sizing: border-box;">
    <div style="width: 100%; max-width: 100%; margin: 0 auto; padding: 0; background-color: #F4F0E4; padding: 30px; border-radius: 12px; box-shadow: 0 6px 16px rgba(0,0,0,0.15); border: none;">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
            <h2 style="color: #4B2F1F; margin: 0;">Catalogue des Produits</h2>
            <?php if ($can_view_history): ?>
                <a href="product_price_history.php" style="padding: 10px 20px; background-color: #FF9800; color: #FFFFFF; border: none; border-radius: 8px; text-decoration: none; cursor: pointer; font-weight: 600; display: inline-block; font-size: 14px;">📊 Historique des Prix</a>
            <?php endif; ?>
        </div>
        <p style="color: #000000; text-align: center; margin-bottom: 20px;">
            Vue d'ensemble de tous les produits enregistrés, y compris les détails d'emballage et les prix.
        </p>
        <div style="display: flex; gap: 10px; max-width: 500px; margin: 0 auto 20px; flex-wrap: wrap; justify-content: center;">
            <input
                type="text"
                id="product-search"
                placeholder="Rechercher des produits..."
                style="flex: 1; min-width: 200px; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; font-size: 14px;"
            >
            <a href="?download_csv=1" style="padding: 10px 20px; background-color: #4B2F1F; color: #F4F0E4; border: none; border-radius: 8px; text-decoration: none; cursor: pointer; font-weight: 600; display: inline-block; font-size: 14px;">📥 Télécharger CSV</a>
        </div>

        <?php if (!empty($errors)): ?>
            <div style="background-color:#f8d7da; border:1px solid #f5c6cb; color:#721c24; padding:12px 16px; border-radius:8px; margin-bottom:15px; border: none;">
                <?php foreach ($errors as $error): ?>
                    <p style="margin:0;"><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
        <?php if (!empty($successMessages)): ?>
            <div style="background-color:#d4edda; border:1px solid #c3e6cb; color:#155724; padding:12px 16px; border-radius:8px; margin-bottom:15px; border: none;">
                <?php foreach ($successMessages as $message): ?>
                    <p style="margin:0;"><?php echo htmlspecialchars($message, ENT_QUOTES, 'UTF-8'); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <?php if (empty($products)): ?>
            <p style="color: #4B2F1F; text-align: center;">Aucun produit trouvé.</p>
        <?php else: ?>
            <div style="overflow-x: auto;">
                <table style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 10px; overflow: hidden; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                    <thead>
                        <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                            <th style="padding: 12px; text-align: center; width: 40px; border: none;">#</th>
                            <th style="padding: 12px; text-align: left; border: none;">Nom</th>
                            <th style="padding: 12px; text-align: left; border: none;">Volume (cl)</th>
                            <th style="padding: 12px; text-align: left; border: none;">Quantité par Caisse</th>
                            <th style="padding: 12px; text-align: left; border: none;">Date d'Expiration</th>
                            <th style="padding: 12px; text-align: left; border: none;">Prix / Caisse (BIF)</th>
                            <th style="padding: 12px; text-align: left; border: none;">Alerte Stock Faible (Caisses)</th>
                            <th style="padding: 12px; text-align: left; border: none;">Alerte Stock Faible (Bouteilles)</th>
                            <th style="padding: 12px; text-align: left; border: none;">Créé Par</th>
                            <th style="padding: 12px; text-align: left; border: none;">Date de Création</th>
                            <th style="padding: 12px; text-align: center; border: none;">Actions</th>
                        </tr>
                    </thead>
                    <tbody id="product-table-body">
                        <?php foreach ($products as $index => $product): ?>
                            <tr style="border-bottom: 1px solid #E0E0E0;" data-search="<?php echo htmlspecialchars(strtolower($product['name']), ENT_QUOTES, 'UTF-8'); ?>">
                                <td style="padding: 12px; text-align: center; color: #4B2F1F; font-weight: 600; border: none;"><?php echo $index + 1; ?></td>
                                <td style="padding: 12px; color: #4B2F1F; font-weight: 600; border: none;"><?php echo htmlspecialchars($product['name'], ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px; border: none;"><?php echo number_format((int)$product['volume_cl'], 0, ',', '.'); ?></td>
                                <td style="padding: 12px; border: none;"><?php echo number_format((int)$product['crate_quantity'], 0, ',', '.'); ?></td>
                                <td style="padding: 12px; border: none;"><?php echo htmlspecialchars($product['expiration_date'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px; font-weight: 600; border: none;"><?php echo number_format((int)$product['price_per_crate'], 0, ',', '.'); ?> BIF</td>
                                <td style="padding: 12px; border: none;"><?php echo (int)($product['low_stock_alert_crates'] ?? 0) > 0 ? number_format((int)$product['low_stock_alert_crates'], 0, ',', '.') : '—'; ?></td>
                                <td style="padding: 12px; border: none;"><?php echo (int)($product['low_stock_alert_units'] ?? 0) > 0 ? number_format((int)$product['low_stock_alert_units'], 0, ',', '.') : '—'; ?></td>
                                <td style="padding: 12px; border: none;"><?php echo htmlspecialchars($product['created_by_name'] ?? 'Inconnu', ENT_QUOTES, 'UTF-8'); ?></td>
                                <td style="padding: 12px; border: none;"><?php echo $product['created_at'] ? htmlspecialchars(date('Y-m-d H:i', strtotime($product['created_at'])), ENT_QUOTES, 'UTF-8') : 'N/A'; ?></td>
                                <td style="padding: 12px; text-align: center; border: none;">
                                    <div style="display: flex; gap: 5px; justify-content: center;">
                                        <button type="button" class="view-product-btn" data-product-id="<?php echo $product['id']; ?>" style="background-color: #2196F3; color: #FFFFFF; padding: 6px 12px; border: none; border-radius: 6px; cursor: pointer; font-size: 13px; font-weight: 600;">Voir</button>
                                        <?php if ($can_manage_prices): ?>
                                            <button type="button" class="edit-product-btn" data-product-id="<?php echo $product['id']; ?>" style="background-color: #4B2F1F; color: #F4F0E4; padding: 6px 12px; border: none; border-radius: 6px; cursor: pointer; font-size: 13px; font-weight: 600;">Modifier</button>
                                        <?php endif; ?>
                                        <?php if ($can_view_history): ?>
                                            <a href="/masunzu_bar_hotel/modules/operation/product_history.php?id=<?php echo $product['id']; ?>" class="view-history-link">Historique</a>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
        <!-- View Product Details Modal -->
        <div id="viewProductModal" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.5); z-index:1000; justify-content:center; align-items:center;">
            <div style="background-color:#F4F0E4; padding:20px; border-radius:10px; width:90%; max-width:500px; box-shadow:0 6px 20px rgba(0,0,0,0.2); position:relative; max-height:90vh; overflow-y:auto;">
                <button type="button" id="closeViewProductModal" style="position:absolute; top:10px; right:12px; background:none; border:none; font-size:24px; color:#4B2F1F; cursor:pointer;">&times;</button>
                <h3 style="color:#4B2F1F; margin-top:0;">Détails du Produit</h3>
                <div id="view-product-details" style="display:flex; flex-direction:column; gap:12px;">
                    <!-- Details will be filled by JavaScript -->
                </div>
                <div style="margin-top:15px;">
                    <button type="button" id="closeViewProductBtn" style="background-color:#4B2F1F; color:#F4F0E4; padding:10px 20px; border:none; border-radius:6px; cursor:pointer; font-weight:600; width:100%;">Fermer</button>
                </div>
            </div>
        </div>

        <?php if ($can_manage_prices): ?>
            <!-- Edit Product Modal -->
            <div id="editProductModal" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.5); z-index:1001; justify-content:center; align-items:center;">
                <div style="background-color:#F4F0E4; padding:20px; border-radius:10px; width:90%; max-width:420px; box-shadow:0 6px 20px rgba(0,0,0,0.2); position:relative;">
                    <button type="button" id="closeEditProductModal" style="position:absolute; top:10px; right:12px; background:none; border:none; font-size:24px; color:#4B2F1F; cursor:pointer;">&times;</button>
                    <h3 style="color:#4B2F1F; margin-top:0;">Modifier le Prix et les Alertes du Produit</h3>
                    <p style="color:#2C1810; font-weight:600; margin-bottom:12px;" id="edit-product-name">&nbsp;</p>
                    <form method="POST" id="editProductForm" style="display:flex; flex-direction:column; gap:12px;">
                        <input type="hidden" name="update_product" value="1">
                        <input type="hidden" name="product_id" id="edit-product-id">
                        <label style="display:flex; flex-direction:column; color:#4B2F1F; font-weight:600;">
                            Prix / Caisse (BIF)
                            <input type="number" name="price_per_crate" id="edit-price-per-crate" min="0" step="1" required style="margin-top:6px; padding:8px; border-radius:6px; border:1px solid #4B2F1F; background-color:#FFFFFF; color:#4B2F1F;">
                        </label>
                        <label style="display:flex; flex-direction:column; color:#4B2F1F; font-weight:600;">
                            Alerte Stock Faible &ndash; Caisses
                            <input type="number" name="low_stock_alert_crates" id="edit-alert-crates" min="0" step="1" style="margin-top:6px; padding:8px; border-radius:6px; border:1px solid #4B2F1F; background-color:#FFFFFF; color:#4B2F1F;" placeholder="0 pour désactiver">
                        </label>
                        <label style="display:flex; flex-direction:column; color:#4B2F1F; font-weight:600;">
                            Alerte Stock Faible &ndash; Bouteilles
                            <input type="number" name="low_stock_alert_units" id="edit-alert-units" min="0" step="1" style="margin-top:6px; padding:8px; border-radius:6px; border:1px solid #4B2F1F; background-color:#FFFFFF; color:#4B2F1F;" placeholder="0 pour désactiver">
                        </label>
                        <div style="display:flex; gap:10px; margin-top:10px;">
                            <button type="submit" style="flex:1; background-color:#4B2F1F; color:#F4F0E4; padding:10px 0; border:none; border-radius:6px; cursor:pointer; font-weight:600;">Enregistrer</button>
                            <button type="button" id="cancelEditProduct" style="flex:1; background-color:#E0D6CE; color:#4B2F1F; padding:10px 0; border:none; border-radius:6px; cursor:pointer; font-weight:600;">Annuler</button>
                        </div>
                    </form>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>
<style>
    body {
        margin: 0;
        padding: 0;
        overflow-x: hidden;
    }
    table {
        border-collapse: collapse;
    }
    table th, table td {
        border: none;
    }
    button:hover {
        opacity: 0.9;
        transform: translateY(-1px);
        box-shadow: 0 2px 4px rgba(0,0,0,0.2);
    }
    .view-history-link {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        padding: 6px 12px;
        border-radius: 6px;
        background-color: #6e4b34;
        color: #f4f0e4;
        text-decoration: none;
        font-size: 13px;
        font-weight: 600;
    }
</style>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('product-search');
    const tableBody = document.getElementById('product-table-body');
    if (!searchInput || !tableBody) return;

    searchInput.addEventListener('input', function() {
        const term = this.value.trim().toLowerCase();
        Array.from(tableBody.getElementsByTagName('tr')).forEach(row => {
            const searchData = row.getAttribute('data-search') || '';
            row.style.display = searchData.includes(term) ? '' : 'none';
        });
    });

    const productsData = <?php echo json_encode($products); ?>;
    
    // View Product Modal
    const viewModal = document.getElementById('viewProductModal');
    const closeViewBtn = document.getElementById('closeViewProductModal');
    const closeViewBtn2 = document.getElementById('closeViewProductBtn');
    const viewDetailsEl = document.getElementById('view-product-details');

    function openViewModal(productId) {
        const product = productsData.find(p => parseInt(p.id, 10) === parseInt(productId, 10));
        if (!product) {
            alert('Impossible de charger les détails du produit.');
            return;
        }
        
        const totalBottles = parseInt(product.crate_quantity || 0);
        const pricePerBottle = parseInt(product.price_per_crate || 0) / totalBottles;
        
        viewDetailsEl.innerHTML = `
            <div style="background:#FFFFFF; padding:15px; border-radius:8px; border:1px solid #E0E0E0;">
                <table style="width:100%; border-collapse:collapse;">
                    <tr><td style="padding:8px; font-weight:600; color:#4B2F1F; border:none;">Nom du Produit:</td><td style="padding:8px; border:none;">${product.name || 'N/A'}</td></tr>
                    <tr><td style="padding:8px; font-weight:600; color:#4B2F1F; border:none;">Volume:</td><td style="padding:8px; border:none;">${parseInt(product.volume_cl || 0).toLocaleString('fr-FR')} cl</td></tr>
                    <tr><td style="padding:8px; font-weight:600; color:#4B2F1F; border:none;">Quantité par Caisse:</td><td style="padding:8px; border:none;">${parseInt(product.crate_quantity || 0).toLocaleString('fr-FR')} bouteilles</td></tr>
                    <tr><td style="padding:8px; font-weight:600; color:#4B2F1F; border:none;">Date d'Expiration:</td><td style="padding:8px; border:none;">${product.expiration_date || 'N/A'}</td></tr>
                    <tr><td style="padding:8px; font-weight:600; color:#4B2F1F; border:none;">Prix par Caisse:</td><td style="padding:8px; border:none; font-weight:600; color:#4CAF50;">${parseInt(product.price_per_crate || 0).toLocaleString('fr-FR')} BIF</td></tr>
                    <tr><td style="padding:8px; font-weight:600; color:#4B2F1F; border:none;">Prix par Bouteille:</td><td style="padding:8px; border:none;">${pricePerBottle > 0 ? pricePerBottle.toLocaleString('fr-FR', {maximumFractionDigits: 0}) : 'N/A'} BIF</td></tr>
                    <tr><td style="padding:8px; font-weight:600; color:#4B2F1F; border:none;">Alerte Stock Faible (Caisses):</td><td style="padding:8px; border:none;">${parseInt(product.low_stock_alert_crates || 0) > 0 ? parseInt(product.low_stock_alert_crates).toLocaleString('fr-FR') : 'Désactivée'}</td></tr>
                    <tr><td style="padding:8px; font-weight:600; color:#4B2F1F; border:none;">Alerte Stock Faible (Bouteilles):</td><td style="padding:8px; border:none;">${parseInt(product.low_stock_alert_units || 0) > 0 ? parseInt(product.low_stock_alert_units).toLocaleString('fr-FR') : 'Désactivée'}</td></tr>
                    <tr><td style="padding:8px; font-weight:600; color:#4B2F1F; border:none;">Créé Par:</td><td style="padding:8px; border:none;">${product.created_by_name || 'Inconnu'}</td></tr>
                    <tr><td style="padding:8px; font-weight:600; color:#4B2F1F; border:none;">Date de Création:</td><td style="padding:8px; border:none;">${product.created_at ? new Date(product.created_at).toLocaleString('fr-FR') : 'N/A'}</td></tr>
                </table>
            </div>
        `;
        viewModal.style.display = 'flex';
        document.body.style.overflow = 'hidden';
    }

    function closeViewModal() {
        viewModal.style.display = 'none';
        document.body.style.overflow = 'auto';
    }

    document.querySelectorAll('.view-product-btn').forEach(button => {
        button.addEventListener('click', () => {
            const productId = button.getAttribute('data-product-id');
            openViewModal(productId);
        });
    });

    closeViewBtn?.addEventListener('click', closeViewModal);
    closeViewBtn2?.addEventListener('click', closeViewModal);
    viewModal?.addEventListener('click', event => {
        if (event.target === viewModal) {
            closeViewModal();
        }
    });

<?php if ($can_manage_prices): ?>
    // Edit Product Modal
    const editModal = document.getElementById('editProductModal');
    const closeEditBtn = document.getElementById('closeEditProductModal');
    const cancelEditBtn = document.getElementById('cancelEditProduct');
    const editForm = document.getElementById('editProductForm');
    const productNameEl = document.getElementById('edit-product-name');
    const productIdInput = document.getElementById('edit-product-id');
    const pricePerCrateInput = document.getElementById('edit-price-per-crate');
    const alertCratesInput = document.getElementById('edit-alert-crates');
    const alertUnitsInput = document.getElementById('edit-alert-units');

    function openEditModal(productId) {
        const product = productsData.find(p => parseInt(p.id, 10) === parseInt(productId, 10));
        if (!product) {
            alert('Impossible de charger les détails du produit.');
            return;
        }
        productNameEl.textContent = product.name || 'Produit inconnu';
        productIdInput.value = product.id;
        pricePerCrateInput.value = product.price_per_crate || 0;
        alertCratesInput.value = product.low_stock_alert_crates ? parseInt(product.low_stock_alert_crates, 10) : 0;
        alertUnitsInput.value = product.low_stock_alert_units ? parseInt(product.low_stock_alert_units, 10) : 0;
        editModal.style.display = 'flex';
        document.body.style.overflow = 'hidden';
    }

    function closeEditModal() {
        editModal.style.display = 'none';
        document.body.style.overflow = 'auto';
    }

    document.querySelectorAll('.edit-product-btn').forEach(button => {
        button.addEventListener('click', () => {
            const productId = button.getAttribute('data-product-id');
            openEditModal(productId);
        });
    });

    closeEditBtn?.addEventListener('click', closeEditModal);
    cancelEditBtn?.addEventListener('click', closeEditModal);

    editModal?.addEventListener('click', event => {
        if (event.target === editModal) {
            closeEditModal();
        }
    });
<?php endif; ?>
});
</script>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
