<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// Restrict to Stock Manager role (role_id = 5)
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in approve_transfers.php");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}

// Ensure main_stock has reserved_quantity column for crate-based reservation/finalization
try {
    $checkReserved = $conn->query("SHOW COLUMNS FROM main_stock LIKE 'reserved_quantity'");
    if ($checkReserved && $checkReserved->num_rows === 0) {
        $conn->query("ALTER TABLE main_stock ADD COLUMN reserved_quantity INT(11) NOT NULL DEFAULT 0 AFTER quantity");
        error_log("Added reserved_quantity column to main_stock table (approve_transfers.php)");
    }
    if ($checkReserved) {
        $checkReserved->free();
    }
} catch (Exception $e) {
    error_log("Error ensuring reserved_quantity on main_stock in approve_transfers.php: " . $e->getMessage());
}
$stmt = $conn->prepare("SELECT r.name, u.province_id FROM role r JOIN user u ON u.role_id = r.id WHERE u.id = ?");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$user = $stmt->get_result()->fetch_assoc();
$role = $user['name'] ?? '';
$province_id = $user['province_id'] ?? null;
$stmt->close();
if ($role !== 'Stock Manager' || !$province_id) {
    error_log("Unauthorized access to approve_transfers.php by user {$_SESSION['full_name']} (Role: $role, Province: $province_id)");
    header("Location: /masunzu_bar_hotel/dashboards/stock_manager_dashboard.php");
    exit;
}

// Fetch pending transfers for the province
$stmt = $conn->prepare("
    SELECT t.id, t.shipment_id, t.product_id, t.quantity, t.sent_at, p.name AS product_name
    FROM stock_transfer t
    JOIN product p ON t.product_id = p.id
    WHERE t.province_id = ? AND t.stock_manager_id = ? AND t.status = 'pending'
");
$stmt->bind_param("ii", $province_id, $_SESSION['user_id']);
$stmt->execute();
$pending_transfers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Handle transfer approval
$errors = [];
$success = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit'])) {
    $transfer_id = (int)($_POST['transfer_id'] ?? 0);

    // Validate transfer
    $stmt = $conn->prepare("
        SELECT product_id, quantity, province_id
        FROM stock_transfer
        WHERE id = ? AND stock_manager_id = ? AND status = 'pending'
    ");
    $stmt->bind_param("ii", $transfer_id, $_SESSION['user_id']);
    $stmt->execute();
    $transfer = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    if (!$transfer) {
        $errors[] = "Invalid or unauthorized transfer.";
    }

    if (empty($errors)) {
        $conn->begin_transaction();
        try {
            // Finalize reservation in main_stock: move from reserved to permanently deducted
            $stmt = $conn->prepare("
                SELECT quantity, COALESCE(reserved_quantity, 0) AS reserved_quantity
                FROM main_stock
                WHERE product_id = ? AND (unit_type = 'crates' OR unit_type IS NULL OR unit_type = '')
                FOR UPDATE
            ");
            $stmt->bind_param("i", $transfer['product_id']);
            $stmt->execute();
            $result = $stmt->get_result();
            $mainStock = $result->fetch_assoc();
            $stmt->close();

            if (!$mainStock) {
                throw new Exception("Main stock record not found for product_id={$transfer['product_id']}");
            }

            $currentQuantity = (int)$mainStock['quantity'];
            $currentReserved = (int)$mainStock['reserved_quantity'];

            if ($currentReserved < $transfer['quantity']) {
                throw new Exception("Reserved crates in main stock ({$currentReserved}) are less than transfer quantity ({$transfer['quantity']}).");
            }

            $stmt = $conn->prepare("
                UPDATE main_stock
                SET reserved_quantity = GREATEST(0, reserved_quantity - ?),
                    quantity = GREATEST(0, quantity - ?),
                    unit_type = 'crates',
                    last_updated = NOW()
                WHERE product_id = ? AND (unit_type = 'crates' OR unit_type IS NULL OR unit_type = '')
            ");
            $stmt->bind_param("iii", $transfer['quantity'], $transfer['quantity'], $transfer['product_id']);
            if (!$stmt->execute()) {
                throw new Exception("Failed to finalize main stock: " . $stmt->error);
            }
            $stmt->close();

            // Update province_stock (destination) in crates
            $stmt = $conn->prepare("
                SELECT quantity FROM province_stock 
                WHERE product_id = ? AND province_id = ? AND unit_type = 'crates'
            ");
            $stmt->bind_param("ii", $transfer['product_id'], $transfer['province_id']);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result->num_rows > 0) {
                $stmt = $conn->prepare("
                    UPDATE province_stock 
                    SET quantity = quantity + ?, last_updated = NOW()
                    WHERE product_id = ? AND province_id = ? AND unit_type = 'crates'
                ");
            } else {
                $stmt = $conn->prepare("
                    INSERT INTO province_stock (product_id, province_id, quantity, reserved_quantity, unit_type, last_updated)
                    VALUES (?, ?, ?, 0, 'crates', NOW())
                ");
            }

            if ($result->num_rows > 0) {
                $stmt->bind_param("iii", $transfer['quantity'], $transfer['product_id'], $transfer['province_id']);
            } else {
                $stmt->bind_param("iii", $transfer['product_id'], $transfer['province_id'], $transfer['quantity']);
            }

            if (!$stmt->execute()) {
                throw new Exception("Failed to update province stock: " . $stmt->error);
            }
            $stmt->close();

            // Update stock_transfer status
            $stmt = $conn->prepare("
                UPDATE stock_transfer SET status = 'confirmed', received_at = NOW()
                WHERE id = ?
            ");
            $stmt->bind_param("i", $transfer_id);
            if (!$stmt->execute()) {
                throw new Exception("Failed to update transfer status: " . $stmt->error);
            }
            $stmt->close();

            $conn->commit();
            $success = "Transfer approved successfully.";
            // Refresh pending transfers
            $stmt = $conn->prepare("
                SELECT t.id, t.shipment_id, t.product_id, t.quantity, t.sent_at, p.name AS product_name
                FROM stock_transfer t
                JOIN product p ON t.product_id = p.id
                WHERE t.province_id = ? AND t.stock_manager_id = ? AND t.status = 'pending'
            ");
            $stmt->bind_param("ii", $province_id, $_SESSION['user_id']);
            $stmt->execute();
            $pending_transfers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmt->close();
        } catch (Exception $e) {
            $conn->rollback();
            error_log("Transaction failed in approve_transfers.php: " . $e->getMessage());
            $errors[] = "Failed to approve transfer: " . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
        }
    }
}

$conn->close();

include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
?>
<div class="approve-transfers-container" style="padding: 20px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 70vh;">
    <div class="approve-transfers-content" style="background-color: #F4F0E4; padding: 30px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.2); max-width: 800px; margin: 0 auto;">
        <h2 style="color: #4B2F1F; font-size: 28px; margin-bottom: 15px;">Approve Stock Transfers</h2>
        <p style="color: #000000; font-size: 16px;">Review and approve pending stock transfers to your province.</p>
        <?php if (!empty($errors)): ?>
            <div style="background-color: #FFFFFF; color: #FF0000; padding: 10px; border-radius: 5px; border: 1px solid #FF0000; margin-bottom: 15px;">
                <?php foreach ($errors as $error): ?>
                    <p><?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
        <?php if ($success): ?>
            <div style="background-color: #FFFFFF; color: #008000; padding: 10px; border-radius: 5px; border: 1px solid #008000; margin-bottom: 15px;">
                <p><?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?></p>
            </div>
        <?php endif; ?>
        <?php if (!empty($pending_transfers)): ?>
            <table style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 8px; overflow: hidden; margin-bottom: 20px;">
                <thead>
                    <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                        <th style="padding: 12px; text-align: left; width: 15%;">Shipment ID</th>
                        <th style="padding: 12px; text-align: left; width: 25%;">Product</th>
                        <th style="padding: 12px; text-align: left; width: 15%;">Quantity</th>
                        <th style="padding: 12px; text-align: left; width: 25%;">Sent At</th>
                        <th style="padding: 12px; text-align: left; width: 20%;">Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($pending_transfers as $transfer): ?>
                        <tr style="border-bottom: 1px solid #F4F0E4;">
                            <td style="padding: 12px; text-align: left; width: 15%;"><?php echo htmlspecialchars($transfer['shipment_id'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: left; width: 25%;"><?php echo htmlspecialchars($transfer['product_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding: 12px; text-align: left; width: 15%;"><?php echo number_format($transfer['quantity'], 0, ',', '.'); ?></td>
                            <td style="padding: 12px; text-align: left; width: 25%;"><?php echo date('Y-m-d H:i', strtotime($transfer['sent_at'])); ?></td>
                            <td style="padding: 12px; text-align: left; width: 20%;">
                                <form method="POST" style="display: inline;">
                                    <input type="hidden" name="transfer_id" value="<?php echo htmlspecialchars($transfer['id'], ENT_QUOTES, 'UTF-8'); ?>">
                                    <button type="submit" name="submit" style="background-color: #4B2F1F; color: #F4F0E4; padding: 8px 16px; border: none; border-radius: 8px; cursor: pointer; font-size: 14px;">Approve</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <p style="color: #000000; font-size: 16px;">No pending transfers to approve.</p>
        <?php endif; ?>
        <a href="/masunzu_bar_hotel/dashboards/stock_manager_dashboard.php" style="display: block; text-align: center; margin-top: 25px; background-color: #F4A261; color: #4B2F1F; padding: 15px; border-radius: 8px; text-decoration: none; font-size: 18px; font-weight: 600; transition: all 0.3s ease;">
            Back to Dashboard
        </a>
        <style>
            button:hover {
                background-color: #F4A261;
                color: #4B2F1F;
            }
            @media (max-width: 600px) {
                .approve-transfers-container { padding: 15px; }
                .approve-transfers-content { padding: 20px; }
                h2 { font-size: 24px; }
                table { font-size: 14px; }
                button { padding: 8px 12px; font-size: 12px; }
            }
        </style>
    </div>
</div>
<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>