<?php
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/auth.php';
require_once '/opt/lampp/htdocs/masunzu_bar_hotel/includes/functions.php';
require_login();

// Restrict to Operation Manager role
if ($_SESSION['role_name'] !== 'Operation Manager') {
    error_log("Unauthorized access to send_stock.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/dashboards/operation_dashboard.php");
    exit;
}

$manager_region_id = $_SESSION['region_id'] ?? null;
if (!$manager_region_id) {
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
    echo "<div style='padding:20px;background:#F4F0E4;min-height:60vh;'><div style='background:#FFF3E0;border:1px solid #E67E22;border-radius:8px;padding:20px;color:#8C5312;font-size:16px;'>Aucune région n'est assignée à votre profil. Contactez l'administration pour compléter votre compte.</div></div>";
    include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php';
    exit;
}

// Fetch products, provinces, and stock managers
$conn = get_db_connection();
$products = [];
$provinces = [];
$stock_managers = [];
$error = '';

$result = $conn->query("SELECT id, name, price_per_crate FROM product");
if ($result instanceof mysqli_result) {
    $products = $result->fetch_all(MYSQLI_ASSOC);
}

$stmt = $conn->prepare("SELECT id, name FROM province WHERE region_id = ? ORDER BY name");
$stmt->bind_param("i", $manager_region_id);
$stmt->execute();
$provinces = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

$stmt = $conn->prepare("
    SELECT u.id, u.full_name, u.province_id
    FROM user u
    JOIN role r ON r.id = u.role_id
    JOIN province p ON p.id = u.province_id
    WHERE r.name = 'Stock Manager' AND u.is_active = 1 AND p.region_id = ?
");
$stmt->bind_param("i", $manager_region_id);
$stmt->execute();
$stock_managers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();
$conn->close();

$region_stock_products = [];
if ($manager_region_id) {
    $conn = get_db_connection();
    $regionStockStmt = $conn->prepare("
        SELECT p.id AS product_id, p.name AS product_name, p.crate_quantity, p.price_per_crate, COALESCE(SUM(rs.quantity), 0) AS total_quantity
        FROM region_stock rs
        JOIN product p ON rs.product_id = p.id
        WHERE rs.region_id = ?
        GROUP BY p.id, p.name, p.crate_quantity, p.price_per_crate
        ORDER BY p.name
    ");
    if ($regionStockStmt) {
        $regionStockStmt->bind_param("i", $manager_region_id);
        $regionStockStmt->execute();
        $rows = $regionStockStmt->get_result()->fetch_all(MYSQLI_ASSOC);
        foreach ($rows as $row) {
            $quantity = (int)($row['total_quantity'] ?? 0);
            $region_stock_products[] = [
                'product_name' => $row['product_name'],
                'quantity' => $quantity,
                'value' => $quantity * (int)($row['price_per_crate'] ?? 0),
            ];
        }
        $regionStockStmt->close();
    }
    $conn->close();
}

$error = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $province_id = (int)($_POST['province_id'] ?? 0);
    $stock_manager_id = (int)($_POST['stock_manager_id'] ?? 0);
    $products_sent = $_POST['products'] ?? [];
    $estimated_arrival = trim($_POST['estimated_arrival'] ?? '');
    $driver_name = trim($_POST['driver_name'] ?? '');
    $driver_mobile = trim($_POST['driver_mobile'] ?? '');
    $plate_number = trim($_POST['plate_number'] ?? '');

    if ($province_id <= 0 || $stock_manager_id <= 0 || empty($products_sent)) {
        $error = "Veuillez sélectionner un dépôt, un gestionnaire et au moins un produit.";
    } else {
        $conn = get_db_connection();
        $conn->begin_transaction();
        try {
            ensure_region_stock_table($conn);
            $available_stock = mbh_get_region_stock_map($conn, (int)$manager_region_id);

            // Validate province and region alignment
            $stmt = $conn->prepare("SELECT region_id, name FROM province WHERE id = ?");
            $stmt->bind_param("i", $province_id);
            $stmt->execute();
            $provinceRow = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            if (!$provinceRow) {
                throw new Exception("Dépôt sélectionné invalide.");
            }
            if ((int)($provinceRow['region_id'] ?? 0) !== (int)$manager_region_id) {
                throw new Exception("Le dépôt sélectionné n'appartient pas à votre région.");
            }
            $province_name = $provinceRow['name'] ?? '';

            // Validate stock manager and ensure they belong to same province
            $stmt = $conn->prepare("
                SELECT u.id, u.full_name, u.province_id
                FROM user u
                JOIN role r ON r.id = u.role_id
                WHERE u.id = ? AND r.name = 'Stock Manager' AND u.is_active = 1
            ");
            $stmt->bind_param("i", $stock_manager_id);
            $stmt->execute();
            $stockManagerRow = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            if (!$stockManagerRow) {
                throw new Exception("Gestionnaire de stock sélectionné invalide.");
            }
            if ((int)($stockManagerRow['province_id'] ?? 0) !== $province_id) {
                throw new Exception("Ce gestionnaire n'est pas rattaché au dépôt choisi.");
            }

            // Validate product entries
            $product_details = [];
            $total_amount = 0;
            $total_products = 0;
            $seen_products = [];
            foreach ($products_sent as $product) {
                $product_id = (int)($product['product_id'] ?? 0);
                $quantity = (int)($product['quantity'] ?? 0);
                if ($product_id <= 0 || $quantity <= 0) {
                    throw new Exception("Chaque ligne produit doit être complète et positive.");
                }
                if (in_array($product_id, $seen_products, true)) {
                    throw new Exception("Veuillez éviter de dupliquer les mêmes produits dans le transfert.");
                }
                $seen_products[] = $product_id;

                $stmt = $conn->prepare("SELECT name, price_per_crate FROM product WHERE id = ?");
                $stmt->bind_param("i", $product_id);
                $stmt->execute();
                $productRow = $stmt->get_result()->fetch_assoc();
                $stmt->close();
                if (!$productRow) {
                    throw new Exception("Produit sélectionné invalide.");
                }

                $available_quantity = $available_stock[$product_id] ?? 0;
                if ($quantity > $available_quantity) {
                    throw new Exception(sprintf(
                        "Stock insuffisant pour %s : disponible %s caisses.",
                        $productRow['name'],
                        number_format($available_quantity, 0, ',', ' ')
                    ));
                }

                $line_value = $quantity * (int)$productRow['price_per_crate'];
                $product_details[] = [
                    'product_id' => $product_id,
                    'product_name' => $productRow['name'],
                    'quantity' => $quantity,
                    'line_total' => $line_value,
                ];
                $total_products += $quantity;
                $total_amount += $line_value;
            }

            foreach ($product_details as $detail) {
                if (!mbh_decrement_region_stock($conn, (int)$manager_region_id, (int)$detail['product_id'], (int)$detail['quantity'])) {
                    throw new Exception("Stock insuffisant pour {$detail['product_name']}. Veuillez ajuster les quantités.");
                }
            }

            $shipment_id = mt_rand(100000, 999999);
            $insertStmt = $conn->prepare("
                INSERT INTO stock_transfer (
                    operation_manager_id, operation_receiver_id, stock_manager_id, initiator_id, product_id, quantity, unit_type,
                    province_id, region_id, status, sent_at, estimated_arrival, driver_name,
                    driver_mobile, plate_number, shipment_id, total_amount, total_products
                ) VALUES (?, NULL, ?, ?, ?, ?, 'crates', ?, ?, 'pending', NOW(), ?, ?, ?, ?, ?, ?, ?)
            ");

            $operationManagerId = (int)$_SESSION['user_id'];
            $stockManagerIdVal = $stock_manager_id;
            $initiatorId = $operationManagerId;
            $provinceIdVal = (int)$province_id;
            $regionIdVal = (int)$manager_region_id;
            $arrivalParam = $estimated_arrival ?: date('Y-m-d H:i:s');
            $driverNameParam = $driver_name ?: '';
            $driverMobileParam = $driver_mobile ?: '';
            $plateNumberParam = $plate_number ?: '';

            foreach ($product_details as $detail) {
                $productId = (int)$detail['product_id'];
                $quantityValue = (int)$detail['quantity'];
                $insertStmt->bind_param(
                    'iiiiiiissssiii',
                    $operationManagerId,
                    $stockManagerIdVal,
                    $initiatorId,
                    $productId,
                    $quantityValue,
                    $provinceIdVal,
                    $regionIdVal,
                    $arrivalParam,
                    $driverNameParam,
                    $driverMobileParam,
                    $plateNumberParam,
                    $shipment_id,
                    $total_amount,
                    $total_products
                );
                if (!$insertStmt->execute()) {
                    throw new Exception("Échec de l'enregistrement du produit {$detail['product_name']}.");
                }
            }
            $insertStmt->close();

            log_action($_SESSION['user_id'], 'send_stock', "Transfert #$shipment_id vers {$province_name} (Gestionnaire ID: $stock_manager_id, Total caisses: $total_products)");
            $conn->commit();

            $message = sprintf("Un nouveau transfert #%d arrive bientôt dans %s.", $shipment_id, $province_name ?: 'votre dépôt');
            mbh_notify_user($conn, $stock_manager_id, $message, '/masunzu_bar_hotel/modules/operation/receive_stock.php');

            $conn->close();
            $_SESSION['send_stock_success'] = "Le transfert #$shipment_id vers {$province_name} a été initié avec succès.";
            header("Location: /masunzu_bar_hotel/modules/operation/send_stock.php");
            exit;
        } catch (Exception $e) {
            $conn->rollback();
            $error = $e->getMessage();
            $conn->close();
        }
    }
}

if (empty($provinces)) {
    $error = "Aucun dépôt n'est encore enregistré dans votre région. Contactez l'administration pour créer les dépôts.";
}
include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/header.php';
$successMessage = '';
if (!empty($_SESSION['send_stock_success'])) {
    $successMessage = $_SESSION['send_stock_success'];
    unset($_SESSION['send_stock_success']);
}
?>
<style>
    .send-layout input[type="number"]::-webkit-outer-spin-button,
    .send-layout input[type="number"]::-webkit-inner-spin-button {
        -webkit-appearance: none;
        margin: 0;
    }
    .send-layout input[type="number"] {
        -moz-appearance: textfield;
    }
</style>
<div class="dashboard-container" style="padding: 20px; background-color: #F4F0E4; min-height: 70vh;">
    <div class="dashboard-content send-layout" style="background-color: #FFFFFF; padding: 20px; border-radius: 10px; box-shadow: 0 4px 8px rgba(0,0,0,0.1); display:flex; gap:20px; flex-wrap:wrap;">
        <div class="region-stock-panel" style="flex:1 1 320px; background:#FDF8F7; border:1px solid #E0DED8; border-radius:12px; padding:16px; max-height:600px; overflow:auto;">
            <h3 style="color:#4B2F1F; margin-top:0;">Stock régional non réparti</h3>
            <?php if (empty($region_stock_products)): ?>
                <p style="color:#6B5B52;">Aucun stock régional en attente.</p>
            <?php else: ?>
                <ul style="list-style:none; padding:0; margin:0;">
                    <?php foreach ($region_stock_products as $product): ?>
                        <li style="border-bottom:1px dashed rgba(75,47,31,0.25); padding:8px 0;">
                            <div style="font-weight:600;"><?php echo htmlspecialchars($product['product_name'] ?? 'Produit', ENT_QUOTES, 'UTF-8'); ?></div>
                            <div style="font-size:13px; color:#4B2F1F;"><?php echo number_format($product['quantity'], 0, ',', '.'); ?> caisses — <?php echo number_format($product['value'], 0, ',', '.'); ?> BIF</div>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>
        </div>
        <div class="form-panel" style="flex:2 1 600px;">
            <h2 style="color: #4B2F1F; margin-bottom: 10px;">Envoyer du Stock aux Dépôts de la Région</h2>
            <?php if (!empty($successMessage)): ?>
                <div style="background:#E6FFE6;color:#256029;border:1px solid #A7E0A7;padding:12px;border-radius:8px;margin-bottom:15px;">
                    <?php echo htmlspecialchars($successMessage, ENT_QUOTES, 'UTF-8'); ?>
                </div>
            <?php endif; ?>
            <?php if (!empty($error)): ?>
                <div style="background:#FFECEC;color:#B00020;border:1px solid #FFB4B4;padding:12px;border-radius:8px;margin-bottom:15px;">
                    <?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?>
                </div>
            <?php endif; ?>
            <form method="POST" id="stockForm" style="display: flex; flex-direction: column; gap: 15px;">
            <div>
                <label style="color: #000000; font-weight: bold;">Dépôt:</label>
                <select name="province_id" id="province_id" required style="width: 100%; padding: 10px; border-radius: 10px; border: 1px solid #4B2F1F;" <?php echo empty($provinces) ? 'disabled' : ''; ?>>
                    <option value=""><?php echo empty($provinces) ? 'Aucun dépôt disponible' : 'Sélectionnez un dépôt'; ?></option>
                    <?php foreach ($provinces as $province): ?>
                        <option value="<?php echo $province['id']; ?>"><?php echo htmlspecialchars($province['name']); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div>
                <label style="color: #000000; font-weight: bold;">Gestionnaire de dépôt:</label>
                <select name="stock_manager_id" id="stock_manager_id" required style="width: 100%; padding: 10px; border-radius: 10px; border: 1px solid #4B2F1F;" <?php echo empty($stock_managers) ? 'disabled' : ''; ?>>
                    <option value="">Sélectionnez un gestionnaire</option>
                </select>
                <?php if (empty($stock_managers)): ?>
                    <small style="color:#B00020;">Aucun gestionnaire actif n'est rattaché à votre région.</small>
                <?php endif; ?>
            </div>
            <div>
                <label style="color: #000000; font-weight: bold;">Date & heure d'arrivée estimée (optionnel) :</label>
                <input type="datetime-local" name="estimated_arrival" style="width: 100%; padding: 10px; border-radius: 10px; border: 1px solid #4B2F1F;">
            </div>
            <div>
                <label style="color: #000000; font-weight: bold;">Nom du conducteur (optionnel) :</label>
                <input type="text" name="driver_name" style="width: 100%; padding: 10px; border-radius: 10px; border: 1px solid #4B2F1F;">
            </div>
            <div>
                <label style="color: #000000; font-weight: bold;">Téléphone du conducteur :</label>
                <input type="text" name="driver_mobile" pattern="\d{8}" style="width: 100%; padding: 10px; border-radius: 10px; border: 1px solid #4B2F1F;">
            </div>
            <div>
                <label style="color: #000000; font-weight: bold;">Plaque du véhicule (optionnel) :</label>
                <input type="text" name="plate_number" style="width: 100%; padding: 10px; border-radius: 10px; border: 1px solid #4B2F1F;">
            </div>
            <div id="productList" style="display: flex; flex-direction: column; gap: 10px;">
                <div class="product-entry">
                    <label style="color: #000000; font-weight: bold;">Produit :</label>
                    <select name="products[0][product_id]" required style="width: 100%; padding: 10px; border-radius: 10px; border: 1px solid #4B2F1F;">
                        <?php foreach ($products as $product): ?>
                            <option value="<?php echo $product['id']; ?>" data-price="<?php echo $product['price_per_crate']; ?>">
                                <?php echo htmlspecialchars($product['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <label style="color: #000000; font-weight: bold;">Quantité (caisses) :</label>
                    <input type="number" name="products[0][quantity]" min="1" step="1" required style="width: 100%; padding: 10px; border-radius: 10px; border: 1px solid #4B2F1F;" placeholder="Caisses">
                </div>
            </div>
            <button type="button" id="addProduct" style="background-color: #4B2F1F; color: #F4F0E4; padding: 12px 20px; border-radius: 10px; border: none; font-weight: bold; cursor: pointer; margin-top: 10px;">Ajouter un produit</button>
            <div style="margin-top: 10px; display:flex; gap:20px; flex-wrap:wrap;">
                <p style="color: #000000;">Total caisses : <strong><span id="totalProducts">0</span></strong></p>
                <p style="color: #000000;">Valeur totale : <strong><span id="totalValue">0,00</span> BIF</strong></p>
            </div>
            <button type="submit" style="background-color: #F4A261; color: #000000; padding: 12px 20px; border-radius: 10px; border: none; font-weight: bold; cursor: pointer;" <?php echo (empty($provinces) || empty($stock_managers)) ? 'disabled' : ''; ?>>Initier le transfert</button>
        </form>
    </div>
</div>
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const provinceSelect = document.getElementById('province_id');
        const stockManagerSelect = document.getElementById('stock_manager_id');
        const stockManagers = <?php echo json_encode($stock_managers); ?>;

    provinceSelect.addEventListener('change', function() {
        const selectedProvince = this.value;
        stockManagerSelect.innerHTML = '<option value="">Sélectionnez un gestionnaire</option>';
        stockManagers.forEach(manager => {
            if (manager.province_id == selectedProvince) {
                stockManagerSelect.innerHTML += `<option value="${manager.id}">${manager.full_name} (Province ID: ${manager.province_id})</option>`;
            }
        });
    });

    let productIndex = 1;
        const productList = document.getElementById('productList');
        const addProductBtn = document.getElementById('addProduct');
        const totalProducts = document.getElementById('totalProducts');
        const totalValue = document.getElementById('totalValue');

        addProductBtn.addEventListener('click', function() {
            const entry = document.createElement('div');
            entry.classList.add('product-entry');
            entry.innerHTML = `
                <label style="color: #000000; font-weight: bold;">Product:</label>
                <select name="products[${productIndex}][product_id]" required style="width: 100%; padding: 8px; border-radius: 4px; border: 1px solid #4B2F1F;">
                    <?php foreach ($products as $product): ?>
                        <option value="<?php echo $product['id']; ?>" data-price="<?php echo $product['price_per_crate']; ?>">
                            <?php echo htmlspecialchars($product['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <label style="color: #000000; font-weight: bold;">Quantité (caisses) :</label>
                <input type="number" name="products[${productIndex}][quantity]" min="1" step="1" required style="width: 100%; padding: 8px; border-radius: 4px; border: 1px solid #4B2F1F;" placeholder="Caisses">
                <button type="button" class="remove-product" style="background-color: #FF0000; color: #F4F0E4; padding: 5px 10px; border: none; border-radius: 4px; cursor: pointer; margin-top: 10px;">Remove</button>
            `;
            productList.appendChild(entry);
            productIndex++;
            attachRemoveListeners();
            refreshProductOptions();
            updateTotals();
        });

        function refreshProductOptions() {
            const selects = document.querySelectorAll('.product-entry select');
            const selectedValues = Array.from(selects).map(select => select.value).filter(Boolean);
            selects.forEach(select => {
                Array.from(select.options).forEach(option => {
                    if (!option.value) {
                        option.disabled = false;
                        return;
                    }
                    option.disabled = selectedValues.includes(option.value) && select.value !== option.value;
                });
            });
        }

        function attachRemoveListeners() {
            const removeButtons = document.querySelectorAll('.remove-product');
            removeButtons.forEach(button => {
                button.addEventListener('click', function() {
                    button.parentElement.remove();
                    refreshProductOptions();
                    updateTotals();
                });
            });
        }

    function updateTotals() {
        let totalProds = 0;
        let totalVal = 0.00;
        const entries = document.querySelectorAll('.product-entry');
        entries.forEach(entry => {
            const select = entry.querySelector('select[name^="products"][name$="[product_id]"]');
            const quantity = parseInt(entry.querySelector('input[name^="products"][name$="[quantity]"]').value) || 0;
            const price = parseFloat(select.options[select.selectedIndex]?.getAttribute('data-price') || 0);
            totalProds += quantity;
            totalVal += quantity * price;
        });
        totalProducts.textContent = totalProds.toString().replace(/\B(?=(\d{3})+(?!\d))/g, '.');
        const formattedTotal = Math.round(totalVal).toLocaleString('fr-FR');
        totalValue.textContent = formattedTotal;
    }

        productList.addEventListener('change', function(event) {
            if (event.target.name && event.target.name.endsWith('[product_id]')) {
                refreshProductOptions();
                updateTotals();
            }
        });

        productList.addEventListener('input', function(event) {
            if (event.target.name && event.target.name.endsWith('[quantity]')) {
                updateTotals();
            }
        });

    const quantityInputs = productList.querySelectorAll('input[name$="[quantity]"]');
    const disableSpinner = (input) => {
        if (input.dataset.preventSpinnerHandled) {
            return;
        }
        input.dataset.preventSpinnerHandled = '1';
        input.addEventListener('wheel', function(e) {
            e.preventDefault();
        });
        input.addEventListener('keydown', function(e) {
            if (e.key === 'ArrowUp' || e.key === 'ArrowDown') {
                e.preventDefault();
            }
        });
    };

    quantityInputs.forEach(disableSpinner);
    productList.addEventListener('focusin', function(event) {
        if (event.target.matches('input[name$="[quantity]"]')) {
            disableSpinner(event.target);
        }
    });

    refreshProductOptions();
    updateTotals();
});
</script>
<?php include '/opt/lampp/htdocs/masunzu_bar_hotel/includes/footer.php'; ?>
