<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();
$allowed_roles = ['DG', 'DGA', 'Board', 'Admin'];
if (!in_array($_SESSION['role_name'] ?? '', $allowed_roles, true)) {
    error_log("Unauthorized access to stock_transfer.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
    redirect("/masunzu_bar_hotel/dashboards/" . strtolower(str_replace(' ', '_', $_SESSION['role_name'] ?? '')) . "_dashboard.php");
}

$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in stock_transfer.php");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Database connection failed. Contact administrator.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Ensure main_stock has reserved_quantity column for crate-based reservation
try {
    $checkReserved = $conn->query("SHOW COLUMNS FROM main_stock LIKE 'reserved_quantity'");
    if ($checkReserved && $checkReserved->num_rows === 0) {
        $conn->query("ALTER TABLE main_stock ADD COLUMN reserved_quantity INT(11) NOT NULL DEFAULT 0 AFTER quantity");
        error_log("Added reserved_quantity column to main_stock table");
    }
    if ($checkReserved) {
        $checkReserved->free();
    }
} catch (Exception $e) {
    error_log("Error ensuring reserved_quantity on main_stock: " . $e->getMessage());
}

$error = '';
$success = '';
$products = [];
$provinces = [];
$stock_managers = [];

// Fetch products
$result = $conn->query("SELECT id, name, crate_quantity FROM product");
if ($result) {
    $products = $result->fetch_all(MYSQLI_ASSOC);
}

// Fetch provinces
$result = $conn->query("SELECT id, name FROM province");
if ($result) {
    $provinces = $result->fetch_all(MYSQLI_ASSOC);
}

// Fetch stock managers
$result = $conn->query("SELECT id, full_name FROM user WHERE role_id = 5 AND is_active = 1");
if ($result) {
    $stock_managers = $result->fetch_all(MYSQLI_ASSOC);
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_id = (int)($_POST['product_id'] ?? 0);
    $quantity = (int)($_POST['quantity'] ?? 0);
    $province_id = (int)($_POST['province_id'] ?? 0);
    $stock_manager_id = (int)($_POST['stock_manager_id'] ?? 0);
    $driver_name = trim($_POST['driver_name'] ?? '');
    $driver_mobile = trim($_POST['driver_mobile'] ?? '');
    $plate_number = trim($_POST['plate_number'] ?? '');
    $estimated_arrival = $_POST['estimated_arrival'] ?? '';

    // Validate inputs
    if ($product_id <= 0 || $quantity <= 0 || $province_id <= 0 || $stock_manager_id <= 0 || empty($driver_name) || empty($driver_mobile) || empty($plate_number) || empty($estimated_arrival)) {
        $error = 'All fields are required.';
    } else {
        $provinceRegionId = null;
        $stmt = $conn->prepare("SELECT region_id FROM province WHERE id = ?");
        $stmt->bind_param("i", $province_id);
        $stmt->execute();
        $provinceRow = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        if (!$provinceRow || empty($provinceRow['region_id'])) {
            $error = 'La province sélectionnée doit appartenir à une région.';
        } else {
            $provinceRegionId = (int)$provinceRow['region_id'];
        }

        if (!$error) {
            // Get crate_quantity and price
            $stmt = $conn->prepare("SELECT crate_quantity, price_per_crate FROM product WHERE id = ?");
            $stmt->bind_param("i", $product_id);
            $stmt->execute();
            $result = $stmt->get_result();
        $product = $result->fetch_assoc();
        $stmt->close();

        if (!$product) {
            $error = 'Invalid product selected.';
        } else {
            // Read main_stock in crates and compute available as quantity - reserved_quantity
            $stmt = $conn->prepare("SELECT quantity, COALESCE(reserved_quantity, 0) AS reserved_quantity FROM main_stock WHERE product_id = ? AND (unit_type = 'crates' OR unit_type IS NULL OR unit_type = '')");
            $stmt->bind_param("i", $product_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $stock = $result->fetch_assoc();
            $stmt->close();

            $total_crates = (int)($stock['quantity'] ?? 0);
            $reserved_crates = (int)($stock['reserved_quantity'] ?? 0);
            $available_crates = $total_crates - $reserved_crates;

            if ($available_crates < $quantity) {
                $error = 'Insufficient stock in main stock: ' . format_quantity(max($available_crates, 0)) . ' crates available.';
            } else {
                $shipment_id = rand(100000, 999999);
                $total_amount = $quantity * (int)$product['price_per_crate'];

                $stmt = $conn->prepare("INSERT INTO stock_transfer (operation_manager_id, operation_receiver_id, stock_manager_id, initiator_id, product_id, quantity, unit_type, province_id, region_id, status, estimated_arrival, driver_name, driver_mobile, plate_number, shipment_id, total_amount, total_products) VALUES (?, ?, ?, ?, ?, ?, 'crates', ?, ?, 'pending', ?, ?, ?, ?, ?, ?, ?)");
                $paramTypes = str_repeat('i', 8) . 'ssssiii';
                $stmt->bind_param($paramTypes, $_SESSION['user_id'], null, $stock_manager_id, $_SESSION['user_id'], $product_id, $quantity, $province_id, $provinceRegionId, $estimated_arrival, $driver_name, $driver_mobile, $plate_number, $shipment_id, $total_amount, $quantity);
                if ($stmt->execute()) {
                    $stmt->close();

                    // Reserve crates in main_stock: increase reserved_quantity only
                    $stmt = $conn->prepare("
                        UPDATE main_stock 
                        SET reserved_quantity = COALESCE(reserved_quantity, 0) + ?, 
                            unit_type = 'crates',
                            last_updated = NOW()
                        WHERE product_id = ? AND (unit_type = 'crates' OR unit_type IS NULL OR unit_type = '')
                    ");
                    $stmt->bind_param("ii", $quantity, $product_id);
                    $stmt->execute();
                    $stmt->close();

                    log_action($_SESSION['user_id'], 'initiate_transfer', "Initiated stock transfer with shipment ID $shipment_id for product ID $product_id", $product_id, $quantity);

                    $provinceName = '';
                    if ($provinceStmt = $conn->prepare("SELECT name FROM province WHERE id = ?")) {
                        $provinceStmt->bind_param('i', $province_id);
                        $provinceStmt->execute();
                        $provinceName = $provinceStmt->get_result()->fetch_assoc()['name'] ?? '';
                        $provinceStmt->close();
                    }
                    $provinceName = $provinceName ?: ('Province #' . $province_id);
                    $managers = mbh_get_stock_manager_ids_by_province($conn, $province_id);
                    if (!empty($managers)) {
                        $message = "Nouveau transfert #$shipment_id pour $provinceName. Merci de le réceptionner.";
                        $link = "/masunzu_bar_hotel/modules/operation/receive_stock.php";
                        foreach ($managers as $managerId) {
                            mbh_notify_user($conn, $managerId, $message, $link);
                        }
                    }

                    $success = 'Stock transfer initiated successfully! Shipment ID: ' . $shipment_id;
                } else {
                    $error = 'Failed to initiate stock transfer: ' . $stmt->error;
                    error_log("Stock transfer failed: " . $stmt->error);
                    $stmt->close();
                }
            }
        }
    }
    }
}

// Fetch transfer history
$transfers = [];
$stmt = $conn->prepare("
    SELECT st.id, st.shipment_id, st.product_id, p.name as product_name, st.quantity, st.unit_type, st.province_id, pr.name as province_name, st.status, st.sent_at, st.estimated_arrival, st.driver_name, st.driver_mobile, st.plate_number, st.total_amount, p.crate_quantity 
    FROM stock_transfer st 
    JOIN product p ON st.product_id = p.id 
    JOIN province pr ON st.province_id = pr.id 
    WHERE st.operation_manager_id = ? 
    ORDER BY st.sent_at DESC
");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$transfers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

$status_labels = [
    'pending' => 'En attente',
    'confirmed' => 'Reçu',
    'rejected' => 'Rejeté',
];
$transfer_status_summary = [];
foreach ($status_labels as $code => $label) {
    $transfer_status_summary[$code] = [
        'label' => $label,
        'count' => 0,
        'quantity' => 0,
    ];
}
foreach ($transfers as $transfer) {
    $statusKey = strtolower($transfer['status'] ?? 'inconnu');
    if (!isset($transfer_status_summary[$statusKey])) {
        $transfer_status_summary[$statusKey] = [
            'label' => ucfirst(str_replace('_', ' ', $statusKey)),
            'count' => 0,
            'quantity' => 0,
        ];
    }
    $transfer_status_summary[$statusKey]['count'] += 1;
    $transfer_status_summary[$statusKey]['quantity'] += (int)$transfer['quantity'];
}
$conn->close();

require_once __DIR__ . '/../../includes/header.php';
?>

<style>
    .transfer-container {
        padding: 30px;
        background: #F4F0E4;
        min-height: 100vh;
    }
    .transfer-header {
        margin-bottom: 30px;
    }
    .transfer-header h1 {
        color: #4B2F1F;
        font-size: 28px;
        margin: 0 0 10px 0;
    }
    .transfer-header p {
        color: #6B5B52;
        margin: 0;
        font-size: 14px;
    }
    .transfer-metric-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(180px, 1fr));
        gap: 14px;
        margin-bottom: 20px;
    }
    .metric-card {
        border-radius: 12px;
        padding: 14px 16px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
        color: #fff;
    }
    .metric-card.pending {
        background: linear-gradient(135deg, #FFB74D, #F57C00);
    }
    .metric-card.confirmed {
        background: linear-gradient(135deg, #4CAF50, #2E7D32);
    }
    .metric-card.rejected {
        background: linear-gradient(135deg, #E57373, #C62828);
    }
    .metric-card.unknown {
        background: linear-gradient(135deg, #90A4AE, #607D8B);
    }
    .metric-label {
        font-size: 12px;
        letter-spacing: 0.08em;
        text-transform: uppercase;
        opacity: 0.9;
    }
    .metric-value {
        font-size: 28px;
        font-weight: 700;
        margin-top: 6px;
    }
    .transfer-table-wrapper {
        background-color: #FFFFFF;
        border-radius: 8px;
        overflow: hidden;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    }
    .transfer-table {
        width: 100%;
        border-collapse: collapse;
    }
    .transfer-table thead {
        background-color: #4B2F1F;
        color: #F4F0E4;
    }
    .transfer-table th {
        padding: 16px;
        text-align: left;
        font-weight: 600;
        font-size: 13px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    .transfer-table tbody tr {
        border-bottom: 1px solid #E8E4D8;
        transition: background-color 0.2s;
    }
    .transfer-table tbody tr:hover {
        background-color: #FDFBF7;
    }
    .transfer-table td {
        padding: 14px 16px;
        color: #4B2F1F;
        font-size: 13px;
    }
    .status-badge {
        display: inline-block;
        padding: 6px 12px;
        border-radius: 4px;
        font-size: 11px;
        font-weight: 600;
        text-transform: uppercase;
    }
    .status-pending {
        background-color: #FFF3CD;
        color: #856404;
    }
    .status-confirmed {
        background-color: #D4EDDA;
        color: #155724;
    }
    .status-rejected {
        background-color: #F8D7DA;
        color: #721C24;
    }
    .status-inconnu {
        background-color: #E2E3E5;
        color: #41464B;
    }
    .empty-state {
        padding: 60px 20px;
        text-align: center;
        color: #6B5B52;
    }
    .footer-actions {
        margin-top: 30px;
        text-align: center;
    }
    .btn-back {
        display: inline-block;
        background-color: #4B2F1F;
        color: #F4F0E4;
        padding: 12px 30px;
        border-radius: 8px;
        text-decoration: none;
        font-weight: 600;
        transition: background-color 0.2s;
    }
    .btn-back:hover {
        background-color: #3A2418;
    }
    @media (max-width: 768px) {
        .transfer-container {
            padding: 15px;
        }
        .transfer-table {
            font-size: 12px;
        }
        .transfer-table th,
        .transfer-table td {
            padding: 10px 8px;
        }
    }
</style>

<div class="transfer-container">
    <div class="transfer-header">
        <h1>Historique des Transferts de Stock</h1>
        <p>Suivi complet de tous les transferts de stock initiés et supervisés</p>
    </div>

    <div class="transfer-metric-grid">
        <?php foreach ($status_labels as $code => $label): ?>
            <?php $statusInfo = $transfer_status_summary[$code] ?? ['count' => 0, 'quantity' => 0]; ?>
            <?php $cardClass = in_array($code, ['pending', 'confirmed', 'rejected'], true) ? $code : 'unknown'; ?>
            <div class="metric-card <?php echo $cardClass; ?>">
                <div class="metric-label"><?php echo htmlspecialchars($label, ENT_QUOTES, 'UTF-8'); ?></div>
                <div class="metric-value"><?php echo number_format($statusInfo['count'], 0, ',', '.'); ?></div>
                <div style="font-size:12px; opacity:0.85;"><?php echo number_format($statusInfo['quantity'], 0, ',', '.'); ?> caisses</div>
            </div>
        <?php endforeach; ?>
        <?php foreach ($transfer_status_summary as $code => $statusInfo): ?>
            <?php if (isset($status_labels[$code])) {
                continue;
            }
            $cardClass = 'unknown';
            $label = ucfirst(str_replace('_', ' ', $code));
            ?>
            <div class="metric-card <?php echo $cardClass; ?>">
                <div class="metric-label"><?php echo htmlspecialchars($label, ENT_QUOTES, 'UTF-8'); ?></div>
                <div class="metric-value"><?php echo number_format($statusInfo['count'], 0, ',', '.'); ?></div>
                <div style="font-size:12px; opacity:0.85;"><?php echo number_format($statusInfo['quantity'], 0, ',', '.'); ?> caisses</div>
            </div>
        <?php endforeach; ?>
    </div>

    <div style="margin-bottom: 20px; display: flex; gap: 10px; flex-wrap: wrap; align-items: flex-end;">
        <div style="flex: 1; min-width: 200px;">
            <label for="transfer-search" style="display: block; color: #4B2F1F; font-size: 14px; font-weight: 500; margin-bottom: 5px;">Rechercher</label>
            <input type="text" id="transfer-search" placeholder="ID d'expédition, produit, province..." style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
        </div>
        <div style="flex: 1; min-width: 150px;">
            <label for="transfer-date-from" style="display: block; color: #4B2F1F; font-size: 14px; font-weight: 500; margin-bottom: 5px;">Date de début</label>
            <input type="date" id="transfer-date-from" style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
        </div>
        <div style="flex: 1; min-width: 150px;">
            <label for="transfer-date-to" style="display: block; color: #4B2F1F; font-size: 14px; font-weight: 500; margin-bottom: 5px;">Date de fin</label>
            <input type="date" id="transfer-date-to" style="width: 100%; padding: 10px; border: 1px solid #4B2F1F; border-radius: 8px; font-size: 14px;">
        </div>
        <button id="export-csv-btn" style="background-color: #4B2F1F; color: #F4F0E4; padding: 10px 16px; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.3s ease; box-shadow: 0 2px 4px rgba(0,0,0,0.2); min-width: 120px;">📥 Exporter CSV</button>
    </div>

    <div class="transfer-table-wrapper">
        <?php if (empty($transfers)): ?>
            <div class="empty-state">
                <p>Aucun transfert de stock trouvé.</p>
            </div>
        <?php else: ?>
            <table class="transfer-table">
                <thead>
                    <tr>
                        <th style="width: 12%;">ID d'expédition</th>
                        <th style="width: 14%;">Produit</th>
                        <th style="width: 10%;">Qté (caisses)</th>
                        <th style="width: 12%;">Province</th>
                        <th style="width: 10%;">Statut</th>
                        <th style="width: 13%;">Envoyé le</th>
                        <th style="width: 13%;">Arrivée prévue</th>
                        <th style="width: 10%;">Montant</th>
                        <th style="width: 16%;">Informations conducteur</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($transfers as $transfer):
                        $statusKey = strtolower($transfer['status'] ?? 'inconnu');
                        $status_class = in_array($statusKey, ['pending', 'confirmed', 'rejected'], true) ? $statusKey : 'inconnu';
                        $status_label = $status_labels[$statusKey] ?? ucfirst(str_replace('_', ' ', $statusKey));
                    ?>
                        <tr class="transfer-row" data-sent-date="<?php echo $transfer['sent_at'] ? date('Y-m-d', strtotime($transfer['sent_at'])) : ''; ?>">
                            <td><strong><?php echo htmlspecialchars($transfer['shipment_id']); ?></strong></td>
                            <td><?php echo htmlspecialchars($transfer['product_name']); ?></td>
                            <td><?php echo number_format((int)$transfer['quantity'], 0, ',', '.'); ?></td>
                            <td><?php echo htmlspecialchars($transfer['province_name']); ?></td>
                            <td><span class="status-badge <?php echo $status_class; ?>"><?php echo htmlspecialchars($status_label); ?></span></td>
                            <td><?php echo $transfer['sent_at'] ? date('Y-m-d H:i', strtotime($transfer['sent_at'])) : '—'; ?></td>
                            <td><?php echo $transfer['estimated_arrival'] ? date('Y-m-d H:i', strtotime($transfer['estimated_arrival'])) : '—'; ?></td>
                            <td><strong><?php echo number_format($transfer['total_amount'], 0, ',', '.'); ?> BIF</strong></td>
                            <td>
                                <div style="font-size: 12px; line-height: 1.4;">
                                    <div><strong><?php echo htmlspecialchars($transfer['driver_name']); ?></strong></div>
                                    <div style="color: #6B5B52;">☎ <?php echo htmlspecialchars($transfer['driver_mobile']); ?></div>
                                    <div style="color: #6B5B52;">🚗 <?php echo htmlspecialchars($transfer['plate_number']); ?></div>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>

    <div class="footer-actions">
        <a href="/masunzu_bar_hotel/dashboards/operation_dashboard.php" class="btn-back">← Retour au tableau de bord</a>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        const searchInput = document.getElementById('transfer-search');
        const dateFromInput = document.getElementById('transfer-date-from');
        const dateToInput = document.getElementById('transfer-date-to');
        const exportBtn = document.getElementById('export-csv-btn');
        const transferRows = document.querySelectorAll('.transfer-row');
        
        const filterTransfers = () => {
            const searchTerm = (searchInput ? searchInput.value.toLowerCase().trim() : '');
            const dateFrom = dateFromInput ? dateFromInput.value : '';
            const dateTo = dateToInput ? dateToInput.value : '';
            
            const dateFromObj = dateFrom ? new Date(dateFrom + 'T00:00:00') : null;
            const dateToObj = dateTo ? new Date(dateTo + 'T23:59:59') : null;
            
            let visibleCount = 0;
            transferRows.forEach(function(row) {
                const text = row.textContent.toLowerCase();
                const sentDate = row.getAttribute('data-sent-date');
                
                const matchesSearch = !searchTerm || text.includes(searchTerm);
                
                let matchesDateFrom = true;
                let matchesDateTo = true;
                if (sentDate) {
                    const rowDate = new Date(sentDate + 'T00:00:00');
                    if (dateFromObj) matchesDateFrom = rowDate >= dateFromObj;
                    if (dateToObj) matchesDateTo = rowDate <= dateToObj;
                }
                
                const shouldShow = matchesSearch && matchesDateFrom && matchesDateTo;
                row.style.display = shouldShow ? '' : 'none';
                if (shouldShow) visibleCount++;
            });
            
            const tbody = document.querySelector('.transfer-table tbody');
            let emptyMsg = document.getElementById('transfers-empty-msg');
            if (visibleCount === 0 && tbody) {
                if (!emptyMsg) {
                emptyMsg = document.createElement('tr');
                emptyMsg.id = 'transfers-empty-msg';
                emptyMsg.innerHTML = '<td colspan="9" style="padding: 20px; text-align: center; color: #6B5B52;">Aucun transfert ne correspond aux filtres.</td>';
                    tbody.appendChild(emptyMsg);
                }
                emptyMsg.style.display = '';
            } else if (emptyMsg) {
                emptyMsg.style.display = 'none';
            }
        };
        
        const exportToCSV = () => {
            const rows = Array.from(transferRows).filter(row => row.style.display !== 'none');
            
            if (rows.length === 0) {
                alert('Aucun transfert à exporter. Ajustez vos filtres.');
                return;
            }
            
            const headers = ['Shipment ID', 'Product', 'Qty (Crates)', 'Province', 'Status', 'Sent At', 'Est. Arrival', 'Amount', 'Driver Name', 'Driver Phone', 'Driver Plate'];
            const csvContent = [];
            csvContent.push(headers.join(','));
            
            rows.forEach(row => {
                const cells = row.querySelectorAll('td');
                if (cells.length >= 9) {
                    const rowData = [
                        `"${cells[0].textContent.trim().replace(/"/g, '""')}"`,
                        `"${cells[1].textContent.trim().replace(/"/g, '""')}"`,
                        cells[2].textContent.trim().replace(/"/g, '""'),
                        `"${cells[3].textContent.trim().replace(/"/g, '""')}"`,
                        `"${cells[4].textContent.trim().replace(/"/g, '""')}"`,
                        cells[5].textContent.trim(),
                        cells[6].textContent.trim(),
                        cells[7].textContent.trim().replace(/"/g, '""'),
                        `"${cells[8].textContent.split('☎')[0].trim().replace(/"/g, '""')}"`,
                        `"${(cells[8].textContent.match(/☎\s*([^\n🚗]*)/)?.[1] || '').trim().replace(/"/g, '""')}"`,
                        `"${(cells[8].textContent.match(/🚗\s*(.*)/)?.[1] || '').trim().replace(/"/g, '""')}"`
                    ];
                    csvContent.push(rowData.join(','));
                }
            });
            
            const csv = csvContent.join('\n');
            const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
            const link = document.createElement('a');
            const url = URL.createObjectURL(blob);
            
            const dateStr = new Date().toISOString().split('T')[0];
            link.setAttribute('href', url);
            link.setAttribute('download', `stock_transfers_${dateStr}.csv`);
            link.style.visibility = 'hidden';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        };
        
        if (searchInput) {
            searchInput.addEventListener('input', filterTransfers);
        }
        if (dateFromInput) {
            dateFromInput.addEventListener('change', filterTransfers);
        }
        if (dateToInput) {
            dateToInput.addEventListener('change', filterTransfers);
        }
        if (exportBtn) {
            exportBtn.addEventListener('click', exportToCSV);
        }
    });
</script>

<?php require_once __DIR__ . '/../../includes/footer.php'; ?>
